<?php
/**
 * Plugin Name: RCP API WP Integration
 * Plugin URI: https://radiocontentpro.com/wordpress-plugin/
 * Description: Pull posts from the RCP API and import them into WordPress.
 * Version: 0.9.0
 * Author: Radio Content Pro
 * Author URI: http://radiocontentpro.com/
 * Text Domain: rcp-api-wp-integration
 * Domain Path: /languages
 */

if ( ! defined( "ABSPATH" ) ) {
    exit;
}

// Define plugin constants
define( 'RCP_API_PLUGIN_FILE', __FILE__ );
define( 'RCP_API_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'RCP_API_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Configuration constants (can be overridden in wp-config.php)
if ( ! defined( 'RCP_MAX_IMPORT_BATCH' ) ) {
    define( 'RCP_MAX_IMPORT_BATCH', 50 );
}
if ( ! defined( 'RCP_API_TIMEOUT' ) ) {
    define( 'RCP_API_TIMEOUT', 30 );
}
if ( ! defined( 'RCP_CACHE_TTL' ) ) {
    define( 'RCP_CACHE_TTL', 300 );
}
if ( ! defined( 'RCP_MAX_RETRIES' ) ) {
    define( 'RCP_MAX_RETRIES', 3 );
}
if ( ! defined( 'RCP_DEBUG_PERFORMANCE' ) ) {
    define( 'RCP_DEBUG_PERFORMANCE', false );
}

// Load required classes
require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-validator.php';
require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-rate-limiter.php';
require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-performance-monitor.php';
require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-job-queue.php';
require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-error-handler.php';
require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-database-manager.php';
// WP-CLI commands (if available)
if ( defined( 'WP_CLI' ) && WP_CLI ) {
    require_once RCP_API_PLUGIN_DIR . 'includes/class-rcp-api-cli.php';
}

// Load plugin text domain for translations
add_action( 'plugins_loaded', function() {
    load_plugin_textdomain( 'rcp-api-wp-integration', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
} );

if ( ! class_exists( 'RCP_API_WP_Integration' ) ) {

class RCP_API_WP_Integration {

    /** Number of posts to request from the API per page */
    const POSTS_PER_PAGE = 30;

    const OPTION_USERNAME   = "rcp_feed_username";
    const OPTION_PASSWORD   = "rcp_feed_password";
    const OPTION_CATEGORY_MAPPING = 'rcp_category_mapping';
    const OPTION_IMPORT_FEATURED = 'rcp_import_featured_image';
    const OPTION_ENABLE_AUTO_PUBLISH = 'rcp_enable_auto_publish';
    const OPTION_ENABLE_AUTO_DRAFT   = 'rcp_enable_auto_draft';
    const OPTION_AUTO_PUBLISH_CATS   = 'rcp_auto_publish_cats';
    const OPTION_AUTO_DRAFT_CATS     = 'rcp_auto_draft_cats';
    const OPTION_CATEGORY_AUTOMATION = 'rcp_category_automation';
    const OPTION_CATEGORY_AUTHORS    = 'rcp_category_authors';
    const OPTION_FEATURED_IMAGE_CATS = 'rcp_featured_image_cats'; // legacy checkbox list
    const OPTION_CATEGORY_IMAGE_MODES = 'rcp_category_image_modes';
    const OPTION_CRON_FREQUENCY     = 'rcp_auto_import_frequency';
    const OPTION_LAST_IMPORT        = 'rcp_last_auto_import';
    const OPTION_LAST_SUCCESS       = 'rcp_last_successful_import';
    const OPTION_LAST_HEARTBEAT     = 'rcp_cron_last_heartbeat';
    const OPTION_CATEGORY_CUTOFFS   = 'rcp_category_cutoffs';
    const OPTION_VERSION            = 'rcp_plugin_version';
    const OPTION_IMPORT_LOG         = 'rcp_auto_import_log';
    const OPTION_DELETED_POSTS      = 'rcp_deleted_posts';
    const OPTION_DEBUG_MODE         = 'rcp_debug_mode';
    // AI Images: generate on import only if no featured image (opt-in)
    const OPTION_AI_AUTOGEN         = 'rcp_ai_autogen_on_import';
    const LOG_RETENTION_HOURS       = 48;
    const CRON_HOOK = 'rcp_auto_import_hook';
    const TRANSIENT_FEED    = "rcp_feed_cache";
    const RATE_LIMIT_WINDOW = 60; // 1 minute
    const RATE_LIMIT_REQUESTS = 30; // 30 requests per minute
    const API_TIMEOUT = 30; // 30 seconds timeout for API requests

    /** Maximum posts to import on a new installation */
    const MAX_INITIAL_POSTS = 15;

    /** Current plugin version */
    const VERSION = '0.9.0';

    /** Custom capability for managing the plugin */
    const CAPABILITY = 'manage_rcp_api';

    // Base URL for the Radio Content Pro API.
    private $api_base = "https://app.radiocontentpro.com/wp-json/rcp/v1";

    public function __construct() {
        // Initialize error handler first
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            RCP_API_Error_Handler::init();
        }
        
        // Initialize job queue
        add_action( 'init', [ 'RCP_API_Job_Queue', 'init' ] );
        
        // Initialize database manager
        RCP_API_Database_Manager::init();
        
        // Core hooks
        add_action( "admin_menu", [ $this, "register_menu" ] );
        add_action( "admin_init", [ $this, "register_settings" ] );
        add_action( "admin_enqueue_scripts", [ $this, "enqueue_assets" ] );
        add_action( "wp_ajax_rcp_fetch_feed", [ $this, "ajax_fetch_feed" ] );
        add_action( "wp_ajax_rcp_import_post", [ $this, "ajax_import_post" ] );
        add_action( "wp_ajax_rcp_fetch_categories", [ $this, "ajax_fetch_categories" ] );
        add_action( "wp_ajax_rcp_generate_ai_image", [ $this, "ajax_generate_ai_image" ] );
        add_action( "wp_ajax_rcp_get_health_check", [ $this, "ajax_health_check" ] );
        add_action( 'admin_post_rcp_api_disconnect', [ $this, 'handle_disconnect' ] );
        add_action( 'admin_post_rcp_force_auto_import', [ $this, 'handle_force_auto_import' ] );
        add_action( 'admin_post_rcp_clear_import_lock', [ $this, 'handle_clear_import_lock' ] );
        add_action( 'admin_post_rcp_ai_images_clear_log', [ $this, 'handle_ai_images_clear_log' ] );
        add_action( 'admin_post_rcp_ai_images_retry', [ $this, 'handle_ai_images_retry' ] );
        add_filter( 'cron_schedules', [ $this, 'add_cron_schedules' ] );
        add_action( 'init', [ $this, 'maybe_schedule_auto_import' ] );
        add_action( 'init', [ $this, 'maybe_run_missed_import' ] );
        add_action( 'init', [ $this, 'load_media_functions' ] );
        add_action( 'init', [ $this, 'maybe_upgrade' ] );
        add_action( self::CRON_HOOK, [ $this, 'handle_auto_import' ], 10 );
        add_action( 'updated_option', [ $this, 'maybe_clear_cache' ], 10, 3 );
        add_action( 'added_option', [ $this, 'maybe_clear_cache' ], 10, 2 );
        
        // Track when imported posts are deleted
        add_action( 'before_delete_post', [ $this, 'track_deleted_post' ] );
        add_action( 'wp_trash_post', [ $this, 'track_deleted_post' ] );
        
        // Add cleanup to daily cron
        add_action( 'wp_scheduled_delete', [ $this, 'scheduled_cleanup' ] );
    }

    public function register_menu() {
        add_menu_page(
            __( 'RCP API', 'rcp-api-wp-integration' ),
            __( 'RCP API', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-feed',
            [ $this, 'feed_page' ],
            'dashicons-rss'
        );

        add_submenu_page(
            'rcp-api-feed',
            __( 'Feed', 'rcp-api-wp-integration' ),
            __( 'Feed', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-feed',
            [ $this, 'feed_page' ]
        );

        add_submenu_page(
            'rcp-api-feed',
            __( 'Settings', 'rcp-api-wp-integration' ),
            __( 'Settings', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-settings',
            [ $this, 'settings_page' ]
        );

        add_submenu_page(
            'rcp-api-feed',
            __( 'API Connection', 'rcp-api-wp-integration' ),
            __( 'API Connection', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-info',
            [ $this, 'info_page' ]
        );

        add_submenu_page(
            'rcp-api-feed',
            __( 'Automation Log', 'rcp-api-wp-integration' ),
            __( 'Automation Log', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-log',
            [ $this, 'log_page' ]
        );
        
        add_submenu_page(
            'rcp-api-feed',
            __( 'Diagnostics', 'rcp-api-wp-integration' ),
            __( 'Diagnostics', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-diagnostics',
            [ $this, 'diagnostics_page' ]
        );

        add_submenu_page(
            'rcp-api-feed',
            __( 'AI Images', 'rcp-api-wp-integration' ),
            __( 'AI Images', 'rcp-api-wp-integration' ),
            self::CAPABILITY,
            'rcp-api-ai-images',
            [ $this, 'ai_images_page' ]
        );
    }

    public function register_settings() {
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_USERNAME,
            [
                'sanitize_callback' => [ $this, 'sanitize_username' ],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_PASSWORD,
            [
                'sanitize_callback' => [ $this, 'sanitize_password' ],
            ]
        );
        register_setting(
            "rcp-api-wp-integration",
            self::OPTION_CATEGORY_MAPPING,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_category_mapping' ],
                'default'           => [],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_IMPORT_FEATURED,
            [
                'type'              => 'boolean',
                'sanitize_callback' => [ $this, 'sanitize_featured_image_option' ],
                'default'           => false,
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_CATEGORY_AUTOMATION,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_category_automation' ],
                'default'           => [],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_CATEGORY_AUTHORS,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_category_authors' ],
                'default'           => [],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_ENABLE_AUTO_PUBLISH,
            [
                'type'              => 'boolean',
                'sanitize_callback' => [ $this, 'sanitize_boolean_option' ],
                'default'           => false,
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_ENABLE_AUTO_DRAFT,
            [
                'type'              => 'boolean',
                'sanitize_callback' => [ $this, 'sanitize_boolean_option' ],
                'default'           => false,
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_AUTO_PUBLISH_CATS,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_auto_publish_cats' ],
                'default'           => [],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_AUTO_DRAFT_CATS,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_auto_draft_cats' ],
                'default'           => [],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_DEBUG_MODE,
            [
                'type'              => 'boolean',
                'sanitize_callback' => [ $this, 'sanitize_boolean_option' ],
                'default'           => false,
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_FEATURED_IMAGE_CATS,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_featured_image_cats' ],
                'default'           => [],
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_CATEGORY_IMAGE_MODES,
            [
                'type'              => 'array',
                'sanitize_callback' => [ $this, 'sanitize_category_image_modes' ],
                'default'           => [],
            ]
        );
        // AI Images automation toggle
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_AI_AUTOGEN,
            [
                'type'              => 'boolean',
                'sanitize_callback' => [ $this, 'sanitize_boolean_option' ],
                'default'           => false,
            ]
        );
        register_setting(
            'rcp-api-wp-integration',
            self::OPTION_CRON_FREQUENCY,
            [
                'type'              => 'string',
                'sanitize_callback' => [ $this, 'sanitize_cron_frequency' ],
                'default'           => 'hourly',
            ]
        );
    }

    /**
     * Keep existing option value when nothing is submitted.
     *
     * @param mixed  $value  Submitted value.
     * @param string $option Option name.
     * @return string Sanitized value.
     */
    private function sanitize_persistent_option( $value, $option ) {
        if ( null === $value || '' === $value ) {
            return get_option( $option, '' );
        }

        return sanitize_text_field( $value );
    }

    public function sanitize_username( $value ) {
        return $this->sanitize_persistent_option( $value, self::OPTION_USERNAME );
    }

    public function sanitize_password( $value ) {
        return $this->sanitize_persistent_option( $value, self::OPTION_PASSWORD );
    }

    public function sanitize_featured_image_option( $value ) {
        if ( null === $value ) {
            $value = get_option( self::OPTION_IMPORT_FEATURED, false );
        }
        return $value ? true : false;
    }

    public function sanitize_boolean_option( $value ) {
        return $value ? true : false;
    }

    public function sanitize_auto_publish_cats( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        $sanitized = [];
        foreach ( $value as $cat => $row ) {
            if ( empty( $row['enabled'] ) ) {
                continue;
            }
            $sanitized[ strtolower( sanitize_text_field( $cat ) ) ] = [
                'author' => isset( $row['author'] ) ? (int) $row['author'] : 0,
            ];
        }
        return $sanitized;
    }

    public function sanitize_auto_draft_cats( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        $out = [];
        foreach ( $value as $cat ) {
            $cat = sanitize_text_field( $cat );
            if ( '' !== $cat ) {
                $out[] = strtolower( $cat );
            }
        }
        return $out;
    }

    public function sanitize_featured_image_cats( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        $out = [];
        foreach ( $value as $cat ) {
            $cat = sanitize_text_field( $cat );
            if ( '' !== $cat ) {
                $out[] = strtolower( $cat );
            }
        }
        return $out;
    }

    public function sanitize_category_image_modes( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }

        $allowed = [ 'none', 'import', 'ai', 'fallback' ];
        $out     = [];

        foreach ( $value as $cat => $mode ) {
            $mode = sanitize_text_field( $mode );
            if ( ! in_array( $mode, $allowed, true ) ) {
                continue;
            }

            $normalized = $this->normalize_category_name( $cat );
            if ( '' === $normalized ) {
                continue;
            }

            $out[ $normalized ] = $mode;
        }

        return $out;
    }

    public function sanitize_cron_frequency( $value ) {
        $allowed = [ 'hourly', 'rcp_15min', 'rcp_30min' ];
        if ( ! in_array( $value, $allowed, true ) ) {
            $value = 'hourly';
        }
        return $value;
    }

    public function sanitize_category_automation( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        $out = [];
        foreach ( $value as $cat => $mode ) {
            $mode = sanitize_text_field( $mode );
            if ( ! in_array( $mode, [ 'publish', 'draft', 'none' ], true ) ) {
                $mode = 'none';
            }
            $out[ strtolower( sanitize_text_field( $cat ) ) ] = $mode;
        }
        return $out;
    }

    public function sanitize_category_authors( $value ) {
        if ( ! is_array( $value ) ) {
            return [];
        }
        $out = [];
        foreach ( $value as $cat => $author ) {
            $author = (int) $author;
            if ( $author > 0 ) {
                $out[ strtolower( sanitize_text_field( $cat ) ) ] = $author;
            }
        }
        return $out;
    }

    /**
     * Sanitize the submitted category mapping.
     *
     * @param mixed $value Raw option value.
     * @return array Sanitized mapping rows.
     */
    public function sanitize_category_mapping( $value ) {
        if ( null === $value ) {
            $value = get_option( self::OPTION_CATEGORY_MAPPING, [] );
        }

        if ( ! is_array( $value ) ) {
            return [];
        }

        $sanitized = [];
        foreach ( $value as $key => $row ) {
            if ( is_array( $row ) ) {
                // Backwards compatibility with old format.
                $api = isset( $row['api'] ) ? sanitize_text_field( $row['api'] ) : '';
                $wp  = isset( $row['wp'] ) ? (int) $row['wp'] : 0;
            } else {
                $api = sanitize_text_field( $key );
                $wp  = (int) $row;
            }

            if ( '' === $api || 0 === $wp ) {
                continue;
            }

            $sanitized[] = [
                'api' => $api,
                'wp'  => $wp,
            ];
        }

        return $sanitized;
    }

    public function enqueue_assets( $hook ) {
        if ( false === strpos( $hook, 'rcp-api-feed' ) && false === strpos( $hook, 'rcp-api-settings' ) && false === strpos( $hook, 'rcp-api-info' ) ) {
            return;
        }
        wp_enqueue_script( 'rcp-api-wp-integration', plugin_dir_url( __FILE__ ) . 'script.js', [ 'jquery' ], self::VERSION, true );

        $mapping     = get_option( self::OPTION_CATEGORY_MAPPING, [] );
        if ( ! is_array( $mapping ) ) {
            $mapping = [];
        }
        $mapping_js  = [];
        foreach ( (array) $mapping as $row ) {
            if ( empty( $row['api'] ) || empty( $row['wp'] ) ) {
                continue;
            }
            $key  = strtolower( trim( $row['api'] ) );
            $term = get_term( (int) $row['wp'], 'category' );
            if ( $term && ! is_wp_error( $term ) ) {
                $mapping_js[ $key ] = [
                    'id'   => (int) $row['wp'],
                    'name' => $term->name,
                ];
            }
        }

        $per_page = (int) apply_filters( 'rcp_api_posts_per_page', self::POSTS_PER_PAGE );
        wp_localize_script( 'rcp-api-wp-integration', 'RCPFeed', [
            'ajax_url'  => admin_url( 'admin-ajax.php' ),
            'nonce'     => wp_create_nonce( 'rcp_feed_nonce' ),
            'per_page'  => $per_page,
            'select_all' => __( 'Select All', 'rcp-api-wp-integration' ),
            'deselect_all' => __( 'Deselect All', 'rcp-api-wp-integration' ),
            'mapping'   => $mapping_js,
            'no_posts_selected' => __( 'No posts selected', 'rcp-api-wp-integration' ),
            // i18n additions for script.js
            'loading_categories' => __( 'Loading categories...', 'rcp-api-wp-integration' ),
            'load_categories_failed' => __( 'Failed to load categories:', 'rcp-api-wp-integration' ),
            'network_error_categories' => __( 'Failed to load categories due to network error', 'rcp-api-wp-integration' ),
            'queued_for_import' => __( 'Queued for import', 'rcp-api-wp-integration' ),
            'queued' => __( 'Queued', 'rcp-api-wp-integration' ),
            'testing' => __( 'Testing...', 'rcp-api-wp-integration' ),
            'api_connected' => __( 'Connected', 'rcp-api-wp-integration' ),
            'api_connection_failed' => __( 'Failed:', 'rcp-api-wp-integration' ),
            'api_connection_error' => __( 'Error', 'rcp-api-wp-integration' ),
            'api_network_error' => __( 'Network error', 'rcp-api-wp-integration' ),
            'import_in_progress' => __( 'Import in progress...', 'rcp-api-wp-integration' ),
            'force_import_started' => __( 'Force import started', 'rcp-api-wp-integration' ),
            'post_singular' => __( 'post', 'rcp-api-wp-integration' ),
            'post_plural' => __( 'posts', 'rcp-api-wp-integration' ),
            'selected_label' => __( 'selected', 'rcp-api-wp-integration' ),
            'loading_label' => __( 'Loading...', 'rcp-api-wp-integration' ),
            'refresh_feed_label' => __( 'Refresh Feed', 'rcp-api-wp-integration' ),
            'import_complete' => __( 'Import complete.', 'rcp-api-wp-integration' ),
            'posts_imported_successfully' => __( 'posts imported successfully.', 'rcp-api-wp-integration' ),
            'posts_failed_to_import' => __( 'posts failed to import.', 'rcp-api-wp-integration' ),
            'errors_label' => __( 'Errors:', 'rcp-api-wp-integration' ),
            'and_more_errors' => __( '... and %d more errors.', 'rcp-api-wp-integration' ),
            // AI images UI strings
            'ai_generate' => __( 'Generate AI Image', 'rcp-api-wp-integration' ),
            'ai_queued' => __( 'Queued for image generation', 'rcp-api-wp-integration' ),
            'ai_failed' => __( 'AI image request failed', 'rcp-api-wp-integration' ),
            'ai_attached' => __( 'AI image attached successfully', 'rcp-api-wp-integration' ),
            'ai_skipped' => __( 'Post already has a featured image', 'rcp-api-wp-integration' ),
            'ai_done' => __( 'AI Image Ready', 'rcp-api-wp-integration' ),
            'ai_queued_button' => __( 'Queued', 'rcp-api-wp-integration' ),
            'ai_bulk_start' => __( 'Queuing AI image generation...', 'rcp-api-wp-integration' ),
            'ai_bulk_complete' => __( 'AI image generation requests queued.', 'rcp-api-wp-integration' ),
            'ai_bulk_none' => __( 'No imported posts found on this page.', 'rcp-api-wp-integration' ),
        ] );
        wp_enqueue_style( 'rcp-api-wp-integration', plugin_dir_url( __FILE__ ) . 'style.css', [], self::VERSION );
    }

    public function settings_page() {
        if ( ! current_user_can( self::CAPABILITY ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'RCP API Settings', 'rcp-api-wp-integration' ); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields( "rcp-api-wp-integration" ); ?>

                <?php
                $mapping  = get_option( self::OPTION_CATEGORY_MAPPING, [] );
                if ( ! is_array( $mapping ) ) {
                    $mapping = [];
                }
                $enable_auto_publish = get_option( self::OPTION_ENABLE_AUTO_PUBLISH, false );
                $enable_auto_draft   = get_option( self::OPTION_ENABLE_AUTO_DRAFT, false );
                $auto_publish_cats   = get_option( self::OPTION_AUTO_PUBLISH_CATS, [] );
                if ( ! is_array( $auto_publish_cats ) ) { $auto_publish_cats = []; }
                $auto_draft_cats     = get_option( self::OPTION_AUTO_DRAFT_CATS, [] );
                if ( ! is_array( $auto_draft_cats ) ) { $auto_draft_cats = []; }
                $automation          = get_option( self::OPTION_CATEGORY_AUTOMATION, [] );
                if ( ! is_array( $automation ) ) { $automation = []; }
                $category_authors    = get_option( self::OPTION_CATEGORY_AUTHORS, [] );
                if ( ! is_array( $category_authors ) ) { $category_authors = []; }
                $featured_cats       = get_option( self::OPTION_FEATURED_IMAGE_CATS, [] );
                if ( ! is_array( $featured_cats ) ) { $featured_cats = []; }
                $category_image_modes = get_option( self::OPTION_CATEGORY_IMAGE_MODES, [] );
                if ( ! is_array( $category_image_modes ) ) { $category_image_modes = []; }
                $category_image_modes_lower = [];
                foreach ( $category_image_modes as $stored_key => $stored_mode ) {
                    $category_image_modes_lower[ strtolower( $stored_key ) ] = $stored_mode;
                }
                $frequency          = get_option( self::OPTION_CRON_FREQUENCY, 'hourly' );
                $wp_cats  = get_categories( [ 'hide_empty' => false ] );
                $api_cats = $this->get_api_categories();
                if ( is_wp_error( $api_cats ) ) {
                    $api_cats = [];
                }
                sort( $api_cats, SORT_NATURAL | SORT_FLAG_CASE );
                ?>

                <div id="rcp-settings-connection" style="margin: 10px 0 20px;">
                    <button type="button" class="button" id="rcp-test-connection" aria-controls="rcp-connection-result" aria-expanded="false"><?php esc_html_e( 'Test Connection', 'rcp-api-wp-integration' ); ?></button>
                    <span id="rcp-connection-result" style="margin-left: 10px;"></span>
                    <div id="rcp-status-live" role="status" aria-live="polite" class="screen-reader-text"></div>
                </div>

                <table class="wp-list-table widefat striped" role="presentation" id="rcp-category-settings-table">
                    <thead>
                    <tr>
                        <th><?php esc_html_e( 'Category Name', 'rcp-api-wp-integration' ); ?></th>
                        <th title="<?php esc_attr_e( 'Choose whether posts in this category are auto-published, saved as drafts, or ignored.', 'rcp-api-wp-integration' ); ?>">
                            <?php esc_html_e( 'Automation', 'rcp-api-wp-integration' ); ?>
                            <span class="dashicons dashicons-info rcp-info-icon" aria-hidden="true"></span>
                            <span class="screen-reader-text"><?php esc_html_e( 'Choose whether posts in this category are auto-published, saved as drafts, or ignored.', 'rcp-api-wp-integration' ); ?></span>
                        </th>
                        <th title="<?php esc_attr_e( 'Select the default author for imported posts.', 'rcp-api-wp-integration' ); ?>">
                            <?php esc_html_e( 'Author', 'rcp-api-wp-integration' ); ?>
                            <span class="dashicons dashicons-info rcp-info-icon" aria-hidden="true"></span>
                            <span class="screen-reader-text"><?php esc_html_e( 'Select the default author for imported posts.', 'rcp-api-wp-integration' ); ?></span>
                        </th>
                        <th title="<?php esc_attr_e( 'Pick the WordPress category where the post will be placed.', 'rcp-api-wp-integration' ); ?>">
                            <?php esc_html_e( 'WordPress Category', 'rcp-api-wp-integration' ); ?>
                            <span class="dashicons dashicons-info rcp-info-icon" aria-hidden="true"></span>
                            <span class="screen-reader-text"><?php esc_html_e( 'Pick the WordPress category where the post will be placed.', 'rcp-api-wp-integration' ); ?></span>
                        </th>
                        <th title="<?php esc_attr_e( 'Choose how featured images should be handled for posts in this category.', 'rcp-api-wp-integration' ); ?>">
                            <?php esc_html_e( 'Featured Image Options', 'rcp-api-wp-integration' ); ?>
                            <span
                                id="rcp-image-info"
                                class="dashicons dashicons-info rcp-info-icon"
                                role="button"
                                tabindex="0"
                                aria-haspopup="true"
                                aria-expanded="false"
                                aria-controls="rcp-image-options-help"
                                title="<?php esc_attr_e( 'Click for details about image options.', 'rcp-api-wp-integration' ); ?>"
                            ></span>
                            <span class="screen-reader-text"><?php esc_html_e( 'Choose how featured images should be handled for posts in this category.', 'rcp-api-wp-integration' ); ?></span>
                        </th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ( $api_cats as $cat ) :
                        $key      = strtolower( $cat );
                        $mode     = isset( $automation[ $key ] ) ? $automation[ $key ] : 'none';
                        $author   = isset( $category_authors[ $key ] ) ? (int) $category_authors[ $key ] : 0;
                        $selected = 0;
                        foreach ( $mapping as $row ) {
                            if ( isset( $row['api'], $row['wp'] ) && strtolower( $row['api'] ) === strtolower( $cat ) ) {
                                $selected = (int) $row['wp'];
                                break;
                            }
                        }
                        $normalized_category = $this->normalize_category_name( $cat );
                        $checked = in_array( $key, $featured_cats, true );
                        $image_mode = '';
                        if ( isset( $category_image_modes[ $normalized_category ] ) ) {
                            $image_mode = $category_image_modes[ $normalized_category ];
                        } elseif ( isset( $category_image_modes[ $cat ] ) ) {
                            $image_mode = $category_image_modes[ $cat ];
                        } elseif ( isset( $category_image_modes_lower[ $normalized_category ] ) ) {
                            $image_mode = $category_image_modes_lower[ $normalized_category ];
                        } elseif ( isset( $category_image_modes_lower[ $key ] ) ) {
                            $image_mode = $category_image_modes_lower[ $key ];
                        }
                        if ( '' === $image_mode && $checked ) {
                            $image_mode = 'import';
                        }
                        if ( '' === $image_mode ) {
                            $image_mode = 'none';
                        }
                        ?>
                        <tr>
                            <th scope="row"><?php echo esc_html( $cat ); ?></th>
                            <td>
                                <select name="rcp_category_automation[<?php echo esc_attr( $cat ); ?>]">
                                    <option value="none" <?php selected( $mode, 'none' ); ?>><?php esc_html_e( 'None', 'rcp-api-wp-integration' ); ?></option>
                                    <option value="draft" <?php selected( $mode, 'draft' ); ?>><?php esc_html_e( 'Auto Draft', 'rcp-api-wp-integration' ); ?></option>
                                    <option value="publish" <?php selected( $mode, 'publish' ); ?>><?php esc_html_e( 'Auto Publish', 'rcp-api-wp-integration' ); ?></option>
                                </select>
                            </td>
                            <td>
                                <?php wp_dropdown_users( [ 'name' => 'rcp_category_authors[' . esc_attr( $cat ) . ']', 'selected' => $author, 'show_option_none' => __( '-- None --', 'rcp-api-wp-integration' ), 'include_selected' => true ] ); ?>
                            </td>
                            <td>
                                <select name="rcp_category_mapping[<?php echo esc_attr( $cat ); ?>]">
                                    <option value="0">-- <?php esc_html_e( 'None', 'rcp-api-wp-integration' ); ?> --</option>
                                    <?php foreach ( $wp_cats as $wp_cat ) : ?>
                                        <option value="<?php echo esc_attr( $wp_cat->term_id ); ?>" <?php selected( $selected, $wp_cat->term_id ); ?>><?php echo esc_html( $wp_cat->name ); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                            <td class="rcp-featured">
                                <select name="<?php echo esc_attr( self::OPTION_CATEGORY_IMAGE_MODES ); ?>[<?php echo esc_attr( $normalized_category ); ?>]">
                                    <option value="none" <?php selected( $image_mode, 'none' ); ?>><?php esc_html_e( 'None (no featured image)', 'rcp-api-wp-integration' ); ?></option>
                                    <option value="import" <?php selected( $image_mode, 'import' ); ?>><?php esc_html_e( 'Import from RCP Source *', 'rcp-api-wp-integration' ); ?></option>
                                    <option value="ai" <?php selected( $image_mode, 'ai' ); ?>><?php esc_html_e( 'Generate AI image (always)', 'rcp-api-wp-integration' ); ?></option>
                                    <option value="fallback" <?php selected( $image_mode, 'fallback' ); ?>><?php esc_html_e( 'Import from RCP Source, else AI *', 'rcp-api-wp-integration' ); ?></option>
                                </select>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>

                <div class="postbox" style="margin-top: 20px;">
                    <h2 class="hndle"><?php esc_html_e( 'AI Credits', 'rcp-api-wp-integration' ); ?></h2>
                    <div class="inside">
                        <?php
                        $balance_label = '';
                        try {
                            if ( method_exists( $this, 'credits_request' ) ) {
                                $bal = $this->credits_request( 'GET', 'balance', null, [] );
                                if ( ! is_wp_error( $bal ) && is_array( $bal ) && isset( $bal['balance'] ) ) {
                                    $balance_val = (int) $bal['balance'];
                                    $balance_label = sprintf( esc_html__( 'Balance: %d', 'rcp-api-wp-integration' ), $balance_val );
                                }
                            }
                        } catch ( Exception $e ) {
                            // ignore
                        }
                        $credits_url = apply_filters( 'rcp_ai_credits_dashboard_url', 'https://app.radiocontentpro.com/account/credits/' );
                        ?>
                        <p>
                            <strong><?php echo $balance_label ? esc_html( $balance_label ) : esc_html__( 'Balance: unavailable', 'rcp-api-wp-integration' ); ?></strong>
                            &nbsp;•&nbsp;
                            <a href="<?php echo esc_url( $credits_url ); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Buy credits', 'rcp-api-wp-integration' ); ?></a>
                        </p>
                    </div>
                </div>

                <p class="description" style="margin-top: 8px;">
                    <?php
                    esc_html_e( 'Options marked with * download images from your RCP Source. AI options consume AI credits on your RCP account.', 'rcp-api-wp-integration' );
                    echo ' ';
                    $credits_link = apply_filters( 'rcp_ai_credits_dashboard_url', 'https://app.radiocontentpro.com/account/credits/' );
                    printf(
                        '<a href="%1$s" target="_blank" rel="noopener noreferrer">%2$s</a>',
                        esc_url( $credits_link ),
                        esc_html__( 'Track and manage your RCP AI credits here.', 'rcp-api-wp-integration' )
                    );
                    ?>
                </p>

                <details id="rcp-image-options-help" class="postbox" style="padding: 10px;">
                    <summary><strong><?php esc_html_e( 'Featured Image Options Explained', 'rcp-api-wp-integration' ); ?></strong></summary>
                    <ul style="margin-left: 18px; list-style: disc;">
                        <li><strong><?php esc_html_e( 'None (no featured image)', 'rcp-api-wp-integration' ); ?>:</strong> <?php esc_html_e( 'Do not set a featured image.', 'rcp-api-wp-integration' ); ?></li>
                        <li><strong><?php esc_html_e( 'Import from RCP Source', 'rcp-api-wp-integration' ); ?>:</strong> <?php esc_html_e( 'Download and attach the image provided by RCP.', 'rcp-api-wp-integration' ); ?></li>
                        <li><strong><?php esc_html_e( 'Generate AI image (always)', 'rcp-api-wp-integration' ); ?>:</strong> <?php esc_html_e( 'Always generate an AI image using credits.', 'rcp-api-wp-integration' ); ?></li>
                        <li><strong><?php esc_html_e( 'Import from RCP Source, else AI', 'rcp-api-wp-integration' ); ?>:</strong> <?php esc_html_e( 'Try RCP image first; if missing, generate an AI image.', 'rcp-api-wp-integration' ); ?></li>
                    </ul>
                </details>

                <details id="rcp-featured-disclaimer" class="rcp-alert">
                    <summary><strong><?php esc_html_e( 'Featured Image Import Disclaimer', 'rcp-api-wp-integration' ); ?></strong></summary>
                    <p><?php esc_html_e( 'Enabling automatic featured-image imports means images pulled from third-party websites may be subject to copyright protection. By turning this setting ON, you acknowledge and agree that:', 'rcp-api-wp-integration' ); ?></p>
                    <p><?php echo esc_html__( '1. You are solely responsible for verifying that you have the legal right (licenses, permissions, or fair-use justification) to display, reproduce, distribute, and/or modify any imported image.', 'rcp-api-wp-integration' ); ?></p>
                    <p><?php echo esc_html__( '2. Radio Content Pro and WP Media Inc. do not provide or guarantee any copyright clearance for images imported by this feature, nor do we warrant that the images are free of intellectual-property claims.', 'rcp-api-wp-integration' ); ?></p>
                    <p><?php echo esc_html__( '3. All liability arising from the use of imported images—including infringement claims, takedown demands, fees, or damages—rests entirely with you (the site owner/operator).', 'rcp-api-wp-integration' ); ?></p>
                    <p><?php echo esc_html__( '4. Radio Content Pro and WP Media Inc., together with their officers, employees, contractors, and affiliates, bear no responsibility and are fully indemnified and held harmless from any claims, losses, or expenses related to your use of imported images.', 'rcp-api-wp-integration' ); ?></p>
                    <p><?php echo esc_html__( '5. This disclaimer constitutes a material part of the agreement between you and WP Media Inc. regarding the plugin’s use. If you are uncertain about image rights, consult qualified legal counsel before enabling this feature.', 'rcp-api-wp-integration' ); ?></p>
                    <p><?php echo esc_html__( 'If you do not accept these terms, leave the “Import from RCP Source” options off and supply your own licensed images instead, or choose the “Generate AI image” option for a copyright-free graphic.', 'rcp-api-wp-integration' ); ?></p>
                    <p><em><?php esc_html_e( '* Options that import from the API may download copyrighted media. AI options consume AI credits on your RCP account.', 'rcp-api-wp-integration' ); ?></em></p>
                </details>

                <div id="rcp-automation-settings">
                    <h2><?php esc_html_e( 'Automation Settings', 'rcp-api-wp-integration' ); ?></h2>
                    <p>
                        <label for="rcp-cron-frequency"><?php esc_html_e( 'Auto Import Frequency', 'rcp-api-wp-integration' ); ?></label>
                        <select name="<?php echo esc_attr( self::OPTION_CRON_FREQUENCY ); ?>" id="rcp-cron-frequency">
                            <option value="rcp_15min" <?php selected( $frequency, 'rcp_15min' ); ?>><?php esc_html_e( 'Every 15 Minutes', 'rcp-api-wp-integration' ); ?></option>
                            <option value="rcp_30min" <?php selected( $frequency, 'rcp_30min' ); ?>><?php esc_html_e( 'Every 30 Minutes', 'rcp-api-wp-integration' ); ?></option>
                            <option value="hourly" <?php selected( $frequency, 'hourly' ); ?>><?php esc_html_e( 'Hourly', 'rcp-api-wp-integration' ); ?></option>
                        </select>
                    </p>
                    <?php
                    $last_import = get_option( self::OPTION_LAST_IMPORT, '' );
                    $last_ts     = $last_import ? ( is_numeric( $last_import ) ? (int) $last_import : strtotime( $last_import ) ) : 0;
                    $last_display = $last_ts ? date_i18n( 'H:i m/d/Y', $last_ts, true ) : __( 'Never', 'rcp-api-wp-integration' );
                    $next_run    = wp_next_scheduled( self::CRON_HOOK );
                    // wp_next_scheduled() returns a UTC timestamp, so pass $gmt = true
                    // to date_i18n() to convert it to the site's timezone.
                    $next_display = $next_run ? date_i18n( 'H:i m/d/Y', $next_run, true ) : __( 'Not scheduled', 'rcp-api-wp-integration' );
                    ?>
                    <p class="description">
                        <?php printf( esc_html__( 'Last Auto Import: %1$s | Next Scheduled Run: %2$s', 'rcp-api-wp-integration' ), esc_html( $last_display ), esc_html( $next_display ) ); ?>
                    </p>
                </div>
                
                <h2><?php esc_html_e( 'Debug Settings', 'rcp-api-wp-integration' ); ?></h2>
                <div class="postbox" style="padding: 15px;">
                    <label>
                        <input type="checkbox" name="<?php echo esc_attr( self::OPTION_DEBUG_MODE ); ?>" value="1" <?php checked( get_option( self::OPTION_DEBUG_MODE, false ) ); ?> />
                        <?php esc_html_e( 'Enable debug mode', 'rcp-api-wp-integration' ); ?>
                    </label>
                    <p class="description">
                        <?php esc_html_e( 'When enabled, detailed debug information will be written to the WordPress debug.log file (requires WP_DEBUG to be enabled).', 'rcp-api-wp-integration' ); ?>
                    </p>
                    <?php if ( get_option( self::OPTION_DEBUG_MODE, false ) && ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) ) : ?>
                        <p style="color: orange;">
                            <strong><?php esc_html_e( 'Warning:', 'rcp-api-wp-integration' ); ?></strong> 
                            <?php esc_html_e( 'Debug mode is enabled but WP_DEBUG is not active. Enable WP_DEBUG in wp-config.php to see debug logs.', 'rcp-api-wp-integration' ); ?>
                        </p>
                    <?php endif; ?>
                </div>

                <h2><?php esc_html_e( 'Next Import Preview', 'rcp-api-wp-integration' ); ?></h2>
                <div class="postbox" style="padding: 10px;">
                    <?php
                    // Quick preview of what would be imported
                    $test_results = $this->run_test_import( false );
                    if ( is_wp_error( $test_results ) ) {
                        echo '<p style="color: #666;">' . esc_html__( 'Unable to preview imports: ', 'rcp-api-wp-integration' ) . esc_html( $test_results->get_error_message() ) . '</p>';
                    } else {
                        if ( $test_results['would_import'] > 0 ) {
                            echo '<p style="color: green;">';
                            printf( 
                                esc_html( _n( 
                                    'Next import will process %d post', 
                                    'Next import will process %d posts', 
                                    $test_results['would_import'], 
                                    'rcp-api-wp-integration' 
                                ) ), 
                                $test_results['would_import'] 
                            );
                            echo '</p>';
                            if ( ! empty( $test_results['posts'] ) && count( $test_results['posts'] ) <= 5 ) {
                                echo '<ul style="margin: 0; padding-left: 20px;">';
                                foreach ( $test_results['posts'] as $post ) {
                                    echo '<li>';
                                    echo esc_html( $post['title'] );
                                    echo ' <span style="color: #666;">(' . esc_html( $post['import_status'] ) . ')</span>';
                                    echo '</li>';
                                }
                                echo '</ul>';
                            }
                        } else {
                            echo '<p style="color: #666;">' . esc_html__( 'No posts will be imported on the next run.', 'rcp-api-wp-integration' ) . '</p>';
                            if ( $test_results['skipped'] > 0 && ! empty( $test_results['skip_reasons'] ) ) {
                                echo '<p style="color: #666; font-size: 0.9em;">' . esc_html__( 'Reason: ', 'rcp-api-wp-integration' ) . esc_html( $test_results['skip_reasons'][0] ) . '</p>';
                            }
                        }
                        echo '<p style="margin-top: 10px;"><a href="' . esc_url( admin_url( 'admin.php?page=rcp-api-diagnostics' ) ) . '">' . esc_html__( 'View detailed test import →', 'rcp-api-wp-integration' ) . '</a></p>';
                    }
                    ?>
                </div>

                <?php submit_button(); ?>
            </form>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" id="rcp-force-import-form" aria-describedby="rcp-force-import-desc">
                <?php wp_nonce_field( 'rcp_force_auto_import' ); ?>
                <input type="hidden" name="action" value="rcp_force_auto_import" />
                <?php submit_button( __( 'Force Auto Import', 'rcp-api-wp-integration' ), 'secondary', 'submit', false, [ 'id' => 'rcp-force-import-button', 'aria-live' => 'polite' ] ); ?>
                <span id="rcp-force-import-status" style="margin-left:8px;"></span>
                <p id="rcp-force-import-desc" class="description"><?php esc_html_e( 'Imports posts missed since the last scheduled run.', 'rcp-api-wp-integration' ); ?></p>
            </form>
            <p class="description"><?php esc_html_e( 'If automatic imports are not working, make sure WordPress cron jobs can run on your server.', 'rcp-api-wp-integration' ); ?></p>
            <p><a href="https://radiocontentpro.com/wordpress-plugin/" target="_blank"><?php esc_html_e( 'Need help? Visit our WordPress Plugin page.', 'rcp-api-wp-integration' ); ?></a></p>
        </div>
        <?php
    }

    public function feed_page() {
        if ( ! current_user_can( self::CAPABILITY ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'RCP Feed', 'rcp-api-wp-integration' ); ?></h1>
            <div id="rcp-filter">
                <div id="rcp-category-container">
                <button type="button" id="rcp-category-toggle" class="button" aria-haspopup="true" aria-expanded="false" aria-controls="rcp-categories"><?php esc_html_e( 'Filter RCP posts by category...', 'rcp-api-wp-integration' ); ?></button>
                <div id="rcp-categories" class="rcp-dropdown" role="group" aria-label="<?php esc_attr_e( 'RCP Categories', 'rcp-api-wp-integration' ); ?>"></div>
                </div>
                <button id="rcp-refresh" class="button button-primary"><?php esc_html_e( 'Refresh Feed', 'rcp-api-wp-integration' ); ?></button>
            </div>
        
        <div id="rcp-bulk-actions" style="display: none;">
            <span id="rcp-selected-count">0 <?php esc_html_e( 'posts selected', 'rcp-api-wp-integration' ); ?></span>
            <button id="rcp-bulk-publish" class="button"><?php esc_html_e( 'Bulk Publish', 'rcp-api-wp-integration' ); ?></button>
            <button id="rcp-bulk-draft" class="button"><?php esc_html_e( 'Bulk Create Drafts', 'rcp-api-wp-integration' ); ?></button>
            <button id="rcp-clear-selection" class="button"><?php esc_html_e( 'Clear Selection', 'rcp-api-wp-integration' ); ?></button>
        </div>
        
        <div id="rcp-bulk-progress" style="display: none;">
            <div class="rcp-progress-bar">
                <div class="rcp-progress-fill" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0"></div>
            </div>
            <p id="rcp-progress-text" role="status" aria-live="polite"></p>
        </div>
            
            <p id="rcp-total-posts"></p>
            <table id="rcp-feed" class="wp-list-table widefat fixed striped">
                <thead>
                <tr>
                    <td class="check-column"><input type="checkbox" /></td>
                    <th><?php esc_html_e( 'Title', 'rcp-api-wp-integration' ); ?></th>
                    <th><?php esc_html_e( 'Categories', 'rcp-api-wp-integration' ); ?></th>
                    <th><?php esc_html_e( 'Published', 'rcp-api-wp-integration' ); ?></th>
                    <th><?php esc_html_e( 'Actions', 'rcp-api-wp-integration' ); ?></th>
                    <th><?php esc_html_e( 'RCP Source', 'rcp-api-wp-integration' ); ?></th>
                </tr>
                </thead>
                <tbody></tbody>
            </table>
            <div id="rcp-pagination">
                <button id="rcp-prev" class="button">&laquo; <?php esc_html_e( 'Previous', 'rcp-api-wp-integration' ); ?></button>
                <span id="rcp-page-links"><span id="rcp-page-num">1</span></span>
                <button id="rcp-next" class="button"><?php esc_html_e( 'Next', 'rcp-api-wp-integration' ); ?> &raquo;</button>
            </div>
        </div>
        <?php
    }

    public function info_page() {
        if ( ! current_user_can( self::CAPABILITY ) ) {
            return;
        }

        $username = get_option( self::OPTION_USERNAME );
        $password = get_option( self::OPTION_PASSWORD );
        $status   = $this->check_connection();

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'API Connection', 'rcp-api-wp-integration' ); ?></h1>
            <p class="rcp-connection-status <?php echo true === $status ? 'connected' : 'disconnected'; ?>">
                <span class="indicator"></span>
                <strong><?php esc_html_e( 'API Connection Status:', 'rcp-api-wp-integration' ); ?></strong>
                <?php
                if ( true === $status ) {
                    esc_html_e( 'Connected', 'rcp-api-wp-integration' );
                } else {
                    echo esc_html__( 'Connection failed', 'rcp-api-wp-integration' ) . ': ' . esc_html( $status );
                }
                ?>
            </p>
            <form method="post" action="options.php">
                <?php settings_fields( 'rcp-api-wp-integration' ); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="rcp_username"><?php esc_html_e( 'RCP Email Address', 'rcp-api-wp-integration' ); ?></label></th>
                        <td><input name="<?php echo esc_attr( self::OPTION_USERNAME ); ?>" id="rcp_username" value="<?php echo esc_attr( $username ); ?>" class="regular-text" type="text" /></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="rcp_password"><?php esc_html_e( 'Application Password', 'rcp-api-wp-integration' ); ?></label></th>
                        <td><input name="<?php echo esc_attr( self::OPTION_PASSWORD ); ?>" id="rcp_password" value="<?php echo esc_attr( $password ); ?>" class="regular-text" type="password" /></td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
            <?php if ( true === $status ) : ?>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'rcp_api_disconnect' ); ?>
                <input type="hidden" name="action" value="rcp_api_disconnect" />
                <?php submit_button( __( 'Disconnect', 'rcp-api-wp-integration' ), 'delete', 'submit', false ); ?>
            </form>
            <?php endif; ?>
            <p><a href="https://radiocontentpro.com/wordpress-plugin/" target="_blank"><?php esc_html_e( 'Need help? Visit our WordPress Plugin page.', 'rcp-api-wp-integration' ); ?></a></p>
        </div>
        <?php
    }

    public function log_page() {
        if ( ! current_user_can( self::CAPABILITY ) ) {
            return;
        }
        
        // Get and validate log data
        $log = get_option( self::OPTION_IMPORT_LOG, [] );
        if ( ! is_array( $log ) ) {
            $log = [];
        }
        
        // Clean up old entries
        $cutoff = current_time( 'timestamp', true ) - self::LOG_RETENTION_HOURS * HOUR_IN_SECONDS;
        $log = array_filter( $log, function( $row ) use ( $cutoff ) {
            return isset( $row['time'] ) && (int) $row['time'] >= $cutoff;
        } );
        
        // Sort by time, newest first
        usort( $log, function( $a, $b ) {
            return ( $b['time'] ?? 0 ) - ( $a['time'] ?? 0 );
        } );
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Automation Log', 'rcp-api-wp-integration' ); ?></h1>
            <p><?php echo esc_html( sprintf( __( 'Showing import activities from the last %d hours.', 'rcp-api-wp-integration' ), self::LOG_RETENTION_HOURS ) ); ?></p>

            <?php
            // Optional DB-backed history preview (last 20 entries)
            if ( class_exists( 'RCP_API_Database_Manager' ) ) {
                $history = RCP_API_Database_Manager::get_import_history( [ 'limit' => 20 ] );
                if ( ! empty( $history ) ) {
                    echo '<h2>' . esc_html__( 'Recent Database Log', 'rcp-api-wp-integration' ) . '</h2>';
                    echo '<table class="widefat fixed striped"><thead><tr>';
                    echo '<th>' . esc_html__( 'Time', 'rcp-api-wp-integration' ) . '</th>';
                    echo '<th>' . esc_html__( 'Type', 'rcp-api-wp-integration' ) . '</th>';
                    echo '<th>' . esc_html__( 'Status', 'rcp-api-wp-integration' ) . '</th>';
                    echo '<th>' . esc_html__( 'RCP ID', 'rcp-api-wp-integration' ) . '</th>';
                    echo '<th>' . esc_html__( 'WP ID', 'rcp-api-wp-integration' ) . '</th>';
                    echo '</tr></thead><tbody>';
                    foreach ( $history as $hrow ) {
                        echo '<tr>';
                        echo '<td>' . esc_html( $hrow['imported_at'] ?? '' ) . '</td>';
                        echo '<td>' . esc_html( $hrow['import_type'] ?? '' ) . '</td>';
                        echo '<td>' . esc_html( $hrow['status'] ?? '' ) . '</td>';
                        echo '<td>' . esc_html( $hrow['rcp_post_id'] ?? '' ) . '</td>';
                        echo '<td>' . esc_html( $hrow['wp_post_id'] ?? '' ) . '</td>';
                        echo '</tr>';
                    }
                    echo '</tbody></table>';
                }
            }
            ?>
            
            <?php if ( ! empty( $_GET['rcp_import'] ) && 'forced' === $_GET['rcp_import'] ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Force import completed successfully. Check the log entries below for details.', 'rcp-api-wp-integration' ); ?></p>
                </div>
            <?php endif; ?>
            
            <table class="widefat fixed striped">
                <thead>
                <tr>
                    <th style="width: 150px;"><?php esc_html_e( 'Time', 'rcp-api-wp-integration' ); ?></th>
                    <th style="width: 100px;"><?php esc_html_e( 'Type', 'rcp-api-wp-integration' ); ?></th>
                    <th style="width: 80px;"><?php esc_html_e( 'Published', 'rcp-api-wp-integration' ); ?></th>
                    <th style="width: 80px;"><?php esc_html_e( 'Drafted', 'rcp-api-wp-integration' ); ?></th>
                    <th style="width: 100px;"><?php esc_html_e( 'Status', 'rcp-api-wp-integration' ); ?></th>
                    <th><?php esc_html_e( 'Details', 'rcp-api-wp-integration' ); ?></th>
                </tr>
                </thead>
                <tbody>
                <?php if ( empty( $log ) ) : ?>
                    <tr>
                        <td colspan="6"><?php esc_html_e( 'No log entries found. Import activities will appear here once automation runs or manual imports are performed.', 'rcp-api-wp-integration' ); ?></td>
                    </tr>
                <?php else : ?>
                    <?php foreach ( $log as $row ) : ?>
                        <?php
                        $published_count = count( (array) ( $row['published'] ?? [] ) );
                        $drafted_count = count( (array) ( $row['drafted'] ?? [] ) );
                        $error_count = count( (array) ( $row['errors'] ?? [] ) );
                        $import_type = $row['debug']['import_type'] ?? 'automation';
                        $success = $row['success'] ?? ( $error_count === 0 );
                        ?>
                        <tr>
                            <td><?php echo esc_html( date_i18n( 'Y-m-d H:i:s', (int) $row['time'] ) ); ?></td>
                            <td>
                                <?php 
                                // More specific import type labels
                                switch ( $import_type ) {
                                    case 'scheduled':
                                        $type_label = __( 'Scheduled', 'rcp-api-wp-integration' );
                                        break;
                                    case 'force_auto':
                                        $type_label = __( 'Force Auto', 'rcp-api-wp-integration' );
                                        break;
                                    case 'manual_single':
                                        $type_label = __( 'Manual', 'rcp-api-wp-integration' );
                                        break;
                                    case 'bulk':
                                        $type_label = __( 'Bulk Import', 'rcp-api-wp-integration' );
                                        break;
                                    case 'manual':
                                        $type_label = __( 'Manual', 'rcp-api-wp-integration' );
                                        break;
                                    default:
                                        $type_label = __( 'Automation', 'rcp-api-wp-integration' );
                                        break;
                                }
                                echo esc_html( $type_label );
                                ?>
                            </td>
                            <td><?php echo esc_html( $published_count ); ?></td>
                            <td><?php echo esc_html( $drafted_count ); ?></td>
                            <td>
                                <?php if ( $success ) : ?>
                                    <span style="color: #46b450;"><?php esc_html_e( 'Success', 'rcp-api-wp-integration' ); ?></span>
                                <?php else : ?>
                                    <span style="color: #dc3232;"><?php esc_html_e( 'Errors', 'rcp-api-wp-integration' ); ?></span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ( $published_count > 0 || $drafted_count > 0 || $error_count > 0 || ! empty( $row['debug'] ) ) : ?>
                                    <details>
                                        <summary style="cursor: pointer;">
                                            <?php 
                                            $summary_parts = [];
                                            if ( $published_count > 0 ) {
                                                $summary_parts[] = sprintf( _n( '%d published', '%d published', $published_count, 'rcp-api-wp-integration' ), $published_count );
                                            }
                                            if ( $drafted_count > 0 ) {
                                                $summary_parts[] = sprintf( _n( '%d drafted', '%d drafted', $drafted_count, 'rcp-api-wp-integration' ), $drafted_count );
                                            }
                                            if ( $error_count > 0 ) {
                                                $summary_parts[] = sprintf( _n( '%d error', '%d errors', $error_count, 'rcp-api-wp-integration' ), $error_count );
                                            }
                                            echo esc_html( ! empty( $summary_parts ) ? implode( ', ', $summary_parts ) : __( 'View details', 'rcp-api-wp-integration' ) );
                                            ?>
                                        </summary>
                                        <div style="max-height: 300px; overflow-y: auto; background: #f9f9f9; padding: 10px; margin-top: 5px; font-size: 12px;">
                                            <?php 
                                            // Show imported post IDs
                                            if ( ! empty( $row['published'] ) ) {
                                                echo '<strong>' . esc_html__( 'Published Posts:', 'rcp-api-wp-integration' ) . '</strong><br>';
                                                foreach ( (array) $row['published'] as $post_id ) {
                                                    $edit_link = get_edit_post_link( $post_id );
                                                    if ( $edit_link ) {
                                                        echo '• <a href="' . esc_url( $edit_link ) . '" target="_blank">' . esc_html( get_the_title( $post_id ) ?: "Post #$post_id" ) . '</a><br>';
                                                    } else {
                                                        echo '• ' . esc_html( "Post #$post_id" ) . '<br>';
                                                    }
                                                }
                                                echo '<br>';
                                            }
                                            
                                            if ( ! empty( $row['drafted'] ) ) {
                                                echo '<strong>' . esc_html__( 'Drafted Posts:', 'rcp-api-wp-integration' ) . '</strong><br>';
                                                foreach ( (array) $row['drafted'] as $post_id ) {
                                                    $edit_link = get_edit_post_link( $post_id );
                                                    if ( $edit_link ) {
                                                        echo '• <a href="' . esc_url( $edit_link ) . '" target="_blank">' . esc_html( get_the_title( $post_id ) ?: "Post #$post_id" ) . '</a><br>';
                                                    } else {
                                                        echo '• ' . esc_html( "Post #$post_id" ) . '<br>';
                                                    }
                                                }
                                                echo '<br>';
                                            }
                                            
                                            // Show errors
                                            if ( ! empty( $row['errors'] ) ) {
                                                echo '<strong style="color: #dc3232;">' . esc_html__( 'Errors:', 'rcp-api-wp-integration' ) . '</strong><br>';
                                                foreach ( (array) $row['errors'] as $err ) {
                                                    echo '• ' . esc_html( $err ) . '<br>';
                                                }
                                                echo '<br>';
                                            }
                                            
                                            // Show debug info
                                            $debug = $row['debug'] ?? [];
                                            if ( isset( $debug['posts_processed'] ) || isset( $debug['posts_skipped'] ) || isset( $debug['posts_imported'] ) ) {
                                                echo '<strong>' . esc_html__( 'Summary:', 'rcp-api-wp-integration' ) . '</strong><br>';
                                                if ( isset( $debug['posts_processed'] ) ) {
                                                    echo esc_html__( 'Posts Processed:', 'rcp-api-wp-integration' ) . ' ' . esc_html( $debug['posts_processed'] ) . '<br>';
                                                }
                                                if ( isset( $debug['posts_skipped'] ) ) {
                                                    echo esc_html__( 'Posts Skipped:', 'rcp-api-wp-integration' ) . ' ' . esc_html( $debug['posts_skipped'] ) . '<br>';
                                                }
                                                if ( isset( $debug['posts_imported'] ) ) {
                                                    echo esc_html__( 'Posts Imported:', 'rcp-api-wp-integration' ) . ' ' . esc_html( $debug['posts_imported'] ) . '<br>';
                                                }
                                                echo '<br>';
                                            }
                                            
                                            // Show debug messages if WP_DEBUG is enabled
                                            if ( defined( 'WP_DEBUG' ) && WP_DEBUG && ! empty( $debug['debug_messages'] ) ) {
                                                echo '<strong>' . esc_html__( 'Debug Messages:', 'rcp-api-wp-integration' ) . '</strong><br>';
                                                echo '<pre style="white-space: pre-wrap; font-size: 11px;">';
                                                foreach ( (array) $debug['debug_messages'] as $msg ) {
                                                    echo esc_html( $msg ) . "\n";
                                                }
                                                echo '</pre>';
                                            }
                                            ?>
                                        </div>
                                    </details>
                                <?php else : ?>
                                    <?php esc_html_e( 'No activity', 'rcp-api-wp-integration' ); ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>
            
            <p class="description">
                <?php 
                echo esc_html( sprintf( 
                    __( 'Log entries are retained for %d hours. The log shows both automated imports and manual imports triggered via the Feed page.', 'rcp-api-wp-integration' ), 
                    self::LOG_RETENTION_HOURS 
                ) ); 
                ?>
            </p>
        </div>
        <?php
    }
    
    /**
     * Display the diagnostics page for troubleshooting automation issues.
     */
    public function diagnostics_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'RCP API Diagnostics', 'rcp-api-wp-integration' ); ?></h1>
            <div class="notice notice-info" style="margin-top:10px;">
                <p>
                    <?php echo esc_html__( 'Use this page to verify cron health, next/last runs, and connection status. If imports stall, clear the import lock, confirm your schedule, and check the API Connection page.', 'rcp-api-wp-integration' ); ?>
                </p>
                <p>
                    <?php echo esc_html__( 'Tip: Featured images are controlled per category in Settings → Featured Image Options. “Import from RCP Source, else AI” tries your RCP image first and falls back to an AI image if none is available.', 'rcp-api-wp-integration' ); ?>
                </p>
            </div>
            
            <?php if ( isset( $_GET['lock_cleared'] ) && $_GET['lock_cleared'] == '1' ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Import lock has been cleared successfully.', 'rcp-api-wp-integration' ); ?></p>
                </div>
            <?php endif; ?>
            
            <div class="notice notice-info">
                <p><?php esc_html_e( 'This page provides diagnostic information to help troubleshoot automation issues.', 'rcp-api-wp-integration' ); ?></p>
            </div>
            
            <?php
            // Get diagnostic data
            $diagnostics = $this->get_diagnostics_data();
            ?>
            <div style="margin:10px 0;">
                <label><input type="checkbox" id="rcp-auto-refresh" /> <?php esc_html_e( 'Auto-refresh diagnostics every 30s', 'rcp-api-wp-integration' ); ?></label>
                <script>
                    (function(){
                        var timer;
                        function arm(){
                            clearInterval(timer);
                            if(document.getElementById('rcp-auto-refresh').checked){
                                timer = setInterval(function(){ location.reload(); }, 30000);
                            }
                        }
                        document.addEventListener('DOMContentLoaded', arm);
                        document.getElementById('rcp-auto-refresh').addEventListener('change', arm);
                    })();
                </script>
            </div>
            <?php
            ?>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'AI Images (Recent Activity)', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <p class="description" style="margin-top:0;">
                        <?php echo esc_html__( 'Recent per‑post AI generation results. Generate from the Feed (Generate AI Image button) or enable per‑category modes in Settings.', 'rcp-api-wp-integration' ); ?>
                    </p>
                    <?php $ai_log = get_option( 'rcp_ai_image_log', [] ); if ( ! is_array( $ai_log ) ) { $ai_log = []; } ?>
                    <?php if ( empty( $ai_log ) ) : ?>
                        <p><?php esc_html_e( 'No recent AI image activity.', 'rcp-api-wp-integration' ); ?></p>
                    <?php else : ?>
                        <table class="widefat">
                            <thead><tr>
                                <th><?php esc_html_e( 'Time (UTC)', 'rcp-api-wp-integration' ); ?></th>
                                <th><?php esc_html_e( 'Post', 'rcp-api-wp-integration' ); ?></th>
                                <th><?php esc_html_e( 'Status', 'rcp-api-wp-integration' ); ?></th>
                                <th><?php esc_html_e( 'Details', 'rcp-api-wp-integration' ); ?></th>
                            </tr></thead>
                            <tbody>
                            <?php foreach ( array_reverse( array_slice( $ai_log, -50 ) ) as $row ) : ?>
                                <tr>
                                    <td><?php echo esc_html( gmdate( 'Y-m-d H:i:s', (int) ( $row['time'] ?? time() ) ) ); ?></td>
                                    <td>
                                        <?php $pid = (int) ( $row['post_id'] ?? 0 );
                                        if ( $pid ) {
                                            $edit = get_edit_post_link( $pid );
                                            echo $edit ? '<a href="' . esc_url( $edit ) . '" target="_blank">' . esc_html( get_the_title( $pid ) ?: ('Post #'.$pid) ) . '</a>' : esc_html( 'Post #'.$pid );
                                        } else {
                                            echo '&ndash;';
                                        } ?>
                                    </td>
                                    <td><?php echo esc_html( (string) ( $row['status'] ?? '-' ) ); ?></td>
                                    <td>
                                        <?php
                                        $parts = [];
                                        if ( ! empty( $row['generation_log_id'] ) ) { $parts[] = 'log_id=' . sanitize_text_field( $row['generation_log_id'] ); }
                                        if ( ! empty( $row['attachment_id'] ) ) { $parts[] = 'attachment_id=' . (int) $row['attachment_id']; }
                                        if ( ! empty( $row['result_url'] ) ) { $parts[] = 'url=' . esc_url( $row['result_url'] ); }
                                        if ( ! empty( $row['message'] ) ) { $parts[] = (string) $row['message']; }
                                        echo esc_html( implode( ' | ', $parts ) );
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>

            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Rate Limits', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'AJAX Feed', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php $rl = $diagnostics['rate_limits']['ajax'] ?? []; ?>
                                <?php echo esc_html( sprintf( 'Limit: %s, Remaining: %s, Resets: %s', $rl['limit'] ?? '-', $rl['remaining'] ?? '-', isset($rl['reset']) ? gmdate('H:i:s', $rl['reset']) . ' UTC' : '-' ) ); ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Imports', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php $rl = $diagnostics['rate_limits']['import'] ?? []; ?>
                                <?php echo esc_html( sprintf( 'Limit: %s, Remaining: %s, Resets: %s', $rl['limit'] ?? '-', $rl['remaining'] ?? '-', isset($rl['reset']) ? gmdate('H:i:s', $rl['reset']) . ' UTC' : '-' ) ); ?>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>

            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Import Window Settings', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Maximum Post Age', 'rcp-api-wp-integration' ); ?></th>
                            <td>36 hours <span style="color: #666;"><?php esc_html_e( '(Posts older than this are never imported)', 'rcp-api-wp-integration' ); ?></span></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Cutoff Update Window', 'rcp-api-wp-integration' ); ?></th>
                            <td>10 minutes <span style="color: #666;"><?php esc_html_e( '(After import, cutoff moves to 10 minutes ago)', 'rcp-api-wp-integration' ); ?></span></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Grace Period', 'rcp-api-wp-integration' ); ?></th>
                            <td>60 seconds <span style="color: #666;"><?php esc_html_e( '(Posts within this window of cutoff are included)', 'rcp-api-wp-integration' ); ?></span></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Import Lock Timeout', 'rcp-api-wp-integration' ); ?></th>
                            <td>5 minutes <span style="color: #666;"><?php esc_html_e( '(Prevents concurrent imports)', 'rcp-api-wp-integration' ); ?></span></td>
                        </tr>
                    </table>
                </div>
            </div>

            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Cache Status', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <?php $cache = $diagnostics['cache'] ?? []; ?>
                    <p><?php esc_html_e( 'Object Cache:', 'rcp-api-wp-integration' ); ?>
                        <?php echo esc_html( ( $cache['object_cache']['enabled'] ?? false ) ? ( 'enabled (' . ( $cache['object_cache']['type'] ?? 'unknown' ) . ')' ) : 'disabled' ); ?></p>
                    <table class="widefat">
                        <thead><tr><th><?php esc_html_e( 'Group', 'rcp-api-wp-integration' ); ?></th><th><?php esc_html_e( 'Version', 'rcp-api-wp-integration' ); ?></th></tr></thead>
                        <tbody>
                        <?php foreach ( (array) ( $cache['groups'] ?? [] ) as $g => $row ) : ?>
                            <tr><td><?php echo esc_html( $g ); ?></td><td><?php echo esc_html( $row['version'] ?? 0 ); ?></td></tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'System Status', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Plugin Version', 'rcp-api-wp-integration' ); ?></th>
                            <td><?php echo esc_html( self::VERSION ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'WordPress Version', 'rcp-api-wp-integration' ); ?></th>
                            <td><?php echo esc_html( get_bloginfo( 'version' ) ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Server Time (UTC)', 'rcp-api-wp-integration' ); ?></th>
                            <td><?php echo esc_html( gmdate( 'Y-m-d H:i:s' ) ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'WordPress Time', 'rcp-api-wp-integration' ); ?></th>
                            <td><?php echo esc_html( current_time( 'mysql' ) ); ?> (<?php echo esc_html( get_option( 'timezone_string' ) ?: 'UTC' . get_option( 'gmt_offset' ) ); ?>)</td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'API Connection', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php if ( $diagnostics['api_connected'] ) : ?>
                                    <span style="color: green;">✓ <?php esc_html_e( 'Connected', 'rcp-api-wp-integration' ); ?></span>
                                <?php else : ?>
                                    <span style="color: red;">✗ <?php esc_html_e( 'Not Connected', 'rcp-api-wp-integration' ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Deleted Posts Tracking', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php 
                                echo esc_html( sprintf( 
                                    _n( '%d deleted post tracked', '%d deleted posts tracked', $diagnostics['deleted_posts_count'], 'rcp-api-wp-integration' ), 
                                    $diagnostics['deleted_posts_count'] 
                                ) );
                                if ( $diagnostics['deleted_posts_count'] > 0 ) {
                                    echo ' <span style="color: #666;">(' . esc_html__( 'will not be auto-imported', 'rcp-api-wp-integration' ) . ')</span>';
                                }
                                ?>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Automation Status', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Cron Health', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php if ( $diagnostics['cron_healthy'] ) : ?>
                                    <span style="color: green;">✓ <?php echo esc_html( $diagnostics['cron_status'] ); ?></span>
                                <?php else : ?>
                                    <span style="color: red;">✗ <?php echo esc_html( $diagnostics['cron_status'] ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Cron Frequency', 'rcp-api-wp-integration' ); ?></th>
                            <td><?php echo esc_html( $diagnostics['cron_frequency'] ); ?></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Next Scheduled Import', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php if ( $diagnostics['next_scheduled'] ) : ?>
                                    <?php echo esc_html( gmdate( 'Y-m-d H:i:s', $diagnostics['next_scheduled'] ) ); ?> UTC
                                    (<?php echo esc_html( human_time_diff( current_time( 'timestamp', true ), $diagnostics['next_scheduled'] ) ); ?> from now)
                                <?php else : ?>
                                    <span style="color: red;"><?php esc_html_e( 'Not scheduled', 'rcp-api-wp-integration' ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Last Import Attempt', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php if ( $diagnostics['last_import'] ) : ?>
                                    <?php echo esc_html( gmdate( 'Y-m-d H:i:s', $diagnostics['last_import'] ) ); ?> UTC
                                    (<?php echo esc_html( human_time_diff( $diagnostics['last_import'], current_time( 'timestamp', true ) ) ); ?> ago)
                                <?php else : ?>
                                    <?php esc_html_e( 'Never', 'rcp-api-wp-integration' ); ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Last Successful Import', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php if ( $diagnostics['last_success'] ) : ?>
                                    <?php echo esc_html( gmdate( 'Y-m-d H:i:s', $diagnostics['last_success'] ) ); ?> UTC
                                    (<?php echo esc_html( human_time_diff( $diagnostics['last_success'], current_time( 'timestamp', true ) ) ); ?> ago)
                                    <?php if ( $diagnostics['last_success'] != $diagnostics['last_import'] ) : ?>
                                        <span style="color: orange;"><?php esc_html_e( '(Different from last attempt)', 'rcp-api-wp-integration' ); ?></span>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <?php esc_html_e( 'Never', 'rcp-api-wp-integration' ); ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Import Lock Status', 'rcp-api-wp-integration' ); ?></th>
                            <td>
                                <?php if ( $diagnostics['import_locked'] ) : ?>
                                    <span style="color: <?php echo strpos( $diagnostics['lock_status'], 'Stale' ) !== false ? 'orange' : 'blue'; ?>;">
                                        🔒 <?php echo esc_html( $diagnostics['lock_status'] ); ?>
                                    </span>
                                    <?php if ( strpos( $diagnostics['lock_status'], 'Stale' ) !== false ) : ?>
                                        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display: inline; margin-left: 10px;">
                                            <?php wp_nonce_field( 'rcp_clear_import_lock' ); ?>
                                            <input type="hidden" name="action" value="rcp_clear_import_lock">
                                            <button type="submit" class="button button-small"><?php esc_html_e( 'Clear Lock', 'rcp-api-wp-integration' ); ?></button>
                                        </form>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <span style="color: green;">✓ <?php echo esc_html( $diagnostics['lock_status'] ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Category Automation Settings', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <?php if ( ! empty( $diagnostics['automation_settings'] ) ) : ?>
                        <table class="widefat">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Category', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Mode', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Author', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Cutoff Time', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Cutoff Age', 'rcp-api-wp-integration' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $diagnostics['automation_settings'] as $cat => $mode ) : ?>
                                    <tr>
                                        <td>
                                            <?php echo esc_html( $cat ); ?>
                                            <?php 
                                            $normalized = $this->normalize_category_name( $cat );
                                            if ( $normalized !== strtolower( $cat ) ) {
                                                echo '<br><small style="color: #666;">' . esc_html__( 'Normalized:', 'rcp-api-wp-integration' ) . ' ' . esc_html( $normalized ) . '</small>';
                                            }
                                            ?>
                                        </td>
                                        <td><?php echo esc_html( $mode ); ?></td>
                                        <td>
                                            <?php 
                                            $author_id = $diagnostics['category_authors'][$cat] ?? 0;
                                            if ( $author_id ) {
                                                $user = get_user_by( 'id', $author_id );
                                                echo $user ? esc_html( $user->display_name ) : 'User #' . $author_id;
                                            } else {
                                                echo '&ndash;';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <?php 
                                            $normalized_cat = $this->normalize_category_name( $cat );
                                            $cutoff = 0;
                                            // Check both original and normalized names for backwards compatibility
                                            if ( isset( $diagnostics['category_cutoffs'][$cat] ) ) {
                                                $cutoff = $diagnostics['category_cutoffs'][$cat];
                                            } elseif ( isset( $diagnostics['category_cutoffs'][$normalized_cat] ) ) {
                                                $cutoff = $diagnostics['category_cutoffs'][$normalized_cat];
                                            }
                                            
                                            if ( $cutoff ) {
                                                echo esc_html( gmdate( 'Y-m-d H:i:s', $cutoff ) ) . ' UTC';
                                                echo '<br><small style="color: #666;">';
                                                echo esc_html__( 'Will import posts newer than: ', 'rcp-api-wp-integration' );
                                                echo esc_html( gmdate( 'Y-m-d H:i:s', $cutoff - 60 ) ) . ' UTC';
                                                echo ' <em>(' . esc_html__( '60s grace period', 'rcp-api-wp-integration' ) . ')</em>';
                                                echo '</small>';
                                            } else {
                                                echo '<span style="color: orange;">' . esc_html__( 'No cutoff set', 'rcp-api-wp-integration' ) . '</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <?php 
                                            if ( $cutoff ) {
                                                $age = current_time( 'timestamp', true ) - $cutoff;
                                                echo esc_html( human_time_diff( $cutoff, current_time( 'timestamp', true ) ) ) . ' ago';
                                                if ( $age > 72 * HOUR_IN_SECONDS ) {
                                                    echo ' <span style="color: orange;">' . esc_html__( '(may be too old)', 'rcp-api-wp-integration' ) . '</span>';
                                                }
                                            } else {
                                                echo '<span style="color: green;">' . esc_html__( 'Will import all recent posts', 'rcp-api-wp-integration' ) . '</span>';
                                            }
                                            ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else : ?>
                        <p><?php esc_html_e( 'No category automation configured.', 'rcp-api-wp-integration' ); ?></p>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Recent Import Activity', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <?php
                    $log = get_option( self::OPTION_IMPORT_LOG, [] );
                    if ( ! empty( $log ) && is_array( $log ) ) {
                        $recent = array_slice( $log, -5 );
                        ?>
                        <table class="widefat">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Time', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Summary', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Posts Processed', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Posts Skipped', 'rcp-api-wp-integration' ); ?></th>
                                    <th><?php esc_html_e( 'Details', 'rcp-api-wp-integration' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( array_reverse( $recent ) as $entry ) : ?>
                                    <tr>
                                        <td><?php echo isset( $entry['time'] ) ? esc_html( gmdate( 'Y-m-d H:i:s', $entry['time'] ) ) : '&ndash;'; ?></td>
                                        <td>
                                            <?php
                                            $published = count( $entry['published'] ?? [] );
                                            $drafted = count( $entry['drafted'] ?? [] );
                                            $errors = count( $entry['errors'] ?? [] );
                                            
                                            $parts = [];
                                            if ( $published > 0 ) $parts[] = $published . ' published';
                                            if ( $drafted > 0 ) $parts[] = $drafted . ' drafted';
                                            if ( $errors > 0 ) $parts[] = '<span style="color: red;">' . $errors . ' errors</span>';
                                            
                                            echo ! empty( $parts ) ? implode( ', ', $parts ) : 'No changes';
                                            ?>
                                        </td>
                                        <td><?php echo isset( $entry['debug']['posts_processed'] ) ? esc_html( $entry['debug']['posts_processed'] ) : '&ndash;'; ?></td>
                                        <td><?php echo isset( $entry['debug']['posts_skipped'] ) ? esc_html( $entry['debug']['posts_skipped'] ) : '&ndash;'; ?></td>
                                        <td>
                                            <?php if ( ! empty( $entry['errors'] ) || ! empty( $entry['debug']['debug_messages'] ) ) : ?>
                                                <details style="cursor: pointer;">
                                                    <summary><?php esc_html_e( 'View', 'rcp-api-wp-integration' ); ?></summary>
                                                    <?php if ( ! empty( $entry['errors'] ) ) : ?>
                                                        <h4 style="margin: 10px 0 5px;"><?php esc_html_e( 'Errors:', 'rcp-api-wp-integration' ); ?></h4>
                                                        <ul style="margin: 0; list-style-type: disc; padding-left: 20px;">
                                                            <?php foreach ( $entry['errors'] as $error ) : ?>
                                                                <li style="color: red;"><?php echo esc_html( $error ); ?></li>
                                                            <?php endforeach; ?>
                                                        </ul>
                                                    <?php endif; ?>
                                                    <?php if ( ! empty( $entry['info'] ) ) : ?>
                                                        <h4 style="margin: 10px 0 5px;"><?php esc_html_e( 'Info:', 'rcp-api-wp-integration' ); ?></h4>
                                                        <ul style="margin: 0; list-style-type: disc; padding-left: 20px;">
                                                            <?php foreach ( $entry['info'] as $info ) : ?>
                                                                <li style="color: blue;"><?php echo esc_html( $info ); ?></li>
                                                            <?php endforeach; ?>
                                                        </ul>
                                                    <?php endif; ?>
                                                    <?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG && ! empty( $entry['debug']['debug_messages'] ) ) : ?>
                                                        <h4 style="margin: 10px 0 5px;"><?php esc_html_e( 'Debug Log:', 'rcp-api-wp-integration' ); ?></h4>
                                                        <pre style="background: #f5f5f5; padding: 10px; overflow-x: auto; font-size: 11px; max-height: 200px; overflow-y: auto;"><?php
                                                            $debug_output = is_array( $entry['debug']['debug_messages'] ) ? 
                                                                implode( "\n", array_slice( $entry['debug']['debug_messages'], 0, 50 ) ) : 
                                                                $entry['debug']['debug_messages'];
                                                            echo esc_html( $debug_output );
                                                            if ( is_array( $entry['debug']['debug_messages'] ) && count( $entry['debug']['debug_messages'] ) > 50 ) {
                                                                echo "\n... (" . ( count( $entry['debug']['debug_messages'] ) - 50 ) . " more lines)";
                                                            }
                                                        ?></pre>
                                                    <?php endif; ?>
                                                </details>
                                            <?php else : ?>
                                                &ndash;
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php } else { ?>
                        <p><?php esc_html_e( 'No import activity recorded.', 'rcp-api-wp-integration' ); ?></p>
                    <?php } ?>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Test Import (Dry Run)', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <p><?php esc_html_e( 'Run a simulated import to see what posts would be imported without actually importing them.', 'rcp-api-wp-integration' ); ?></p>
                    <form method="post" action="">
                        <?php wp_nonce_field( 'rcp_test_import' ); ?>
                        <input type="hidden" name="action" value="rcp_test_import">
                        <p>
                            <label>
                                <input type="checkbox" name="ignore_cutoffs" value="1">
                                <?php esc_html_e( 'Ignore category cutoffs (show all posts from last 36 hours)', 'rcp-api-wp-integration' ); ?>
                            </label>
                        </p>
                        <p class="submit">
                            <button type="submit" class="button button-primary"><?php esc_html_e( 'Run Test Import', 'rcp-api-wp-integration' ); ?></button>
                        </p>
                    </form>
                    
                    <?php
                    // Handle test import
                    if ( isset( $_POST['action'] ) && $_POST['action'] === 'rcp_test_import' && check_admin_referer( 'rcp_test_import' ) ) {
                        $test_results = $this->run_test_import( isset( $_POST['ignore_cutoffs'] ) );
                        
                        if ( is_wp_error( $test_results ) ) {
                            echo '<div class="notice notice-error"><p>' . esc_html( $test_results->get_error_message() ) . '</p></div>';
                        } else {
                            ?>
                            <h3><?php esc_html_e( 'Test Results', 'rcp-api-wp-integration' ); ?></h3>
                            <div style="background: #f0f0f0; padding: 10px; border: 1px solid #ccc; max-height: 400px; overflow-y: auto;">
                                <p><strong><?php esc_html_e( 'Summary:', 'rcp-api-wp-integration' ); ?></strong></p>
                                <ul>
                                    <li><?php printf( esc_html__( 'Total posts fetched: %d', 'rcp-api-wp-integration' ), $test_results['total_posts'] ); ?></li>
                                    <li><?php printf( esc_html__( 'Posts that would be imported: %d', 'rcp-api-wp-integration' ), $test_results['would_import'] ); ?></li>
                                    <li><?php printf( esc_html__( 'Posts skipped: %d', 'rcp-api-wp-integration' ), $test_results['skipped'] ); ?></li>
                                </ul>
                                
                                <?php if ( ! empty( $test_results['posts'] ) ) : ?>
                                    <h4><?php esc_html_e( 'Posts that would be imported:', 'rcp-api-wp-integration' ); ?></h4>
                                    <table class="widefat">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th><?php esc_html_e( 'Title', 'rcp-api-wp-integration' ); ?></th>
                                                <th><?php esc_html_e( 'Published', 'rcp-api-wp-integration' ); ?></th>
                                                <th><?php esc_html_e( 'Categories', 'rcp-api-wp-integration' ); ?></th>
                                                <th><?php esc_html_e( 'Status', 'rcp-api-wp-integration' ); ?></th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ( $test_results['posts'] as $post ) : ?>
                                                <tr>
                                                    <td><?php echo esc_html( $post['id'] ); ?></td>
                                                    <td><?php echo esc_html( $post['title'] ); ?></td>
                                                    <td><?php echo esc_html( $post['published'] ); ?></td>
                                                    <td><?php echo esc_html( implode( ', ', $post['categories'] ) ); ?></td>
                                                    <td><?php echo esc_html( $post['import_status'] ); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                <?php endif; ?>
                                
                                <?php if ( ! empty( $test_results['skip_reasons'] ) ) : ?>
                                    <h4><?php esc_html_e( 'Skip Reasons:', 'rcp-api-wp-integration' ); ?></h4>
                                    <pre style="background: white; padding: 10px; overflow-x: auto;"><?php 
                                        foreach ( $test_results['skip_reasons'] as $reason ) {
                                            echo esc_html( $reason ) . "\n";
                                        }
                                    ?></pre>
                                <?php endif; ?>
                            </div>
                            <?php
                        }
                    }
                    ?>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Debug Log Export', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <form method="post" action="">
                        <?php wp_nonce_field( 'rcp_export_debug_log' ); ?>
                        <input type="hidden" name="action" value="rcp_export_debug_log">
                        <p>
                            <button type="submit" class="button"><?php esc_html_e( 'Download Debug Log', 'rcp-api-wp-integration' ); ?></button>
                            <span class="description"><?php esc_html_e( 'Export recent import logs with full debug information as a text file.', 'rcp-api-wp-integration' ); ?></span>
                        </p>
                    </form>
                    
                    <?php
                    if ( isset( $_POST['action'] ) && $_POST['action'] === 'rcp_export_debug_log' && check_admin_referer( 'rcp_export_debug_log' ) ) {
                        $this->export_debug_log();
                        exit;
                    }
                    ?>
                </div>
            </div>
            
            <div class="postbox">
                <h2 class="hndle"><?php esc_html_e( 'Actions', 'rcp-api-wp-integration' ); ?></h2>
                <div class="inside">
                    <p>
                        <a href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin.php?page=rcp-api-settings&rcp_force_auto_import=1' ), 'rcp_force_auto_import' ) ); ?>" class="button">
                            <?php esc_html_e( 'Force Auto Import Now', 'rcp-api-wp-integration' ); ?>
                        </a>
                        <span class="description"><?php esc_html_e( 'This will reset category cutoffs and run the import immediately.', 'rcp-api-wp-integration' ); ?></span>
                    </p>
                    
                    <form method="post" action="" style="margin-top: 10px;">
                        <?php wp_nonce_field( 'rcp_clear_cache' ); ?>
                        <input type="hidden" name="action" value="rcp_clear_cache">
                        <p>
                            <button type="submit" class="button"><?php esc_html_e( 'Clear All Caches', 'rcp-api-wp-integration' ); ?></button>
                            <span class="description"><?php esc_html_e( 'Clear feed and category caches.', 'rcp-api-wp-integration' ); ?></span>
                        </p>
                    </form>
                    
                    <?php
                    if ( isset( $_POST['action'] ) && $_POST['action'] === 'rcp_clear_cache' && check_admin_referer( 'rcp_clear_cache' ) ) {
                        $this->clear_feed_cache();
                        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Caches cleared successfully.', 'rcp-api-wp-integration' ) . '</p></div>';
                    }
                    ?>
                    
                    <?php
                    // Show clear lock button if import is locked
                    $lock_value = get_transient( 'rcp_import_running' );
                    if ( $lock_value ) :
                    ?>
                    <form method="post" action="" style="margin-top: 10px;">
                        <?php wp_nonce_field( 'rcp_clear_lock' ); ?>
                        <input type="hidden" name="action" value="rcp_clear_lock">
                        <p>
                            <button type="submit" class="button button-secondary"><?php esc_html_e( 'Clear Import Lock', 'rcp-api-wp-integration' ); ?></button>
                            <span class="description" style="color: orange;"><?php esc_html_e( 'An import appears to be stuck. Click to clear the lock.', 'rcp-api-wp-integration' ); ?></span>
                        </p>
                    </form>
                    <?php endif; ?>
                    
                    <?php
                    if ( isset( $_POST['action'] ) && $_POST['action'] === 'rcp_clear_lock' && check_admin_referer( 'rcp_clear_lock' ) ) {
                        delete_transient( 'rcp_import_running' );
                        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Import lock cleared successfully.', 'rcp-api-wp-integration' ) . '</p></div>';
                    }
                    ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * AI Images management page with filters, pagination, and actions.
     */
    public function ai_images_page() {
        if ( ! current_user_can( self::CAPABILITY ) ) {
            return;
        }
        $status_filter = isset( $_GET['status'] ) ? sanitize_text_field( $_GET['status'] ) : '';
        $allowed_status = [ 'queued', 'success', 'failed', 'timeout' ];
        if ( $status_filter && ! in_array( $status_filter, $allowed_status, true ) ) {
            $status_filter = '';
        }
        $per_page = isset( $_GET['per_page'] ) ? max( 5, (int) $_GET['per_page'] ) : 20;
        $paged = isset( $_GET['paged'] ) ? max( 1, (int) $_GET['paged'] ) : 1;
        $log = get_option( 'rcp_ai_image_log', [] );
        if ( ! is_array( $log ) ) { $log = []; }
        $rows = $log;
        if ( $status_filter ) {
            $rows = array_values( array_filter( $rows, function( $row ) use ( $status_filter ) {
                return isset( $row['status'] ) && $row['status'] === $status_filter;
            } ) );
        }
        $total = count( $rows );
        $total_pages = $per_page ? (int) ceil( $total / $per_page ) : 1;
        $offset = ( $paged - 1 ) * $per_page;
        $page_rows = array_slice( array_reverse( $rows ), $offset, $per_page );

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'AI Images', 'rcp-api-wp-integration' ); ?></h1>

            <div class="notice notice-info">
                <p>
                    <?php echo esc_html__( 'Generate AI images for imported posts, review recent jobs, and retry failures here. To control image behavior, use Settings → Featured Image Options (None, Import from RCP Source, Generate AI image, or RCP Source else AI).', 'rcp-api-wp-integration' ); ?>
                </p>
                <p>
                    <?php echo esc_html__( 'Your credit balance appears in Settings under the AI Credits card. “RCP Source else AI” tries your RCP image first, then generates an AI image if needed.', 'rcp-api-wp-integration' ); ?>
                </p>
            </div>

            <form method="get" action="">
                <input type="hidden" name="page" value="rcp-api-ai-images" />
                <label for="rcp-ai-status"><?php esc_html_e( 'Filter by status:', 'rcp-api-wp-integration' ); ?></label>
                <select id="rcp-ai-status" name="status">
                    <option value="" <?php selected( $status_filter, '' ); ?>><?php esc_html_e( 'All', 'rcp-api-wp-integration' ); ?></option>
                    <?php foreach ( $allowed_status as $st ) : ?>
                        <option value="<?php echo esc_attr( $st ); ?>" <?php selected( $status_filter, $st ); ?>><?php echo esc_html( ucfirst( $st ) ); ?></option>
                    <?php endforeach; ?>
                </select>
                <label for="rcp-ai-per-page" style="margin-left: 10px;"><?php esc_html_e( 'Per page:', 'rcp-api-wp-integration' ); ?></label>
                <input type="number" id="rcp-ai-per-page" name="per_page" value="<?php echo esc_attr( $per_page ); ?>" min="5" max="100" />
                <button class="button"><?php esc_html_e( 'Apply', 'rcp-api-wp-integration' ); ?></button>
            </form>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:10px;">
                <?php wp_nonce_field( 'rcp_ai_images_clear_log' ); ?>
                <input type="hidden" name="action" value="rcp_ai_images_clear_log" />
                <button class="button button-secondary" onclick="return confirm('<?php echo esc_js( __( 'Clear AI images log?', 'rcp-api-wp-integration' ) ); ?>');"><?php esc_html_e( 'Clear Log', 'rcp-api-wp-integration' ); ?></button>
            </form>

            <table class="wp-list-table widefat fixed striped" style="margin-top:15px;">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Time (UTC)', 'rcp-api-wp-integration' ); ?></th>
                        <th><?php esc_html_e( 'Post', 'rcp-api-wp-integration' ); ?></th>
                        <th><?php esc_html_e( 'Status', 'rcp-api-wp-integration' ); ?></th>
                        <th><?php esc_html_e( 'Preview', 'rcp-api-wp-integration' ); ?></th>
                        <th><?php esc_html_e( 'Details', 'rcp-api-wp-integration' ); ?></th>
                        <th><?php esc_html_e( 'Actions', 'rcp-api-wp-integration' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php if ( empty( $page_rows ) ) : ?>
                    <tr><td colspan="6"><?php esc_html_e( 'No entries found.', 'rcp-api-wp-integration' ); ?></td></tr>
                <?php else :
                    foreach ( $page_rows as $row ) :
                        $time = isset( $row['time'] ) ? (int) $row['time'] : time();
                        $pid  = isset( $row['post_id'] ) ? (int) $row['post_id'] : 0;
                        $st   = isset( $row['status'] ) ? (string) $row['status'] : '-';
                        $gen  = isset( $row['generation_log_id'] ) ? (string) $row['generation_log_id'] : '';
                        $att  = isset( $row['attachment_id'] ) ? (int) $row['attachment_id'] : 0;
                        $url  = isset( $row['result_url'] ) ? (string) $row['result_url'] : '';
                        $msg  = isset( $row['message'] ) ? (string) $row['message'] : '';
                        ?>
                        <tr>
                            <td><?php echo esc_html( gmdate( 'Y-m-d H:i:s', $time ) ); ?></td>
                            <td>
                                <?php if ( $pid ) : $edit = get_edit_post_link( $pid ); ?>
                                    <?php if ( $edit ) : ?><a href="<?php echo esc_url( $edit ); ?>" target="_blank"><?php echo esc_html( get_the_title( $pid ) ?: ( 'Post #' . $pid ) ); ?></a><?php else : echo esc_html( 'Post #' . $pid ); endif; ?>
                                <?php else : ?>
                                    &ndash;
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html( ucfirst( $st ) ); ?></td>
                            <td>
                                <?php
                                $preview_html = '&ndash;';
                                $full_url     = '';

                                if ( $att ) {
                                    $thumb = wp_get_attachment_image( $att, 'thumbnail', false, [ 'style' => 'max-width:90px;height:auto;border:1px solid #ccd0d4;border-radius:4px;' ] );
                                    $full  = wp_get_attachment_url( $att );
                                    if ( $thumb ) {
                                        $full_url = $full ? $full : $url;
                                        $image    = $thumb;
                                        if ( $full_url ) {
                                            $image = '<a href="' . esc_url( $full_url ) . '" target="_blank" rel="noopener">' . $image . '</a>';
                                        }
                                        $preview_html = '<div class="rcp-ai-preview">' . $image . '</div>';
                                    }
                                } elseif ( $url ) {
                                    $full_url = $url;
                                    $image    = '<img src="' . esc_url( $url ) . '" alt="" style="max-width:90px;height:auto;border:1px solid #ccd0d4;border-radius:4px;" />';
                                    $preview_html = '<div class="rcp-ai-preview"><a href="' . esc_url( $url ) . '" target="_blank" rel="noopener">' . $image . '</a></div>';
                                }

                                if ( $full_url && '&ndash;' !== $preview_html ) {
                                    $view_label     = __( 'View full size', 'rcp-api-wp-integration' );
                                    $download_label = __( 'Download', 'rcp-api-wp-integration' );
                                    $download_name  = basename( (string) wp_parse_url( $full_url, PHP_URL_PATH ) );
                                    $download_attr  = $download_name ? ' download="' . esc_attr( $download_name ) . '"' : ' download';
                                    $preview_html  .= '<div class="rcp-ai-preview-links"><a href="' . esc_url( $full_url ) . '" target="_blank" rel="noopener">' . esc_html( $view_label ) . '</a> | <a href="' . esc_url( $full_url ) . '"' . $download_attr . '>' . esc_html( $download_label ) . '</a></div>';
                                }

                                echo wp_kses_post( $preview_html );
                                ?>
                            </td>
                            <td>
                                <?php
                                $parts = [];
                                if ( $gen ) { $parts[] = 'log_id=' . esc_html( $gen ); }
                                if ( $att ) { $parts[] = 'attachment_id=' . (int) $att; }
                                if ( $url ) { $parts[] = '<a href="' . esc_url( $url ) . '" target="_blank">URL</a>'; }
                                if ( $msg ) { $parts[] = esc_html( $msg ); }
                                echo implode( ' | ', $parts );
                                ?>
                            </td>
                            <td>
                                <?php if ( in_array( $st, [ 'failed', 'timeout' ], true ) && $pid ) : ?>
                                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                                        <?php wp_nonce_field( 'rcp_ai_images_retry_' . $pid ); ?>
                                        <input type="hidden" name="action" value="rcp_ai_images_retry" />
                                        <input type="hidden" name="post_id" value="<?php echo esc_attr( $pid ); ?>" />
                                        <button class="button button-small"><?php esc_html_e( 'Retry', 'rcp-api-wp-integration' ); ?></button>
                                    </form>
                                <?php else : ?>
                                    &ndash;
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; endif; ?>
                </tbody>
            </table>

            <?php if ( $total_pages > 1 ) : ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <span class="displaying-num"><?php echo esc_html( sprintf( _n( '%d item', '%d items', $total, 'rcp-api-wp-integration' ), $total ) ); ?></span>
                        <?php
                        $base_url = remove_query_arg( [ 'paged' ] );
                        $prev = max( 1, $paged - 1 );
                        $next = min( $total_pages, $paged + 1 );
                        ?>
                        <a class="prev-page button" href="<?php echo esc_url( add_query_arg( 'paged', $prev, $base_url ) ); ?>">&laquo;</a>
                        <span class="paging-input"><?php echo esc_html( $paged . ' / ' . $total_pages ); ?></span>
                        <a class="next-page button" href="<?php echo esc_url( add_query_arg( 'paged', $next, $base_url ) ); ?>">&raquo;</a>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    public function handle_ai_images_clear_log() {
        check_admin_referer( 'rcp_ai_images_clear_log' );
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_die( __( 'Permission denied', 'rcp-api-wp-integration' ) );
        }
        delete_option( 'rcp_ai_image_log' );
        wp_redirect( admin_url( 'admin.php?page=rcp-api-ai-images' ) );
        exit;
    }

    public function handle_ai_images_retry() {
        $post_id = isset( $_POST['post_id'] ) ? (int) $_POST['post_id'] : 0;
        if ( ! $post_id ) {
            wp_redirect( admin_url( 'admin.php?page=rcp-api-ai-images' ) );
            exit;
        }
        check_admin_referer( 'rcp_ai_images_retry_' . $post_id );
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_die( __( 'Permission denied', 'rcp-api-wp-integration' ) );
        }
        $rcp_post_id = (int) get_post_meta( $post_id, '_rcp_original_post_id', true );
        if ( ! has_post_thumbnail( $post_id ) ) {
            $pending_key = 'rcp_ai_gen_pending_' . $post_id;
            if ( ! get_transient( $pending_key ) ) {
                set_transient( $pending_key, 1, 10 * MINUTE_IN_SECONDS );
                RCP_API_Job_Queue::schedule_async( 'ai_generate', [
                    'post_id'     => $post_id,
                    'rcp_post_id' => $rcp_post_id,
                ] );
            }
        }
        wp_redirect( admin_url( 'admin.php?page=rcp-api-ai-images' ) );
        exit;
    }

    private function check_connection() {
        $resp = $this->get_feed( [ 'per_page' => 1 ] );
        if ( is_wp_error( $resp ) ) {
            return $resp->get_error_message();
        }
        return true;
    }

    private function get_feed( $params = [], $bypass_cache = false ) {
        $cache_key    = 'rcp_feed_' . md5( maybe_serialize( $params ) );
        $transient_key = self::TRANSIENT_FEED . md5( maybe_serialize( $params ) );
        
        // Check if we should bypass cache (useful for automation)
        if ( ! $bypass_cache ) {
            // Try object cache first (Redis, Memcached, etc.)
            $cached = wp_cache_get( $cache_key, 'rcp_api' );
            if ( $cached !== false ) {
                return $cached;
            }
            
            // Fall back to transient cache
            $cached = get_transient( $transient_key );
            if ( $cached ) {
                // Store in object cache for next time
                wp_cache_set( $cache_key, $cached, 'rcp_api', 5 * MINUTE_IN_SECONDS );
                return $cached;
            }
        }

        $username = get_option( self::OPTION_USERNAME );
        $password = get_option( self::OPTION_PASSWORD );

        if ( empty( $username ) || empty( $password ) ) {
            return new WP_Error( 'missing_creds', __( 'Missing credentials', 'rcp-api-wp-integration' ) );
        }

        $timeout = apply_filters( 'rcp_api_timeout', self::API_TIMEOUT );
        $args = [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode( $username . ':' . $password ),
            ],
            'timeout' => $timeout,
        ];

        $url  = add_query_arg( $params, trailingslashit( $this->api_base ) . "feed" );
        
        // Log API request
        $this->debug_log( 'Request: GET ' . $url );
        $this->debug_log( 'Params: ', $params );
        
        $start_time = microtime( true );
        $resp = wp_safe_remote_get( $url, array_merge( [
            'redirection' => 3,
            'reject_unsafe_urls' => true,
        ], $args ) );
        $request_time = microtime( true ) - $start_time;
        
        if ( is_wp_error( $resp ) ) {
            $this->debug_log( 'Error: ' . $resp->get_error_message() );
            $this->debug_log( 'Request took ' . number_format( $request_time, 3 ) . ' seconds' );
            return $resp;
        }

        $code = wp_remote_retrieve_response_code( $resp );
        $body = wp_remote_retrieve_body( $resp );
        
        $this->debug_log( 'Response Code: ' . $code );
        $this->debug_log( 'Response Headers: ', wp_remote_retrieve_headers( $resp )->getAll() );
        $this->debug_log( 'Response Body Length: ' . strlen( $body ) . ' bytes' );
        $this->debug_log( 'Request took ' . number_format( $request_time, 3 ) . ' seconds' );
        
        if ( 200 !== $code ) {
            $error_message = sprintf( __( 'API Error (HTTP %d)', 'rcp-api-wp-integration' ), $code );
            if ( ! empty( $body ) ) {
                $error_data = json_decode( $body, true );
                if ( isset( $error_data['message'] ) ) {
                    $error_message .= ': ' . $error_data['message'];
                }
            }
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[RCP API] Full error response: ' . substr( $body, 0, 1000 ) );
            }
            
            return new WP_Error( 'api_error', $error_message );
        }

        $data        = json_decode( $body, true );
        $total_pages = (int) wp_remote_retrieve_header( $resp, 'X-WP-TotalPages' );
        $total_posts = (int) wp_remote_retrieve_header( $resp, 'X-WP-Total' );

        $result = [
            'posts'       => $data,
            'total_pages' => $total_pages ?: 1,
            'total'       => $total_posts,
        ];

        // Store in both caches unless explicitly bypassing cache (TTL filterable)
        if ( ! $bypass_cache ) {
            $ttl = (int) apply_filters( 'rcp_api_feed_ttl', 5 * MINUTE_IN_SECONDS );
            wp_cache_set( $cache_key, $result, 'rcp_api', $ttl );
            set_transient( $transient_key, $result, $ttl );
        }
        return $result;
    }

    /**
     * Retrieve all available categories from the API.
     *
     * @return array|WP_Error
     */
    private function get_api_categories() {
        // Try object cache first
        $cached_categories = wp_cache_get( 'rcp_api_categories', 'rcp_api' );
        if ( $cached_categories !== false ) {
            return $cached_categories;
        }
        
        // Fall back to transient cache
        $cached_categories = get_transient( 'rcp_api_categories' );
        if ( $cached_categories !== false ) {
            wp_cache_set( 'rcp_api_categories', $cached_categories, 'rcp_api', 15 * MINUTE_IN_SECONDS );
            return $cached_categories;
        }

        $categories = [];
        $max_pages = (int) apply_filters( 'rcp_api_categories_max_pages', 5 );
        
        // Use larger batch size for better performance
        $posts_per_page = (int) apply_filters( 'rcp_api_posts_per_page', self::POSTS_PER_PAGE );
        $batch_size = min( $posts_per_page * 2, 100 );
        
        $feed = $this->get_feed( [ 'per_page' => $batch_size, 'page' => 1 ] );
        if ( is_wp_error( $feed ) ) {
            return $feed;
        }

        $total_pages = min( (int) $feed['total_pages'], $max_pages );
        
        // Process first page
        $this->extract_categories_from_posts( $feed['posts'], $categories );
        
        // If we have more pages, process them
        if ( $total_pages > 1 ) {
            $remaining_pages = range( 2, $total_pages );
            
            foreach ( array_chunk( $remaining_pages, 3 ) as $page_batch ) {
                $this->process_category_batch( $page_batch, $batch_size, $categories );
            }
        }

        $result = array_values( $categories );
        
        // Cache categories for a configurable TTL (default 15 minutes)
        $ttl = (int) apply_filters( 'rcp_api_categories_ttl', 15 * MINUTE_IN_SECONDS );
        wp_cache_set( 'rcp_api_categories', $result, 'rcp_api', $ttl );
        set_transient( 'rcp_api_categories', $result, $ttl );
        
        return $result;
    }

    /**
     * Extract categories from posts array.
     *
     * @param array $posts Array of posts.
     * @param array &$categories Categories array to populate.
     */
    private function extract_categories_from_posts( $posts, &$categories ) {
        foreach ( (array) $posts as $post ) {
            if ( empty( $post['categories'] ) ) {
                continue;
            }
            foreach ( (array) $post['categories'] as $cat ) {
                if ( ! empty( $cat['name'] ) ) {
                    $categories[ $cat['name'] ] = $cat['name'];
                }
            }
        }
    }

    /**
     * Process a batch of pages for category extraction.
     *
     * @param array $pages Array of page numbers.
     * @param int   $batch_size Posts per page.
     * @param array &$categories Categories array to populate.
     */
    private function process_category_batch( $pages, $batch_size, &$categories ) {
        foreach ( $pages as $page ) {
            $feed = $this->get_feed( [ 'per_page' => $batch_size, 'page' => $page ] );
            if ( is_wp_error( $feed ) ) {
                continue; // Skip failed pages
            }
            
            $this->extract_categories_from_posts( $feed['posts'], $categories );
        }
    }

    /**
     * Check rate limiting for AJAX requests.
     *
     * @param string $action The AJAX action name.
     * @return bool|WP_Error True if allowed, WP_Error if rate limited.
     */
    private function check_rate_limit( $action ) {
        $user_id = get_current_user_id();
        $key = "rcp_rate_limit_{$action}_{$user_id}";
        $requests = get_transient( $key );
        
        if ( false === $requests ) {
            $requests = 0;
        }
        
        if ( $requests >= self::RATE_LIMIT_REQUESTS ) {
            return new WP_Error( 'rate_limit_exceeded', __( 'Too many requests. Please wait a minute before trying again.', 'rcp-api-wp-integration' ) );
        }
        
        set_transient( $key, $requests + 1, self::RATE_LIMIT_WINDOW );
        return true;
    }

    public function ajax_fetch_feed() {
        try {
            // Start performance monitoring
            RCP_API_Performance_Monitor::start( 'ajax_fetch_feed' );
            
            // Validate AJAX request
            RCP_API_Validator::validate_ajax_request( 'fetch_feed', self::CAPABILITY );
            
            // Check rate limiting with enhanced limiter
            $rate_check = RCP_API_Rate_Limiter::check( 'fetch_feed', null, 'ajax' );
            if ( is_wp_error( $rate_check ) ) {
                RCP_API_Rate_Limiter::add_headers( 'fetch_feed', null, 'ajax' );
                wp_send_json_error( $rate_check->get_error_message(), 429 );
            }
            
            // Validate input parameters
            $page = isset( $_GET['page'] ) ? RCP_API_Validator::validate( 'page', $_GET['page'] ) : 1;

            // Always limit posts to a recent window (filterable, default 36h)
            $max_age = (int) apply_filters( 'rcp_api_max_import_age', 36 * HOUR_IN_SECONDS );
            $timestamp = current_time( 'timestamp', true ) - $max_age;
            $after     = gmdate( 'Y-m-d H:i:s', $timestamp );

            $per_page = (int) apply_filters( 'rcp_api_posts_per_page', self::POSTS_PER_PAGE );
            $params   = [ 'per_page' => $per_page, 'page' => $page ];
            if ( $after ) {
                $params['after'] = $after;
            }

            $feed = $this->get_feed( $params );
            if ( is_wp_error( $feed ) ) {
                throw new RCP_API_Validation_Exception( $feed->get_error_message() );
            }

            $posts       = isset( $feed['posts'] ) ? $feed['posts'] : (array) $feed;
            $total_pages = isset( $feed['total_pages'] ) ? (int) $feed['total_pages'] : 1;
            $total_posts = isset( $feed['total'] ) ? (int) $feed['total'] : 0;

            $limit_ts = current_time( 'timestamp', true ) - 36 * HOUR_IN_SECONDS;
            $posts = array_filter( (array) $posts, function( $p ) use ( $limit_ts ) {
                $published = $p['published'] ?? ( $p['date'] ?? '' );
                if ( ! $published ) {
                    return false;
                }
                return strtotime( $published ) >= $limit_ts;
            } );
            $posts = array_values( $posts );

            $requested_categories = [];
            if ( isset( $_GET['categories'] ) ) {
                $raw_input = $_GET['categories'];
                if ( is_string( $raw_input ) ) {
                    $raw_input = [ $raw_input ];
                } elseif ( ! is_array( $raw_input ) ) {
                    $raw_input = (array) $raw_input;
                }

                foreach ( $raw_input as $cat_name ) {
                    $cat_name = sanitize_text_field( wp_unslash( $cat_name ) );
                    if ( '' === $cat_name ) {
                        continue;
                    }
                    $normalized = $this->normalize_category_name( $cat_name );
                    if ( $normalized ) {
                        $requested_categories[] = $normalized;
                    }
                }
                $requested_categories = array_unique( $requested_categories );
            }

            $all_categories = [];
            $posts_filtered = 0;

            if ( empty( $requested_categories ) ) {
                foreach ( (array) $posts as $p ) {
                    if ( empty( $p['categories'] ) ) {
                        continue;
                    }
                    foreach ( (array) $p['categories'] as $cat ) {
                        if ( ! empty( $cat['name'] ) ) {
                            $all_categories[ $cat['name'] ] = $cat['name'];
                        }
                    }
                }

                // Filter out posts with empty blog_post_html
                $posts_before_filter = count( $posts );
                $posts = array_filter( $posts, function( $p ) {
                    $blog_title = trim( $p['blog_post_title'] ?? '' );
                    $blog_html  = trim( $p['blog_post_html'] ?? '' );
                    return '' !== $blog_title && '' !== $blog_html;
                } );
                $posts = array_values( $posts );
                $posts_filtered = $posts_before_filter - count( $posts );

                if ( $posts_filtered > 0 ) {
                    $this->debug_log( sprintf(
                        'Filtered out %d posts with missing title or content from feed display',
                        $posts_filtered
                    ) );
                }
            } else {
                $start_index  = max( 0, ( $page - 1 ) * $per_page );
                $target_count = $page * $per_page;
                $buffered_posts = [];
                $total_filtered = 0;
                $current_page = 1;
                $total_pages_api = max( 1, $total_pages );

                $process_posts = function( array $page_posts ) use ( &$all_categories, $limit_ts, $requested_categories ) {
                    $processed = [];
                    foreach ( $page_posts as $post ) {
                        $published = $post['published'] ?? ( $post['date'] ?? '' );
                        if ( ! $published || strtotime( $published ) < $limit_ts ) {
                            continue;
                        }

                        if ( ! empty( $post['categories'] ) ) {
                            foreach ( (array) $post['categories'] as $cat ) {
                                if ( ! empty( $cat['name'] ) ) {
                                    $all_categories[ $cat['name'] ] = $cat['name'];
                                }
                            }
                        }

                        $blog_title = trim( $post['blog_post_title'] ?? '' );
                        $blog_html  = trim( $post['blog_post_html'] ?? '' );
                        if ( '' === $blog_title || '' === $blog_html ) {
                            continue;
                        }

                        $matches_category = false;
                        if ( ! empty( $post['categories'] ) ) {
                            foreach ( (array) $post['categories'] as $cat ) {
                                $normalized = $this->normalize_category_name( $cat['name'] ?? '' );
                                if ( $normalized && in_array( $normalized, $requested_categories, true ) ) {
                                    $matches_category = true;
                                    break;
                                }
                            }
                        }

                        if ( $matches_category ) {
                            $processed[] = $post;
                        }
                    }
                    return $processed;
                };

                $initial_processed = $process_posts( (array) $posts );
                foreach ( $initial_processed as $post ) {
                    $total_filtered++;
                    if ( $total_filtered <= $target_count ) {
                        $buffered_posts[] = $post;
                    }
                }

                while ( $current_page < $total_pages_api ) {
                    $current_page++;
                    $params['page'] = $current_page;
                    $feed_page = $this->get_feed( $params );
                    if ( is_wp_error( $feed_page ) ) {
                        break;
                    }
                    $total_pages_api = max( $total_pages_api, (int) ( $feed_page['total_pages'] ?? $total_pages_api ) );
                    $page_posts = isset( $feed_page['posts'] ) ? $feed_page['posts'] : (array) $feed_page;
                    $processed = $process_posts( (array) $page_posts );
                    foreach ( $processed as $post ) {
                        $total_filtered++;
                        if ( $total_filtered <= $target_count ) {
                            $buffered_posts[] = $post;
                        }
                    }
                }

                $total_posts = $total_filtered;
                $total_pages = max( 1, (int) ceil( $total_filtered / $per_page ) );
                $posts = array_slice( $buffered_posts, $start_index, $per_page );
                $posts_filtered = 0;
            }

            $categories = [];

            // Precompute which posts are already imported using a single query
            $rcp_ids = [];
            foreach ( (array) $posts as $p ) {
                if ( isset( $p['id'] ) ) {
                    $rcp_ids[] = (int) $p['id'];
                }
                if ( ! empty( $p['categories'] ) ) {
                    foreach ( (array) $p['categories'] as $cat ) {
                        if ( ! empty( $cat['name'] ) ) {
                            $categories[ $cat['name'] ] = $cat['name'];
                        }
                    }
                }
            }

            $imported_lookup = [];
            if ( ! empty( $rcp_ids ) ) {
                global $wpdb;
                $placeholders = implode( ',', array_fill( 0, count( $rcp_ids ), '%d' ) );
                $sql = $wpdb->prepare(
                    "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value IN ($placeholders)",
                    array_merge( [ '_rcp_original_post_id' ], $rcp_ids )
                );
                $found = $wpdb->get_col( $sql );
                foreach ( (array) $found as $val ) {
                    $imported_lookup[ (int) $val ] = true;
                }
            }

            foreach ( (array) $posts as &$post ) {
                $pid = isset( $post['id'] ) ? (int) $post['id'] : 0;
                $post['imported'] = $pid && isset( $imported_lookup[ $pid ] );
            }
            
            // Record performance metrics
            $metrics = RCP_API_Performance_Monitor::end( 'ajax_fetch_feed', [
                'page' => $page,
                'posts_count' => count( $posts ),
                'filtered_count' => $posts_filtered,
            ] );

            wp_send_json_success(
                [
                    'posts'       => $posts,
                    'categories'  => array_values( $all_categories ),
                    'total_pages' => $total_pages,
                    'total'       => $total_posts,
                ]
            );
            
        } catch ( RCP_API_Validation_Exception $e ) {
            wp_send_json_error( $e->getMessage(), $e->get_http_status_code() );
        } catch ( Exception $e ) {
            wp_send_json_error( __( 'An unexpected error occurred', 'rcp-api-wp-integration' ), 500 );
        }
    }


    public function ajax_import_post() {
        try {
            // Start performance monitoring
            RCP_API_Performance_Monitor::start( 'ajax_import_post' );
            
            // Validate AJAX request
            RCP_API_Validator::validate_ajax_request( 'import_post', self::CAPABILITY );
            
            // Check rate limiting with enhanced limiter
            $rate_check = RCP_API_Rate_Limiter::check( 'import_post', null, 'import' );
            if ( is_wp_error( $rate_check ) ) {
                RCP_API_Rate_Limiter::add_headers( 'import_post', null, 'import' );
                wp_send_json_error( $rate_check->get_error_message(), 429 );
            }
            
            // Validate input parameters
            $post_id = RCP_API_Validator::validate( 'post_id', $_POST['post_id'] ?? 0 );
            $status = RCP_API_Validator::validate( 'status', $_POST['status'] ?? '' );
            $post_date = isset( $_POST['post_date'] ) ? RCP_API_Validator::validate( 'post_date', $_POST['post_date'] ) : '';
            
            // Additional validation for bulk import parameters
            $is_bulk = false;
            if ( isset( $_POST['bulk_import'] ) ) {
                $is_bulk = RCP_API_Validator::validate( 'bulk_import', $_POST['bulk_import'] );
            }
            
            if ( $is_bulk ) {
                // Queue background import for bulk operations
                RCP_API_Job_Queue::schedule_async( 'import_single_post', [
                    'post_id' => $post_id,
                    'status'  => $status,
                    'author_id' => 0,
                    'import_images' => false, // import_post schedules images itself
                ] );

                RCP_API_Performance_Monitor::end( 'ajax_import_post', [
                    'post_id' => $post_id,
                    'status' => $status,
                    'import_type' => 'bulk_queued',
                ] );

                wp_send_json_success( [ 'queued' => true ] );
            } else {
                $result = $this->import_post( $post_id, $status, $post_date, 0 );
                if ( is_wp_error( $result ) ) {
                    // Log the error
                    $this->record_import_log( [
                        'time'      => current_time( 'timestamp', true ),
                        'published' => [],
                        'drafted'   => [],
                        'errors'    => [ 'Manual import failed for post ' . $post_id . ': ' . $result->get_error_message() ],
                        'success'   => false,
                        'debug'     => [
                            'import_type' => 'ajax',
                            'original_post_id' => $post_id,
                            'error_code' => $result->get_error_code(),
                            'posts_processed' => 1,
                            'posts_skipped' => 1,
                            'posts_imported' => 0,
                        ]
                    ] );
                    throw new RCP_API_Validation_Exception( $result->get_error_message() );
                }
                
                // Log successful import
                $import_type = 'manual_single';
                $this->record_manual_import( $post_id, $status, $result, $import_type );
                
                // Record performance metrics
                $metrics = RCP_API_Performance_Monitor::end( 'ajax_import_post', [
                    'post_id' => $post_id,
                    'status' => $status,
                    'import_type' => $import_type,
                ] );

                wp_send_json_success( $result );
            }
            
        } catch ( RCP_API_Validation_Exception $e ) {
            wp_send_json_error( $e->getMessage(), $e->get_http_status_code() );
        } catch ( Exception $e ) {
            wp_send_json_error( __( 'An unexpected error occurred', 'rcp-api-wp-integration' ), 500 );
        }
    }

    public function ajax_fetch_categories() {
        check_ajax_referer( "rcp_feed_nonce", "nonce" );

        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_send_json_error( __( "Permission denied", "rcp-api-wp-integration" ), 403 );
        }

        $rate_check = RCP_API_Rate_Limiter::check( 'fetch_categories', null, 'ajax' );
        if ( is_wp_error( $rate_check ) ) {
            RCP_API_Rate_Limiter::add_headers( 'fetch_categories', null, 'ajax' );
            wp_send_json_error( $rate_check->get_error_message(), 429 );
        }

        $categories = $this->get_api_categories();
        if ( is_wp_error( $categories ) ) {
            wp_send_json_error( $categories->get_error_message() );
        }

        wp_send_json_success( [
            'categories' => $categories
        ] );
    }

    /**
     * AJAX: Queue AI image generation for a (previously imported) post.
     */
    public function ajax_generate_ai_image() {
        check_ajax_referer( 'rcp_feed_nonce', 'nonce' );
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_send_json_error( __( 'Permission denied', 'rcp-api-wp-integration' ), 403 );
        }

        try {
            $wp_post_id  = isset( $_POST['wp_post_id'] ) ? (int) $_POST['wp_post_id'] : 0;
            $rcp_post_id = isset( $_POST['rcp_post_id'] ) ? (int) $_POST['rcp_post_id'] : 0;

            if ( ! $wp_post_id && $rcp_post_id ) {
                $existing = get_posts( [
                    'meta_key'       => '_rcp_original_post_id',
                    'meta_value'     => $rcp_post_id,
                    'post_type'      => 'post',
                    'post_status'    => 'any',
                    'posts_per_page' => 1,
                    'fields'         => 'ids',
                ] );
                $wp_post_id = $existing ? (int) $existing[0] : 0;
            }

            if ( ! $wp_post_id ) {
                wp_send_json_error( __( 'Post not found for AI generation.', 'rcp-api-wp-integration' ), 400 );
            }

            $pending_key = 'rcp_ai_gen_pending_' . $wp_post_id;

            if ( get_transient( $pending_key ) ) {
                delete_transient( $pending_key );
                $this->record_ai_image_log( [
                    'time' => current_time( 'timestamp', true ),
                    'post_id' => $wp_post_id,
                    'rcp_post_id' => $rcp_post_id,
                    'status' => 'manual_retry',
                    'message' => 'Manual generation override cleared stale pending flag.',
                ] );
            }

            set_transient( $pending_key, 1, 10 * MINUTE_IN_SECONDS );

            $result = $this->generate_ai_image_for_post( $wp_post_id, $rcp_post_id );

            if ( is_wp_error( $result ) ) {
                delete_transient( $pending_key );
                $this->record_ai_image_log( [
                    'time' => current_time( 'timestamp', true ),
                    'post_id' => $wp_post_id,
                    'rcp_post_id' => $rcp_post_id,
                    'status' => 'error',
                    'message' => $result->get_error_message(),
                ] );
                wp_send_json_error( $result->get_error_message() );
            }

            if ( isset( $result['skipped'] ) && $result['skipped'] ) {
                delete_transient( $pending_key );
                wp_send_json_success( [ 'skipped' => true ] );
            }

            if ( isset( $result['attachment_id'] ) ) {
                delete_transient( $pending_key );
                wp_send_json_success( [
                    'attached'       => true,
                    'attachment_id'  => (int) $result['attachment_id'],
                    'thumbnail_html' => get_the_post_thumbnail( $wp_post_id, 'thumbnail' ),
                ] );
            }

            // Async flow already scheduled polling and will clear the transient when complete.
            $response = [
                'queued'            => ! empty( $result['queued'] ),
                'generation_log_id' => $result['generation_log_id'] ?? '',
            ];

            if ( ! empty( $result['reservation_id'] ) ) {
                $response['reservation_id'] = $result['reservation_id'];
            }

            $this->record_ai_image_log( [
                'time' => current_time( 'timestamp', true ),
                'post_id' => $wp_post_id,
                'rcp_post_id' => $rcp_post_id,
                'status' => 'queued_manual',
                'generation_log_id' => $result['generation_log_id'] ?? '',
                'reservation_id' => $result['reservation_id'] ?? '',
            ] );

            wp_send_json_success( $response );

        } catch ( Exception $e ) {
            if ( isset( $pending_key ) ) {
                delete_transient( $pending_key );
            }
            wp_send_json_error( __( 'An unexpected error occurred', 'rcp-api-wp-integration' ), 500 );
        }
    }

    public function handle_disconnect() {
        check_admin_referer( 'rcp_api_disconnect' );
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_die( __( 'Permission denied', 'rcp-api-wp-integration' ) );
        }
        delete_option( self::OPTION_USERNAME );
        delete_option( self::OPTION_PASSWORD );
        delete_option( 'rcp_feed_posts_per_page' );
        $this->clear_feed_cache();
        wp_redirect( admin_url( 'admin.php?page=rcp-api-info' ) );
        exit;
    }

    /**
     * Delete any cached feed data.
     */
    private function clear_feed_cache() {
        global $wpdb;
        
        // Clear object cache group (only if supported by the active object cache)
        if ( function_exists( 'wp_cache_flush_group' ) ) {
            wp_cache_flush_group( 'rcp_api' );
        }
        
        // Clear transient cache
        $like = $wpdb->esc_like( self::TRANSIENT_FEED );
        $transient_pattern = '_transient_' . $like . '%';
        $timeout_pattern = '_transient_timeout_' . $like . '%';
        
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                $transient_pattern,
                $timeout_pattern
            )
        );
        
        // Clear specific cached items
        delete_transient( 'rcp_api_categories' );
        wp_cache_delete( 'rcp_api_categories', 'rcp_api' );
    }

    /**
     * Clear cached feed data after relevant settings are updated.
     *
     * @param string $option    Updated option name.
     * @param mixed  $old_value Previous value.
     * @param mixed  $value     New value.
     */
    public function maybe_clear_cache( $option, $old_value = null, $value = null ) {
        $watched = [
            self::OPTION_USERNAME,
            self::OPTION_PASSWORD,
            self::OPTION_CATEGORY_MAPPING,
            self::OPTION_IMPORT_FEATURED,
            self::OPTION_ENABLE_AUTO_PUBLISH,
            self::OPTION_ENABLE_AUTO_DRAFT,
            self::OPTION_AUTO_PUBLISH_CATS,
            self::OPTION_AUTO_DRAFT_CATS,
            self::OPTION_CATEGORY_AUTOMATION,
            self::OPTION_CATEGORY_AUTHORS,
            self::OPTION_FEATURED_IMAGE_CATS,
            self::OPTION_CRON_FREQUENCY,
        ];

        if ( in_array( $option, $watched, true ) ) {
            if ( self::OPTION_CRON_FREQUENCY === $option ) {
                $this->maybe_schedule_auto_import();
            }
            $reset = [
                self::OPTION_ENABLE_AUTO_PUBLISH,
                self::OPTION_ENABLE_AUTO_DRAFT,
                self::OPTION_AUTO_PUBLISH_CATS,
                self::OPTION_AUTO_DRAFT_CATS,
                self::OPTION_CATEGORY_AUTOMATION,
            ];
            if ( in_array( $option, $reset, true ) ) {
                if ( self::OPTION_CATEGORY_AUTOMATION === $option ) {
                    $this->update_category_cutoffs( (array) $old_value, (array) $value );
                }
                // Only update last import time for non-cutoff related changes
                // The cutoff system handles timing more precisely
                if ( self::OPTION_CATEGORY_AUTOMATION !== $option ) {
                    update_option( self::OPTION_LAST_IMPORT, current_time( 'timestamp', true ) );
                }
            }
            // Clear cache after all updates are complete
            $this->clear_feed_cache();
        }
    }

    private function update_category_cutoffs( array $old, array $new ) {
        // Normalize all category names for consistency
        $normalized_old = [];
        $normalized_new = [];
        $cutoffs = get_option( self::OPTION_CATEGORY_CUTOFFS, [] );
        if ( ! is_array( $cutoffs ) ) { $cutoffs = []; }
        
        // Normalize old categories
        foreach ( $old as $cat => $mode ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $normalized_old[ $normalized ] = $mode;
            }
        }
        
        // Normalize new categories
        foreach ( $new as $cat => $mode ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $normalized_new[ $normalized ] = $mode;
            }
        }
        
        // Normalize existing cutoffs
        $normalized_cutoffs = [];
        foreach ( $cutoffs as $cat => $time ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $normalized_cutoffs[ $normalized ] = $time;
            }
        }
        
        $now = current_time( 'timestamp', true );

        // Only set cutoffs for newly enabled categories or categories changing from 'none' to active
        foreach ( $normalized_new as $cat => $mode ) {
            $old_mode = isset( $normalized_old[ $cat ] ) ? $normalized_old[ $cat ] : 'none';
            $is_new_automation = in_array( $mode, [ 'publish', 'draft' ], true ) && 
                                ( ! isset( $normalized_old[ $cat ] ) || $old_mode === 'none' || ! in_array( $old_mode, [ 'publish', 'draft' ], true ) );
            
            if ( $is_new_automation ) {
                // For newly enabled automation, set cutoff to allow posts from the last 36 hours
                // This prevents importing very old posts while allowing recent ones
                $normalized_cutoffs[ $cat ] = $now - ( 36 * HOUR_IN_SECONDS );
            } elseif ( ! in_array( $mode, [ 'publish', 'draft' ], true ) && isset( $normalized_cutoffs[ $cat ] ) ) {
                // Remove cutoff when automation is disabled
                unset( $normalized_cutoffs[ $cat ] );
            }
            // For existing automation settings, preserve the current cutoff
        }

        // Clean up cutoffs for removed categories
        foreach ( $normalized_old as $cat => $mode ) {
            if ( ! isset( $normalized_new[ $cat ] ) ) {
                unset( $normalized_cutoffs[ $cat ] );
            }
        }

        update_option( self::OPTION_CATEGORY_CUTOFFS, $normalized_cutoffs );
    }

    public function handle_force_auto_import() {
        check_admin_referer( 'rcp_force_auto_import' );
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_die( __( 'Permission denied', 'rcp-api-wp-integration' ) );
        }
        $this->clear_feed_cache();
        
        // Reset category cutoffs to allow importing older posts
        $this->reset_category_cutoffs();
        
        // Run the auto import
        $this->handle_auto_import( '', self::MAX_INITIAL_POSTS, true );
        
        // Ensure the log is persisted before redirect
        wp_cache_flush(); // Flush any object cache
        
        // Schedule next auto import
        $this->maybe_schedule_auto_import();
        
        // Add a success message to the redirect
        $redirect_url = wp_get_referer() ? wp_get_referer() : admin_url( 'admin.php?page=rcp-api-settings' );
        $redirect_url = add_query_arg( 'rcp_import', 'forced', $redirect_url );
        
        wp_redirect( $redirect_url );
        exit;
    }
    
    /**
     * Handle clearing the import lock when it gets stuck.
     */
    public function handle_clear_import_lock() {
        check_admin_referer( 'rcp_clear_import_lock' );
        if ( ! current_user_can( self::CAPABILITY ) ) {
            wp_die( __( 'Permission denied', 'rcp-api-wp-integration' ) );
        }
        
        // Clear the import lock
        delete_transient( 'rcp_import_running' );
        
        // Redirect back to diagnostics page with success message
        $redirect_url = admin_url( 'admin.php?page=rcp-api-diagnostics' );
        $redirect_url = add_query_arg( 'lock_cleared', '1', $redirect_url );
        
        wp_redirect( $redirect_url );
        exit;
    }
    
    /**
     * Reset category cutoffs to allow importing older posts.
     * This is useful when Force Auto Import is used or when troubleshooting.
     */
    private function reset_category_cutoffs() {
        $automation = get_option( self::OPTION_CATEGORY_AUTOMATION, [] );
        if ( ! is_array( $automation ) ) { $automation = []; }
        
        $cutoffs = [];
        $limit_ts = current_time( 'timestamp', true ) - 36 * HOUR_IN_SECONDS;
        
        foreach ( $automation as $cat => $mode ) {
            if ( in_array( $mode, [ 'publish', 'draft' ], true ) ) {
                // Normalize category name for consistency
                $normalized = $this->normalize_category_name( $cat );
                if ( $normalized ) {
                    // Set cutoff to 36 hours ago instead of "now" to allow recent posts
                    $cutoffs[ $normalized ] = $limit_ts;
                }
            }
        }
        
        update_option( self::OPTION_CATEGORY_CUTOFFS, $cutoffs );
    }

    public function add_cron_schedules( $schedules ) {
        $schedules['rcp_15min'] = [
            'interval' => 15 * MINUTE_IN_SECONDS,
            'display'  => __( 'Every 15 Minutes', 'rcp-api-wp-integration' ),
        ];
        $schedules['rcp_30min'] = [
            'interval' => 30 * MINUTE_IN_SECONDS,
            'display'  => __( 'Every 30 Minutes', 'rcp-api-wp-integration' ),
        ];
        return $schedules;
    }

    public function maybe_schedule_auto_import() {
        $freq    = get_option( self::OPTION_CRON_FREQUENCY, 'hourly' );
        $current = wp_get_schedule( self::CRON_HOOK );
        if ( $current !== $freq ) {
            wp_clear_scheduled_hook( self::CRON_HOOK );
        }
        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            // wp_schedule_event expects a UTC timestamp. current_time() without
            // the GMT flag returns a value offset to the site's timezone which
            // causes the event to fire at the wrong moment. Use GMT to ensure
            // predictable scheduling regardless of timezone settings.
            wp_schedule_event( current_time( 'timestamp', true ), $freq, self::CRON_HOOK );
        }
    }

    /**
     * Run auto import manually when cron is disabled or significantly overdue.
     */
    public function maybe_run_missed_import() {
        // Only run on admin pages to prevent frontend performance impact
        if ( ! is_admin() ) {
            return;
        }
        
        $next = wp_next_scheduled( self::CRON_HOOK );
        $now  = current_time( 'timestamp', true );
        $last_run = get_option( self::OPTION_LAST_IMPORT, 0 );
        
        // Get the configured frequency
        $freq = get_option( self::OPTION_CRON_FREQUENCY, 'hourly' );
        $schedules = wp_get_schedules();
        $interval = isset( $schedules[ $freq ] ) ? (int) $schedules[ $freq ]['interval'] : HOUR_IN_SECONDS;
        
        // Only run if:
        // 1. No cron scheduled at all, OR
        // 2. Next scheduled is overdue by more than the interval (suggesting cron failure), AND
        // 3. Last run was more than interval + 5 minutes ago (prevent duplicate runs)
        $should_run = false;
        
        if ( ! $next ) {
            // No cron scheduled
            $should_run = true;
        } elseif ( $next + $interval <= $now ) {
            // Cron is significantly overdue
            if ( ! $last_run || $last_run + $interval + 5 * MINUTE_IN_SECONDS <= $now ) {
                $should_run = true;
            }
        }
        
        if ( $should_run ) {
            // Set a transient to prevent concurrent runs
            $lock_key = 'rcp_import_running';
            $lock_value = get_transient( $lock_key );
            
            // Check if lock exists and if it's stale (older than 10 minutes)
            if ( $lock_value ) {
                $lock_time = is_numeric( $lock_value ) ? (int) $lock_value : 0;
                $lock_age = $now - $lock_time;
                
                if ( $lock_age < 5 * MINUTE_IN_SECONDS ) {
                    return; // Import still running or recently failed
                } else {
                    // Lock is stale, clear it
                    delete_transient( $lock_key );
                }
            }
            
            // Set lock with current timestamp (5 minutes max)
            set_transient( $lock_key, $now, 5 * MINUTE_IN_SECONDS );
            
            try {
                // Run the import
                $this->handle_auto_import( '', 0, false );
                
                // Reschedule if needed
                $this->maybe_schedule_auto_import();
            } finally {
                // Always clear the lock, even if import failed
                delete_transient( $lock_key );
            }
        }
    }

    private function get_previous_scheduled_time() {
        $last = get_option( self::OPTION_LAST_IMPORT, '' );
        if ( ! $last ) {
            return '';
        }

        $last_ts = is_numeric( $last ) ? (int) $last : strtotime( $last );
        if ( ! $last_ts ) {
            return '';
        }

        $freq      = get_option( self::OPTION_CRON_FREQUENCY, 'hourly' );
        $schedules  = wp_get_schedules();
        $interval   = isset( $schedules[ $freq ] ) ? (int) $schedules[ $freq ]['interval'] : HOUR_IN_SECONDS;
        $now        = current_time( 'timestamp', true );
        $missed     = floor( ( $now - $last_ts ) / $interval );

        return gmdate( 'Y-m-d H:i:s', $last_ts + $missed * $interval );
    }

    /**
     * Ensure WordPress media handling functions are loaded.
     */
    public function load_media_functions() {
        if ( ! function_exists( 'media_handle_sideload' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/media.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }
    }

    /**
     * Handle plugin upgrades by recording the current time and version.
     */
    public function maybe_upgrade() {
        $stored = get_option( self::OPTION_VERSION, '' );
        if ( version_compare( $stored, self::VERSION, '<' ) ) {
            update_option( self::OPTION_LAST_IMPORT, current_time( 'timestamp', true ) );
            if ( false === get_option( self::OPTION_CATEGORY_CUTOFFS, false ) ) {
                update_option( self::OPTION_CATEGORY_CUTOFFS, [] );
            }
            update_option( self::OPTION_VERSION, self::VERSION );
            $this->maybe_schedule_auto_import();
            // Ensure capability present after upgrade
            if ( $role = get_role( 'administrator' ) ) {
                $role->add_cap( self::CAPABILITY );
            }
        }
    }

    /**
     * Inject HTTP Basic auth header for requests to the RCP API.
     *
     * @param array  $args Request arguments.
     * @param string $url  Request URL.
     * @return array Modified args.
     */
    public function add_auth_header( $args, $url ) {
        $api_host = parse_url( $this->api_base, PHP_URL_HOST );
        $req_host = parse_url( $url, PHP_URL_HOST );

        if ( $api_host && $req_host && $api_host === $req_host ) {
            $username = get_option( self::OPTION_USERNAME );
            $password = get_option( self::OPTION_PASSWORD );
            if ( ! isset( $args['headers'] ) ) {
                $args['headers'] = [];
            }
            $args['headers']['Authorization'] = 'Basic ' . base64_encode( $username . ':' . $password );
            $timeout = apply_filters( 'rcp_api_timeout', self::API_TIMEOUT );
            if ( empty( $args['timeout'] ) || $args['timeout'] < $timeout ) {
                $args['timeout'] = $timeout;
            }
        }

        return $args;
    }

    /**
     * Validate URL for image downloads to prevent SSRF attacks.
     *
     * @param string $url URL to validate.
     * @return string|WP_Error Sanitized URL or error.
     */
    private function validate_download_url( $url ) {
        try {
            $api_host = wp_parse_url( $this->api_base, PHP_URL_HOST );
            $allowed  = [];

            if ( $api_host ) {
                $api_host  = strtolower( $api_host );
                $base_host = preg_replace( '/^www\./i', '', $api_host );

                $allowed[] = $api_host;
                if ( $base_host ) {
                    $allowed[] = strtolower( $base_host );
                    $allowed[] = 'www.' . strtolower( $base_host );
                }
            }

            $allowed = array_values( array_unique( array_filter( $allowed ) ) );
            $allowed = apply_filters( 'rcp_api_allowed_download_hosts', $allowed, $url, $this );

            $validated_url = RCP_API_Validator::validate_url( $url, $allowed );
            return $validated_url;
        } catch ( RCP_API_Validation_Exception $e ) {
            return new WP_Error( 'invalid_url', $e->getMessage() );
        }
    }

    /**
     * Validate post date input.
     *
     * @param string $date Date string to validate.
     * @return string|WP_Error Validated date or error.
     */
    private function validate_post_date( $date ) {
        if ( empty( $date ) ) {
            return '';
        }

        $date = sanitize_text_field( $date );
        
        // Check if it's a valid date format
        $timestamp = strtotime( $date );
        if ( false === $timestamp ) {
            return new WP_Error( 'invalid_date', __( 'Invalid date format', 'rcp-api-wp-integration' ) );
        }

        // Check if date is not too far in the future or past
        $now = current_time( 'timestamp' );
        $max_future = $now + ( 365 * DAY_IN_SECONDS ); // 1 year in future
        $max_past = $now - ( 5 * 365 * DAY_IN_SECONDS ); // 5 years in past

        if ( $timestamp > $max_future ) {
            return new WP_Error( 'date_too_future', __( 'Date cannot be more than 1 year in the future', 'rcp-api-wp-integration' ) );
        }

        if ( $timestamp < $max_past ) {
            return new WP_Error( 'date_too_past', __( 'Date cannot be more than 5 years in the past', 'rcp-api-wp-integration' ) );
        }

        return $date;
    }

    /**
     * Normalize an API datetime string to WordPress local/GMT formats for insertion.
     *
     * @param string $date_string Date string (expected ISO8601).
     * @return array|false { local, gmt } or false on failure.
     */
    private function normalize_post_date_for_insert( $date_string ) {
        if ( empty( $date_string ) || ! is_string( $date_string ) ) {
            return false;
        }

        $date_string = trim( $date_string );
        if ( '' === $date_string ) {
            return false;
        }

        // Replace "T" separator if present to improve strtotime compatibility.
        $normalized = str_replace( 'T', ' ', $date_string );
        $timestamp  = strtotime( $normalized );

        if ( false === $timestamp ) {
            return false;
        }

        $gmt   = gmdate( 'Y-m-d H:i:s', $timestamp );
        $local = get_date_from_gmt( $gmt, 'Y-m-d H:i:s' );

        if ( ! $local ) {
            $local = gmdate( 'Y-m-d H:i:s', $timestamp + ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
        }

        return [
            'local' => $local,
            'gmt'   => $gmt,
        ];
    }

    /**
     * Sideload a protected image using authentication.
     *
     * @param string $url     Image URL.
     * @param int    $post_id Post to attach image to.
     * @return int|WP_Error Attachment ID or error.
     */
    private function sideload_protected_image( $url, $post_id, $filename_from = '', $alt = '', $caption = '', $description = '' ) {
        $validated_url = $this->validate_download_url( $url );
        if ( is_wp_error( $validated_url ) ) {
            return $validated_url;
        }
        $url = $validated_url;

        // Enforce image safety with HEAD check (content-type and content-length)
        $max_bytes = (int) apply_filters( 'rcp_image_max_bytes', 10 * 1024 * 1024 ); // 10MB default
        add_filter( 'http_request_args', [ $this, 'add_auth_header' ], 10, 2 );
        $head = wp_safe_remote_head( $url, [ 'redirection' => 3, 'timeout' => self::API_TIMEOUT ] );
        if ( ! is_wp_error( $head ) ) {
            $headers = wp_remote_retrieve_headers( $head );
            $ctype   = isset( $headers['content-type'] ) ? strtolower( $headers['content-type'] ) : '';
            $clen    = isset( $headers['content-length'] ) ? (int) $headers['content-length'] : 0;
            if ( $ctype && strpos( $ctype, 'image/' ) !== 0 ) {
                remove_filter( 'http_request_args', [ $this, 'add_auth_header' ], 10 );
                return new WP_Error( 'invalid_content_type', __( 'Invalid image content type.', 'rcp-api-wp-integration' ) );
            }
            if ( $clen && $clen > $max_bytes ) {
                remove_filter( 'http_request_args', [ $this, 'add_auth_header' ], 10 );
                return new WP_Error( 'image_too_large', __( 'Image file is too large.', 'rcp-api-wp-integration' ) );
            }
        }

        $this->load_media_functions();

        // Download with timeout and validate size post-download as well
        $tmp = download_url( $url, self::API_TIMEOUT );
        remove_filter( 'http_request_args', [ $this, 'add_auth_header' ], 10 );
        if ( is_wp_error( $tmp ) ) {
            return $tmp;
        }

        // Verify downloaded file size
        if ( file_exists( $tmp ) && filesize( $tmp ) > $max_bytes ) {
            @unlink( $tmp );
            return new WP_Error( 'image_too_large', __( 'Image file is too large after download.', 'rcp-api-wp-integration' ) );
        }

        $filename = '';
        if ( $filename_from ) {
            $filename = basename( parse_url( $filename_from, PHP_URL_PATH ) );
        }
        if ( ! $filename ) {
            $filename = basename( parse_url( $url, PHP_URL_PATH ) );
        }
        if ( ! preg_match( '/\.[A-Za-z0-9]{2,5}$/', $filename ) ) {
            $filename .= '.jpg';
        }

        $file_array = [
            'name'     => sanitize_file_name( $filename ),
            'tmp_name' => $tmp,
        ];

        $attachment_id = media_handle_sideload( $file_array, $post_id );
        if ( is_wp_error( $attachment_id ) ) {
            @unlink( $tmp );
            return $attachment_id;
        }

        if ( $alt ) {
            update_post_meta( $attachment_id, '_wp_attachment_image_alt', sanitize_text_field( $alt ) );
        }

        if ( $caption || $description ) {
            wp_update_post( [
                'ID'           => $attachment_id,
                'post_excerpt' => wp_kses_post( $caption ),
                'post_content' => wp_kses_post( $description ),
            ] );
        }

        return $attachment_id;
    }


    public static function activate() {
        $freq = get_option( self::OPTION_CRON_FREQUENCY, 'hourly' );
        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            // Use GMT for scheduling to avoid timezone discrepancies.
            wp_schedule_event( current_time( 'timestamp', true ), $freq, self::CRON_HOOK );
        }
        // Record activation time so the first cron run starts from now.
        update_option( self::OPTION_LAST_IMPORT, current_time( 'timestamp', true ) );
        update_option( self::OPTION_CATEGORY_CUTOFFS, [] );
        update_option( self::OPTION_VERSION, self::VERSION );
        
        // Ensure custom capability is granted to administrators
        if ( $role = get_role( 'administrator' ) ) {
            $role->add_cap( self::CAPABILITY );
        }

        // Add database indexes for performance
        self::add_database_indexes();
        
        // Run database migrations
        RCP_API_Database_Manager::check_database();
    }

    /**
     * Add database indexes for improved performance.
     */
    private static function add_database_indexes() {
        // Allow gating index creation via filter to avoid altering core tables by default
        if ( ! apply_filters( 'rcp_api_enable_core_indexes', false ) ) {
            return;
        }
        global $wpdb;
        
        // Suppress errors to prevent fatal errors during index creation
        $wpdb->suppress_errors( true );
        
        try {
            // Add index for meta queries on _rcp_original_post_id
            $index_name = 'rcp_original_post_id_idx';
            $table_name = $wpdb->postmeta;
            
            // Properly escape table name and check if index exists
            $safe_table = esc_sql( $table_name );
            $existing_indexes_query = $wpdb->prepare(
                "SHOW INDEX FROM `$safe_table` WHERE Key_name = %s",
                $index_name
            );
            $existing_indexes = $wpdb->get_results( $existing_indexes_query );
            
            if ( empty( $existing_indexes ) ) {
                // Create index with properly escaped names
                $safe_index = esc_sql( $index_name );
                $wpdb->query( "CREATE INDEX `$safe_index` ON `$safe_table` (meta_key(20), meta_value(20))" );
            }
            
            // Add composite index for post status and meta queries
            $composite_index = 'rcp_post_status_meta_idx';
            $posts_table = $wpdb->posts;
            
            $safe_posts_table = esc_sql( $posts_table );
            $existing_composite_query = $wpdb->prepare(
                "SHOW INDEX FROM `$safe_posts_table` WHERE Key_name = %s",
                $composite_index
            );
            $existing_composite = $wpdb->get_results( $existing_composite_query );
            
            if ( empty( $existing_composite ) ) {
                // Create composite index with escaped names
                $safe_composite_index = esc_sql( $composite_index );
                $wpdb->query( "CREATE INDEX `$safe_composite_index` ON `$safe_posts_table` (post_status, post_type, post_date)" );
            }
        } catch ( Exception $e ) {
            // Log error but don't break plugin activation
            error_log( 'RCP API Plugin: Failed to create database indexes: ' . $e->getMessage() );
        }
        
        // Re-enable error reporting
        $wpdb->suppress_errors( false );
    }

    public static function deactivate() {
        wp_clear_scheduled_hook( self::CRON_HOOK );
    }

    /**
     * Retrieve the list of fields required for post imports.
     *
     * @return string Comma-separated field list.
     */
    private function get_post_import_fields( $force_full = false ) {
        if ( $force_full ) {
            return 'all';
        }
        $fields = [
            'id',
            'slug',
            'title',
            'published',
            'date',
            'modified',
            'link',
            'blog_post_title',
            'blog_post_html',
            'content_html',
            'content',
            'categories',
            'featured_image',
            'featured_image_url',
            'featured_media',
            'featured_image_html',
            'ai_image_generation_prompt',
        ];

        $fields = apply_filters( 'rcp_api_post_import_fields', $fields );
        $fields = array_unique( array_filter( array_map( 'trim', (array) $fields ) ) );

        return implode( ',', $fields );
    }

    /**
     * Build a cache key for a fetched RCP post.
     *
     * @param int $rcp_post_id RCP post ID.
     * @return string Cache key.
     */
    private function get_post_cache_key( $rcp_post_id, $force_full = false ) {
        $key = 'rcp_api_post_' . (int) $rcp_post_id;
        if ( $force_full ) {
            $key .= '_full';
        }
        return $key;
    }

    /**
     * Fetch a single RCP post with caching and required fields.
     *
     * @param int  $rcp_post_id  RCP post ID.
     * @param bool $bypass_cache Whether to bypass the object cache.
     * @return array|WP_Error
     */
    private function fetch_rcp_post( $rcp_post_id, $bypass_cache = false, $force_full = false ) {
        $rcp_post_id = (int) $rcp_post_id;
        if ( $rcp_post_id <= 0 ) {
            return new WP_Error( 'invalid_post', __( 'Invalid post.', 'rcp-api-wp-integration' ) );
        }

        $cache_key = $this->get_post_cache_key( $rcp_post_id, $force_full );
        $this->debug_log( 'Fetching RCP post data', [ 'post_id' => $rcp_post_id, 'bypass_cache' => (bool) $bypass_cache, 'force_full' => (bool) $force_full ] );
        if ( ! $bypass_cache ) {
            $cached = wp_cache_get( $cache_key, 'rcp_api' );
            if ( false !== $cached ) {
                $this->debug_log( 'Using cached RCP post response', [ 'post_id' => $rcp_post_id ] );
                return $cached;
            }
        }

        $username = get_option( self::OPTION_USERNAME );
        $password = get_option( self::OPTION_PASSWORD );

        if ( empty( $username ) || empty( $password ) ) {
            return new WP_Error( 'missing_creds', __( 'Missing credentials', 'rcp-api-wp-integration' ) );
        }

        $timeout = apply_filters( 'rcp_api_timeout', self::API_TIMEOUT );
        $args = [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode( $username . ':' . $password ),
            ],
            'timeout' => $timeout,
        ];

        $url = trailingslashit( $this->api_base ) . 'posts/' . $rcp_post_id;
        $fields = $this->get_post_import_fields( $force_full );
        if ( $fields ) {
            $url = add_query_arg( 'fields', $fields, $url );
        }

        $this->debug_log( 'Requesting RCP post via API', [ 'post_id' => $rcp_post_id, 'url' => $url ] );
        $resp = wp_safe_remote_get( $url, array_merge( [
            'redirection' => 3,
            'reject_unsafe_urls' => true,
        ], $args ) );

        if ( is_wp_error( $resp ) ) {
            return $resp;
        }

        if ( 200 !== wp_remote_retrieve_response_code( $resp ) ) {
            $this->debug_log( 'RCP post request failed', [
                'post_id' => $rcp_post_id,
                'status'  => wp_remote_retrieve_response_code( $resp ),
                'body'    => wp_remote_retrieve_body( $resp ),
            ] );
            return new WP_Error( 'api_error', __( 'Failed to fetch post from API', 'rcp-api-wp-integration' ) );
        }

        $data = json_decode( wp_remote_retrieve_body( $resp ), true );
        if ( empty( $data ) || ! is_array( $data ) ) {
            $this->debug_log( 'RCP post response invalid', [
                'post_id' => $rcp_post_id,
                'raw'     => wp_remote_retrieve_body( $resp ),
            ] );
            return new WP_Error( 'invalid_data', __( 'Invalid API response', 'rcp-api-wp-integration' ) );
        }

        $cache_ttl = (int) apply_filters( 'rcp_api_post_cache_ttl', 5 * MINUTE_IN_SECONDS, $rcp_post_id, $data );
        if ( $cache_ttl > 0 ) {
            wp_cache_set( $cache_key, $data, 'rcp_api', $cache_ttl );
            $this->debug_log( 'Cached RCP post response', [ 'post_id' => $rcp_post_id, 'ttl' => $cache_ttl ] );
        }

        // Ensure featured image data is present; if not, retry once with full fields
        if ( ! $force_full && empty( $data['featured_image'] ) && empty( $data['featured_image_url'] ) && empty( $data['featured_media'] ) && empty( $data['featured_image_html'] ) ) {
            $this->debug_log( 'Missing featured image data, refetching with full field set', [ 'post_id' => $rcp_post_id ] );
            return $this->fetch_rcp_post( $rcp_post_id, true, true );
        }

        return $data;
    }

    /**
     * Convert relative API paths to absolute URLs.
     *
     * @param string $url Potentially relative URL.
     * @return string Normalized absolute URL.
     */
    private function normalize_api_url( $url ) {
        if ( ! is_string( $url ) ) {
            return '';
        }

        $url = trim( $url );
        if ( '' === $url ) {
            return '';
        }

        if ( 0 === strpos( $url, 'http://' ) || 0 === strpos( $url, 'https://' ) ) {
            return $url;
        }

        if ( 0 === strpos( $url, '//' ) ) {
            return 'https:' . $url;
        }

        $root = preg_replace( '#/wp-json/rcp/v1/?$#', '', $this->api_base );
        if ( ! $root ) {
            return $url;
        }

        return rtrim( $root, '/' ) . '/' . ltrim( $url, '/' );
    }

    /**
     * Fetch a protected media item, including download details.
     *
     * @param int  $media_id     Media attachment ID in the RCP API.
     * @param bool $bypass_cache Whether to bypass the object cache.
     * @return array|WP_Error
     */
    private function fetch_rcp_media_item( $media_id, $bypass_cache = false ) {
        $media_id = (int) $media_id;
        if ( $media_id <= 0 ) {
            return new WP_Error( 'invalid_media', __( 'Invalid media item.', 'rcp-api-wp-integration' ) );
        }

        $cache_key = 'rcp_api_media_' . $media_id;
        $this->debug_log( 'Fetching RCP media item', [ 'media_id' => $media_id, 'bypass_cache' => (bool) $bypass_cache ] );
        if ( ! $bypass_cache ) {
            $cached = wp_cache_get( $cache_key, 'rcp_api' );
            if ( false !== $cached ) {
                $this->debug_log( 'Using cached RCP media response', [ 'media_id' => $media_id ] );
                return $cached;
            }
        }

        $username = get_option( self::OPTION_USERNAME );
        $password = get_option( self::OPTION_PASSWORD );

        if ( empty( $username ) || empty( $password ) ) {
            return new WP_Error( 'missing_creds', __( 'Missing credentials', 'rcp-api-wp-integration' ) );
        }

        $timeout = apply_filters( 'rcp_api_timeout', self::API_TIMEOUT );
        $args = [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode( $username . ':' . $password ),
            ],
            'timeout' => $timeout,
        ];

        $url  = trailingslashit( $this->api_base ) . 'media/' . $media_id;
        $this->debug_log( 'Requesting RCP media via API', [ 'media_id' => $media_id, 'url' => $url ] );
        $resp = wp_safe_remote_get( $url, array_merge( [
            'redirection' => 3,
            'reject_unsafe_urls' => true,
        ], $args ) );

        if ( is_wp_error( $resp ) ) {
            return $resp;
        }

        if ( 200 !== wp_remote_retrieve_response_code( $resp ) ) {
            $this->debug_log( 'RCP media request failed', [
                'media_id' => $media_id,
                'status'   => wp_remote_retrieve_response_code( $resp ),
                'body'     => wp_remote_retrieve_body( $resp ),
            ] );
            return new WP_Error( 'api_error', __( 'Failed to fetch media item', 'rcp-api-wp-integration' ) );
        }

        $data = json_decode( wp_remote_retrieve_body( $resp ), true );
        if ( empty( $data ) || ! is_array( $data ) ) {
            $this->debug_log( 'RCP media response invalid', [
                'media_id' => $media_id,
                'raw'      => wp_remote_retrieve_body( $resp ),
            ] );
            return new WP_Error( 'invalid_data', __( 'Invalid media response', 'rcp-api-wp-integration' ) );
        }

        $cache_ttl = (int) apply_filters( 'rcp_api_media_cache_ttl', 5 * MINUTE_IN_SECONDS, $media_id, $data );
        if ( $cache_ttl > 0 ) {
            wp_cache_set( $cache_key, $data, 'rcp_api', $cache_ttl );
            $this->debug_log( 'Cached RCP media response', [ 'media_id' => $media_id, 'ttl' => $cache_ttl ] );
        }

        return $data;
    }

    /**
     * Import and set the featured image for a WordPress post based on the RCP post ID.
     * Runs the same download URL resolution as import_post().
     *
     * @param int $rcp_post_id RCP post ID.
     * @param int $wp_post_id  WordPress post ID.
     * @return bool|WP_Error True on success or WP_Error on failure.
     */
    public function import_featured_image_for_rcp_post( $rcp_post_id, $wp_post_id, $prefetched_post = null, $image_mode = null ) {
        if ( is_array( $prefetched_post ) ) {
            $data = $prefetched_post;
            $this->debug_log( 'Using prefetched data for featured image import', [ 'post_id' => $rcp_post_id, 'wp_post_id' => $wp_post_id ] );
        } else {
            $data = $this->fetch_rcp_post( $rcp_post_id );
            if ( is_wp_error( $data ) ) {
                return $data;
            }
        }

        if ( empty( $data['featured_image'] ) && empty( $data['featured_image_url'] ) && empty( $data['featured_media'] ) && empty( $data['featured_image_html'] ) ) {
            $this->debug_log( 'Featured image fields missing; refetching with full payload', [ 'post_id' => $rcp_post_id ] );
            $data = $this->fetch_rcp_post( $rcp_post_id, true, true );
            if ( is_wp_error( $data ) ) {
                return $data;
            }
        }

        if ( null === $image_mode ) {
            $image_mode = $this->resolve_category_image_mode( $data['categories'] ?? [] );
        }

        if ( empty( $image_mode ) ) {
            $image_mode = 'none';
        }

        if ( ! in_array( $image_mode, [ 'import', 'fallback' ], true ) ) {
            $this->debug_log( 'Featured image import skipped (mode)', [ 'post_id' => $rcp_post_id, 'mode' => $image_mode ] );
            return true;
        }

        // Resolve download URL and metadata
        $download_url  = '';
        $filename_from = '';
        $alt           = '';
        $caption       = '';
        $description   = '';

        if ( ! empty( $data['featured_image'] ) ) {
            if ( is_array( $data['featured_image'] ) ) {
                $image = $data['featured_image'];
                $download_url = $image['download_url'] ?? '';

                if ( empty( $download_url ) && ! empty( $image['source_url'] ) ) {
                    $download_url = $image['source_url'];
                }
                if ( empty( $download_url ) && ! empty( $image['url'] ) ) {
                    $download_url = $image['url'];
                }
                if ( empty( $download_url ) && ! empty( $image['sizes'] ) && is_array( $image['sizes'] ) ) {
                    foreach ( [ 'full', 'large', 'medium_large', 'medium' ] as $size_key ) {
                        if ( ! empty( $image['sizes'][ $size_key ] ) ) {
                            $download_url = $image['sizes'][ $size_key ];
                            break;
                        }
                    }
                }

                $filename_from = $image['url'] ?? ( $image['source_url'] ?? $download_url );
                $alt           = $image['alt'] ?? '';
                $caption       = $image['caption'] ?? '';
                $description   = $image['description'] ?? '';
            } elseif ( is_string( $data['featured_image'] ) ) {
                $download_url  = $data['featured_image'];
                $filename_from = $data['featured_image'];
            }
        }

        if ( empty( $download_url ) && ! empty( $data['featured_image_url'] ) ) {
            $download_url = $data['featured_image_url'];
        }

        if ( empty( $download_url ) && ! empty( $data['featured_image_html'] ) && is_string( $data['featured_image_html'] ) ) {
            $html = $data['featured_image_html'];
            if ( preg_match( '/src\s*=\s*"([^"]+)"/i', $html, $matches ) ) {
                $download_url = html_entity_decode( $matches[1], ENT_QUOTES );
            }
            if ( '' === $alt && preg_match( '/alt\s*=\s*"([^"]*)"/i', $html, $matches ) ) {
                $alt = html_entity_decode( $matches[1], ENT_QUOTES );
            }
            if ( '' === $caption && preg_match( '/title\s*=\s*"([^"]*)"/i', $html, $matches ) ) {
                $caption = html_entity_decode( $matches[1], ENT_QUOTES );
            }
            if ( '' === $filename_from && $download_url ) {
                $filename_from = $download_url;
            }
        }

        if ( empty( $download_url ) && ! empty( $data['featured_media'] ) ) {
            $media_id = absint( $data['featured_media'] );
            if ( $media_id ) {
                $media_data = $this->fetch_rcp_media_item( $media_id );
                if ( ! is_wp_error( $media_data ) ) {
                    if ( ! empty( $media_data['download_url'] ) ) {
                        $download_url = $media_data['download_url'];
                    } elseif ( ! empty( $media_data['source_url'] ) ) {
                        $download_url = $media_data['source_url'];
                    }
                    if ( empty( $filename_from ) ) {
                        $filename_from = $media_data['source_url'] ?? '';
                    }
                    if ( '' === $alt && ! empty( $media_data['alt_text'] ) ) {
                        $alt = $media_data['alt_text'];
                    }
                    if ( '' === $caption && ! empty( $media_data['caption'] ) ) {
                        $caption = $media_data['caption'];
                    }
                    if ( '' === $description && ! empty( $media_data['description'] ) ) {
                        $description = $media_data['description'];
                    }
                }
            }
        }

        $download_url  = $this->normalize_api_url( $download_url );
        $filename_from = $filename_from ? $this->normalize_api_url( $filename_from ) : '';

        $this->debug_log( 'Resolved featured image data', [
            'post_id'       => $rcp_post_id,
            'wp_post_id'    => $wp_post_id,
            'download_url'  => $download_url,
            'filename_from' => $filename_from,
            'alt'           => $alt,
            'caption'       => $caption,
            'description'   => $description,
        ] );

        if ( empty( $download_url ) ) {
            $this->debug_log( 'No featured image URL resolved', [ 'post_id' => $rcp_post_id, 'wp_post_id' => $wp_post_id, 'fields' => array_keys( $data ) ] );
            return new WP_Error( 'no_image', __( 'No featured image available', 'rcp-api-wp-integration' ) );
        }

        $image_id = $this->sideload_protected_image( $download_url, $wp_post_id, $filename_from, $alt, $caption, $description );
        if ( is_wp_error( $image_id ) ) {
            $this->debug_log( 'Featured image sideload failed', [
                'post_id'    => $rcp_post_id,
                'wp_post_id' => $wp_post_id,
                'error'      => $image_id->get_error_message(),
            ] );
            return $image_id;
        }
        $this->debug_log( 'Featured image set successfully', [ 'post_id' => $rcp_post_id, 'wp_post_id' => $wp_post_id, 'attachment_id' => $image_id ] );
        set_post_thumbnail( $wp_post_id, $image_id );
        return true;
    }

    /**
     * Generate an AI image for a given WP post using RCP credits and attach as featured image.
     *
     * @param int         $wp_post_id     Target WordPress post ID.
     * @param int         $rcp_post_id    Source RCP post ID (optional for better prompt).
     * @param string|null $prompt_override Optional custom prompt to use.
     * @return array|WP_Error Attachment info or queued job details.
     */
    public function generate_ai_image_for_post( $wp_post_id, $rcp_post_id = 0, $prompt_override = '' ) {
        $wp_post_id  = (int) $wp_post_id;
        $rcp_post_id = (int) $rcp_post_id;

        if ( ! $wp_post_id || ! get_post( $wp_post_id ) ) {
            return new WP_Error( 'invalid_post', __( 'Invalid post.', 'rcp-api-wp-integration' ) );
        }

        if ( ! $rcp_post_id ) {
            $meta_rcp_id = get_post_meta( $wp_post_id, '_rcp_original_post_id', true );
            if ( $meta_rcp_id ) {
                $rcp_post_id = (int) $meta_rcp_id;
            }
        }

        if ( has_post_thumbnail( $wp_post_id ) ) {
            return [ 'skipped' => true ];
        }

        $prompt = (string) $prompt_override;
        if ( '' === $prompt ) {
            $prompt = $this->get_ai_prompt( $rcp_post_id, $wp_post_id );
        }
        if ( empty( $prompt ) ) {
            return new WP_Error( 'no_prompt', __( 'Could not determine an AI prompt for this post.', 'rcp-api-wp-integration' ) );
        }

        $reservation_id = '';
        $reservation    = $this->reserve_ai_image_credits();
        if ( is_wp_error( $reservation ) ) {
            $this->record_ai_image_log( [
                'time' => current_time( 'timestamp', true ),
                'post_id' => $wp_post_id,
                'rcp_post_id' => $rcp_post_id,
                'status' => 'reservation_failed',
                'message' => $reservation->get_error_message(),
            ] );
        } else {
            $reservation_id = isset( $reservation['reservation_id'] ) ? sanitize_text_field( (string) $reservation['reservation_id'] ) : '';
            if ( ! $reservation_id ) {
                $this->record_ai_image_log( [
                    'time' => current_time( 'timestamp', true ),
                    'post_id' => $wp_post_id,
                    'rcp_post_id' => $rcp_post_id,
                    'status' => 'reservation_missing',
                    'message' => __( 'Credits API did not provide a reservation ID.', 'rcp-api-wp-integration' ),
                ] );
            }
        }

        $response = $this->request_ai_image_generate( [
            'prompt'         => $prompt,
            'rcp_post_id'    => $rcp_post_id,
            'wp_post_id'     => $wp_post_id,
            'reservation_id' => $reservation_id,
        ] );
        if ( is_wp_error( $response ) ) {
            if ( $reservation_id ) {
                $this->cancel_ai_image_reservation( $reservation_id );
            }
            return $response;
        }

        if ( isset( $response['result_url'] ) ) {
            $attach = $this->attach_image_from_url( $wp_post_id, $response['result_url'], $prompt );
            if ( is_wp_error( $attach ) ) {
                if ( $reservation_id ) {
                    $this->cancel_ai_image_reservation( $reservation_id );
                }
                return $attach;
            }

            $context = [
                'wp_post_id'  => $wp_post_id,
                'rcp_post_id' => $rcp_post_id,
                'result_url'  => $response['result_url'],
            ];
            if ( isset( $response['generation_log_id'] ) ) {
                $context['generation_log_id'] = sanitize_text_field( $response['generation_log_id'] );
            }
            if ( $reservation_id ) {
                $confirm = $this->confirm_ai_image_reservation( $reservation_id, $context );
                if ( is_wp_error( $confirm ) ) {
                    $this->record_ai_image_log( [
                        'time' => current_time( 'timestamp', true ),
                        'post_id' => $wp_post_id,
                        'rcp_post_id' => $rcp_post_id,
                        'status' => 'confirm_failed',
                        'reservation_id' => $reservation_id,
                        'message' => $confirm->get_error_message(),
                    ] );
                }
            }

            $attach['reservation_id'] = $reservation_id;
            if ( empty( $attach['reservation_id'] ) ) {
                unset( $attach['reservation_id'] );
            }
            return $attach;
        }

        if ( isset( $response['generation_log_id'] ) && $response['generation_log_id'] ) {
            // Store in-flight marker for idempotency/diagnostics
            update_post_meta( $wp_post_id, '_rcp_ai_generation_log_id', sanitize_text_field( $response['generation_log_id'] ) );
            if ( $reservation_id ) {
                update_post_meta( $wp_post_id, '_rcp_ai_reservation_id', $reservation_id );
            }
            $this->record_ai_image_log( [
                'time' => current_time( 'timestamp', true ),
                'post_id' => $wp_post_id,
                'rcp_post_id' => $rcp_post_id,
                'status' => 'queued',
                'generation_log_id' => $response['generation_log_id'],
                'reservation_id' => $reservation_id,
            ] );
            RCP_API_Job_Queue::schedule_single( 'ai_poll', [
                'post_id'           => $wp_post_id,
                'generation_log_id' => $response['generation_log_id'],
                'attempts'          => 0,
                'reservation_id'   => $reservation_id,
            ], time() + 10 );
            return [ 'queued' => true, 'generation_log_id' => $response['generation_log_id'], 'reservation_id' => $reservation_id ];
        }

        if ( $reservation_id ) {
            $this->cancel_ai_image_reservation( $reservation_id );
        }
        return new WP_Error( 'invalid_response', __( 'Unexpected AI generation response.', 'rcp-api-wp-integration' ) );
    }

    /**
     * Attach an image by URL as featured image.
     *
     * @param int    $wp_post_id Post ID.
     * @param string $url        Image URL.
     * @param string $alt        Alt text.
     * @return array|WP_Error
     */
    public function attach_image_from_url( $wp_post_id, $url, $alt = '' ) {
        if ( has_post_thumbnail( $wp_post_id ) ) {
            return new WP_Error( 'has_thumbnail', __( 'Post already has a featured image.', 'rcp-api-wp-integration' ) );
        }
        $filename_from = basename( parse_url( $url, PHP_URL_PATH ) );
        $post_title = get_the_title( $wp_post_id );
        if ( ! is_string( $post_title ) ) { $post_title = ''; }
        $post_title = trim( wp_strip_all_tags( $post_title ) );
        if ( '' === $post_title ) {
            $post_title = __( 'Featured image', 'rcp-api-wp-integration' );
        }
        // For AI-generated images, set both alt text and attachment title to the blog post title
        $image_id = $this->sideload_protected_image( $url, $wp_post_id, $filename_from, $post_title );
        if ( is_wp_error( $image_id ) ) {
            return $image_id;
        }
        // Ensure the attachment post_title matches the blog post title
        wp_update_post( [ 'ID' => (int) $image_id, 'post_title' => $post_title ] );
        set_post_thumbnail( $wp_post_id, $image_id );
        // Clear any in-flight marker
        delete_post_meta( $wp_post_id, '_rcp_ai_generation_log_id' );
        // Log success
        $this->record_ai_image_log( [
            'time' => current_time( 'timestamp', true ),
            'post_id' => $wp_post_id,
            'status' => 'success',
            'result_url' => esc_url_raw( $url ),
            'attachment_id' => (int) $image_id,
        ] );
        return [ 'attachment_id' => $image_id ];
    }

    /**
     * Record an AI image generation log entry (recent history kept).
     *
     * @param array $entry
     */
    public function record_ai_image_log( array $entry ) {
        $log = get_option( 'rcp_ai_image_log', [] );
        if ( ! is_array( $log ) ) { $log = []; }
        $entry['time'] = isset( $entry['time'] ) ? (int) $entry['time'] : current_time( 'timestamp', true );
        $log[] = $entry;
        // Trim to last 200 entries
        if ( count( $log ) > 200 ) {
            $log = array_slice( $log, -200 );
        }
        update_option( 'rcp_ai_image_log', $log, false );
    }

    /**
     * Build AI image prompt (default via API; fallback from post data).
     */
    private function get_ai_prompt( $rcp_post_id, $wp_post_id = 0 ) {
        $prompt_meta_key = '_rcp_ai_prompt';
        $prompt          = '';

        if ( $wp_post_id ) {
            $stored = get_post_meta( $wp_post_id, $prompt_meta_key, true );
            if ( is_string( $stored ) ) {
                $stored = trim( $stored );
            } else {
                $stored = '';
            }
            if ( '' !== $stored ) {
                $prompt = $stored;
            }
        }

        $username = get_option( self::OPTION_USERNAME );
        $password = get_option( self::OPTION_PASSWORD );

        if ( '' === $prompt && ! empty( $username ) && ! empty( $password ) && $rcp_post_id ) {
            $endpoint_path = apply_filters( 'rcp_api_ai_prompt_endpoint', 'ai_image_generation_prompt' );
            $param_name    = apply_filters( 'rcp_api_ai_prompt_param', 'post_id' );
            $url           = add_query_arg( [ $param_name => (int) $rcp_post_id ], trailingslashit( $this->api_base ) . $endpoint_path );
            $args          = [
                'headers'            => [ 'Authorization' => 'Basic ' . base64_encode( $username . ':' . $password ) ],
                'timeout'            => apply_filters( 'rcp_api_timeout', self::API_TIMEOUT ),
                'redirection'        => 3,
                'reject_unsafe_urls' => true,
            ];
            $resp = wp_safe_remote_get( $url, $args );
            if ( ! is_wp_error( $resp ) && 200 === wp_remote_retrieve_response_code( $resp ) ) {
                $body      = wp_remote_retrieve_body( $resp );
                $json      = json_decode( $body, true );
                $raw_prompt = '';
                if ( is_array( $json ) && isset( $json['prompt'] ) ) {
                    $raw_prompt = (string) $json['prompt'];
                } else {
                    $raw_prompt = (string) $body;
                }
                $raw_prompt = trim( $raw_prompt );
                if ( '' !== $raw_prompt ) {
                    $prompt = $raw_prompt;
                    if ( $wp_post_id ) {
                        update_post_meta( $wp_post_id, $prompt_meta_key, $raw_prompt );
                    }
                }
            }
        }

        if ( '' === $prompt && $wp_post_id ) {
            $title   = get_the_title( $wp_post_id );
            $excerpt = get_the_excerpt( $wp_post_id );
            $excerpt = is_string( $excerpt ) ? wp_strip_all_tags( $excerpt ) : '';
            $excerpt = wp_trim_words( $excerpt, 32 );
            $prompt  = sprintf( __( 'Create a compelling, license-safe featured image that visually represents: %1$s. Theme/details: %2$s', 'rcp-api-wp-integration' ), $title, $excerpt );
        }

        return apply_filters( 'rcp_api_default_image_prompt', $prompt, $wp_post_id, $rcp_post_id );
    }

    /**
     * Request AI image generation via primary site (sync or async).
     */
    private function request_ai_image_generate( array $payload ) {
        $body = [
            'prompt'  => (string) ( $payload['prompt'] ?? '' ),
            'context' => [
                'wp_post_id'  => (int) ( $payload['wp_post_id'] ?? 0 ),
                'rcp_post_id' => (int) ( $payload['rcp_post_id'] ?? 0 ),
                'source'      => 'rcp-api-wp-integration',
            ],
        ];

        if ( ! empty( $payload['reservation_id'] ) ) {
            $body['reservation_id'] = sanitize_text_field( (string) $payload['reservation_id'] );
        }

        $timeout_filter = function( $timeout ) use ( $payload ) {
            $override = (int) apply_filters( 'rcp_ai_generate_timeout', 60, $payload );
            return max( (int) $timeout, $override );
        };

        add_filter( 'rcp_api_timeout', $timeout_filter );

        try {
        $this->debug_log( 'AI generate request', [
            'post_id' => $payload['wp_post_id'] ?? 0,
            'rcp_post_id' => $payload['rcp_post_id'] ?? 0,
            'has_reservation' => ! empty( $body['reservation_id'] ),
        ] );
        $json = $this->credits_request( 'POST', 'generate-image', $body );
        } finally {
            remove_filter( 'rcp_api_timeout', $timeout_filter );
        }
        if ( is_wp_error( $json ) ) {
            $this->debug_log( 'AI generate request failed', [
                'post_id' => $payload['wp_post_id'] ?? 0,
                'rcp_post_id' => $payload['rcp_post_id'] ?? 0,
                'error' => $json->get_error_message(),
            ] );
            return $json;
        }
        $out = [];
        if ( isset( $json['result_url'] ) ) {
            $out['result_url'] = esc_url_raw( $json['result_url'] );
        }
        if ( isset( $json['generation_log_id'] ) && $json['generation_log_id'] ) {
            $out['generation_log_id'] = sanitize_text_field( $json['generation_log_id'] );
        }
        $this->debug_log( 'AI generate response', [
            'post_id' => $payload['wp_post_id'] ?? 0,
            'rcp_post_id' => $payload['rcp_post_id'] ?? 0,
            'result_keys' => array_keys( $out ),
        ] );
        return $out;
    }

    public function reserve_ai_image_credits( $amount = 0 ) {
        $cost = $amount > 0 ? (int) $amount : 0;
        if ( $cost <= 0 ) {
            $cost = $this->get_action_cost( 'image_generation' );
            if ( is_wp_error( $cost ) || $cost <= 0 ) {
                $this->debug_log( 'AI reservation cost lookup failed', [ 'error' => is_wp_error( $cost ) ? $cost->get_error_message() : 'invalid cost' ] );
                $cost = (int) apply_filters( 'rcp_ai_default_credit_cost', 4 );
            }
        }

        $body = [
            'action_id' => 'image_generation',
            'amount'    => max( 1, (int) apply_filters( 'rcp_ai_reservation_amount', $cost ) ),
        ];
        $response = $this->credits_request( 'POST', 'reserve', $body );
        if ( is_wp_error( $response ) ) {
            $this->debug_log( 'AI reservation failed', [ 'error' => $response->get_error_message() ] );
        } else {
            $this->debug_log( 'AI reservation created', [ 'reservation' => $response ] );
        }
        return $response;
    }

    public function confirm_ai_image_reservation( $reservation_id, array $context = [] ) {
        if ( ! $reservation_id ) {
            return new WP_Error( 'invalid_reservation', __( 'Missing reservation ID.', 'rcp-api-wp-integration' ) );
        }
        $body = [
            'reservation_id' => sanitize_text_field( (string) $reservation_id ),
        ];
        if ( ! empty( $context ) ) {
            $body['context'] = $context;
        }
        $response = $this->credits_request( 'POST', 'confirm', $body );
        if ( is_wp_error( $response ) ) {
            $this->debug_log( 'AI reservation confirm failed', [ 'reservation_id' => $body['reservation_id'], 'error' => $response->get_error_message() ] );
        } else {
            $this->debug_log( 'AI reservation confirmed', [ 'reservation_id' => $body['reservation_id'] ] );
        }
        return $response;
    }

    public function cancel_ai_image_reservation( $reservation_id ) {
        if ( ! $reservation_id ) {
            return new WP_Error( 'invalid_reservation', __( 'Missing reservation ID.', 'rcp-api-wp-integration' ) );
        }
        $response = $this->credits_request( 'POST', 'cancel', [
            'reservation_id' => sanitize_text_field( (string) $reservation_id ),
        ] );
        if ( is_wp_error( $response ) ) {
            $this->debug_log( 'AI reservation cancel failed', [ 'reservation_id' => $reservation_id, 'error' => $response->get_error_message() ] );
        } else {
            $this->debug_log( 'AI reservation cancelled', [ 'reservation_id' => $reservation_id ] );
        }
        return $response;
    }

    /**
     * Poll job status from the primary site.
     */
    public function poll_ai_job_status( $generation_log_id ) {
        $generation_log_id = sanitize_text_field( (string) $generation_log_id );
        return $this->credits_request( 'GET', 'job-status', null, [ 'generation_log_id' => $generation_log_id ] );
    }

    /**
     * Compute the credits API base from the content API base.
     */
    private function get_credits_api_base() {
        $base = trailingslashit( $this->api_base );
        $base = preg_replace( '#/rcp/\s*v1/?$#i', '/rcp-ai-credits/v1', rtrim( $base, '/' ) );
        if ( ! $base ) {
            $base = 'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1';
        }
        // Enforce HTTPS
        if ( stripos( $base, 'https://' ) !== 0 ) {
            $base = preg_replace( '#^http://#i', 'https://', $base );
        }
        if ( stripos( $base, 'https://' ) !== 0 ) {
            $base = 'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1';
        }
        return rtrim( $base, '/' );
    }

    /**
     * Credits API request helper.
     */
    private function credits_request( $method, $path, $body = null, $query = [] ) {
        $username = get_option( self::OPTION_USERNAME );
        $password = get_option( self::OPTION_PASSWORD );
        if ( empty( $username ) || empty( $password ) ) {
            return new WP_Error( 'missing_creds', __( 'Missing credentials', 'rcp-api-wp-integration' ) );
        }
        $base = $this->get_credits_api_base();
        $url  = trailingslashit( $base ) . ltrim( $path, '/' );
        if ( ! empty( $query ) ) {
            $url = add_query_arg( $query, $url );
        }
        if ( stripos( $url, 'https://' ) !== 0 ) {
            return new WP_Error( 'insecure_url', __( 'Insecure API URL', 'rcp-api-wp-integration' ) );
        }
        $args = [
            'method'  => strtoupper( $method ),
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode( $username . ':' . $password ),
                'Content-Type'  => 'application/json',
            ],
            'timeout' => apply_filters( 'rcp_api_timeout', self::API_TIMEOUT ),
            'redirection' => 3,
            'reject_unsafe_urls' => true,
        ];
        if ( null !== $body ) {
            $args['body'] = wp_json_encode( $body );
        }
        $resp = wp_remote_request( $url, $args );
        if ( is_wp_error( $resp ) ) {
            return $resp;
        }
        $code = wp_remote_retrieve_response_code( $resp );
        $json = json_decode( wp_remote_retrieve_body( $resp ), true );
        if ( $code >= 200 && $code < 300 ) {
            return is_array( $json ) ? $json : [];
        }
        $message = is_array( $json ) && isset( $json['message'] ) ? $json['message'] : __( 'Remote error', 'rcp-api-wp-integration' );
        return new WP_Error( 'remote_error', $message, [ 'status' => $code, 'response' => $json ] );
    }

    /**
     * Public helper: get credits balance from the primary site.
     *
     * @return array|WP_Error
     */
    public function get_credits_balance() {
        return $this->credits_request( 'GET', 'balance', null, [] );
    }

    /**
     * Retrieve the credit cost for a given action from the credits API.
     *
     * @param string $action_id Action identifier.
     * @return int|WP_Error Cost in credits or error response.
     */
    public function get_action_cost( $action_id = 'image_generation' ) {
        $action_id = sanitize_key( $action_id );
        $cache_key = 'rcp_ai_action_cost_' . $action_id;
        $cached    = get_transient( $cache_key );
        if ( false !== $cached ) {
            return (int) $cached;
        }

        $response = $this->credits_request( 'GET', 'action-cost', null, [ 'action_id' => $action_id ] );
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $cost = isset( $response['cost'] ) ? (int) $response['cost'] : 0;
        if ( $cost <= 0 ) {
            return new WP_Error( 'invalid_cost', __( 'Invalid action cost returned by the credits API.', 'rcp-api-wp-integration' ) );
        }

        set_transient( $cache_key, $cost, (int) apply_filters( 'rcp_ai_action_cost_cache_ttl', 10 * MINUTE_IN_SECONDS, $action_id, $cost ) );
        return $cost;
    }

    /**
     * Process a single import batch for use with the job queue.
     * This is a lightweight wrapper that fetches a page from the feed and
     * imports posts using the provided arguments, without automation logic.
     *
     * @param array $args {
     *   @type int    $page        Page number to fetch. Default 1.
     *   @type string $after       "Y-m-d H:i:s" UTC lower bound for posts.
     *   @type int    $limit       Max posts to import from this page. Default 0 (no hard cap).
     *   @type string $status      Post status to import as. Default 'draft'.
     *   @type int    $author_id   Author ID to assign. Default 0.
     * }
     * @return array {
     *   @type bool $has_more   True if more pages available.
     *   @type int  $imported   Number of posts imported.
     *   @type int  $skipped    Number of posts skipped.
     * }
     */
    public function process_import_batch( $args = [] ) {
        $defaults = [
            'page'      => 1,
            'after'     => '',
            'limit'     => 0,
            'status'    => 'draft',
            'author_id' => 0,
        ];
        $args = wp_parse_args( $args, $defaults );

        $params = [ 'per_page' => self::POSTS_PER_PAGE, 'page' => (int) $args['page'] ];
        if ( ! empty( $args['after'] ) ) {
            $params['after'] = $args['after'];
        }

        $feed = $this->get_feed( $params, true );
        if ( is_wp_error( $feed ) ) {
            return [ 'has_more' => false, 'imported' => 0, 'skipped' => 0 ];
        }

        $posts       = isset( $feed['posts'] ) ? (array) $feed['posts'] : [];
        $total_pages = isset( $feed['total_pages'] ) ? (int) $feed['total_pages'] : 1;

        $imported = 0;
        $skipped  = 0;
        $count    = 0;
        foreach ( $posts as $post ) {
            $pid = isset( $post['id'] ) ? (int) $post['id'] : 0;
            if ( ! $pid ) {
                $skipped++;
                continue;
            }

            // Skip if already imported
            $existing = get_posts( [
                'meta_key'       => '_rcp_original_post_id',
                'meta_value'     => $pid,
                'post_type'      => 'post',
                'post_status'    => 'any',
                'posts_per_page' => 1,
                'fields'         => 'ids',
            ] );
            if ( $existing ) {
                $skipped++;
                continue;
            }

            $published = $post['published'] ?? ( $post['date'] ?? '' );
            $result = $this->import_post( $pid, $args['status'], $published, (int) $args['author_id'] );
            if ( is_wp_error( $result ) ) {
                $skipped++;
            } else {
                $imported++;
            }

            $count++;
            if ( $args['limit'] && $count >= (int) $args['limit'] ) {
                break;
            }
        }

        return [
            'has_more' => ( (int) $args['page'] < $total_pages ),
            'imported' => $imported,
            'skipped'  => $skipped,
        ];
    }

    private function import_post( $post_id, $status = 'draft', $post_date = '', $author_id = 0 ) {
        $data = $this->fetch_rcp_post( $post_id, true );
        if ( is_wp_error( $data ) ) {
            return $data;
        }
        
        if ( empty( trim( $data['blog_post_title'] ?? '' ) ) || empty( trim( $data['blog_post_html'] ?? '' ) ) ) {
            return new WP_Error( 'missing_blog_data', __( 'Blog version not available', 'rcp-api-wp-integration' ) );
        }

        if ( ! $author_id && ! empty( $data['categories'] ) ) {
            $authors = get_option( self::OPTION_CATEGORY_AUTHORS, [] );
            if ( is_array( $authors ) ) {
                $authors = array_change_key_case( $authors, CASE_LOWER );
                foreach ( (array) $data['categories'] as $cat ) {
                    if ( empty( $cat['name'] ) ) {
                        continue;
                    }
                    $key = strtolower( $cat['name'] );
                    if ( ! empty( $authors[ $key ] ) ) {
                        $author_id = (int) $authors[ $key ];
                        break;
                    }
                }
            }
        }

        $post_title   = ! empty( $data['blog_post_title'] ) ? $data['blog_post_title'] : ( $data['title'] ?? '' );
        if ( is_array( $post_title ) ) {
            $post_title = $post_title['rendered'] ?? reset( $post_title );
        }

        $post_content = ! empty( $data['blog_post_html'] ) ? $data['blog_post_html'] : ( $data['content_html'] ?? ( $data['content'] ?? '' ) );
        if ( is_array( $post_content ) ) {
            $post_content = $post_content['rendered'] ?? reset( $post_content );
        }
        
        // Check content requirements before processing
        if ( ! $this->check_content_requirements( $post_title, $post_content ) ) {
            return new WP_Error( 'content_requirements', __( 'Post is missing required title or content', 'rcp-api-wp-integration' ) );
        }
        
        // Check if post is already imported to prevent duplicates
        $existing_posts = get_posts( [
            'meta_key'       => '_rcp_original_post_id',
            'meta_value'     => $post_id,
            'post_type'      => 'post',
            'post_status'    => 'any',
            'fields'         => 'ids',
            'posts_per_page' => 1,
        ] );
        
        if ( ! empty( $existing_posts ) ) {
            $existing_id = $existing_posts[0];
            $edit_link = get_edit_post_link( $existing_id, 'raw' );
            $view_link = get_permalink( $existing_id );
            $view_label = get_post_status( $existing_id ) === 'publish' ? __( 'View Post', 'rcp-api-wp-integration' ) : __( 'Preview Post', 'rcp-api-wp-integration' );
            
            return [
                'id'         => $existing_id,
                'edit_link'  => $edit_link,
                'view_link'  => $view_link,
                'view_label' => $view_label,
                'already_imported' => true,
            ];
        }
        
        $post_excerpt = '';
        if ( ! empty( $post_content ) ) {
            $post_excerpt = wp_trim_words( wp_strip_all_tags( $post_content ), 55 );
        }

        $api_prompt = '';
        if ( ! empty( $data['ai_image_generation_prompt'] ) ) {
            if ( is_array( $data['ai_image_generation_prompt'] ) ) {
                $api_prompt = trim( reset( $data['ai_image_generation_prompt'] ) );
            } else {
                $api_prompt = trim( (string) $data['ai_image_generation_prompt'] );
            }
        }

        $post_data = [
            'post_title'   => wp_kses_post( $post_title ),
            'post_content' => wp_kses_post( $post_content ),
            'post_excerpt' => wp_kses_post( $post_excerpt ),
            'post_status'  => $status,
        ];

        if ( $post_date ) {
            $normalized = $this->normalize_post_date_for_insert( $post_date );
            if ( $normalized && ! is_wp_error( $normalized ) ) {
                $post_data['post_date']     = $normalized['local'];
                $post_data['post_date_gmt'] = $normalized['gmt'];
            } else {
                $post_data['post_date'] = current_time( 'mysql' );
            }
        } else {
            $post_data['post_date'] = current_time( 'mysql' );
        }
        if ( $author_id ) {
            $post_data['post_author'] = (int) $author_id;
        }

        $new_id = wp_insert_post( $post_data, true );
        if ( is_wp_error( $new_id ) ) {
            return $new_id;
        }

        $image_mode       = $this->resolve_category_image_mode( $data['categories'] ?? [] );
        $should_queue_ai  = false;
        $attempt_api_mode = in_array( $image_mode, [ 'import', 'fallback' ], true );

        if ( ! has_post_thumbnail( $new_id ) && $attempt_api_mode ) {
            $immediate_result = $this->import_featured_image_for_rcp_post( $post_id, $new_id, $data, $image_mode );

            if ( is_wp_error( $immediate_result ) ) {
                $error_code = $immediate_result->get_error_code();
                $this->debug_log( 'Immediate featured image import failed', [
                    'rcp_post_id' => $post_id,
                    'wp_post_id'  => $new_id,
                    'code'        => $error_code,
                    'message'     => $immediate_result->get_error_message(),
                ] );

                if ( 'no_image' === $error_code && 'fallback' === $image_mode ) {
                    $should_queue_ai = true;
                } elseif ( 'no_image' !== $error_code ) {
                    $this->debug_log( 'Queueing featured image retry job', [ 'rcp_post_id' => $post_id, 'wp_post_id' => $new_id ] );
                    RCP_API_Job_Queue::schedule_async( 'process_images', [
                        'post_id'     => $new_id,
                        'rcp_post_id' => $post_id,
                        'image_mode'  => $image_mode,
                    ] );
                }
            }
        }

        $prompt_override = '';
        if ( '' !== $api_prompt ) {
            update_post_meta( $new_id, '_rcp_ai_prompt', $api_prompt );
        }

        if ( ! has_post_thumbnail( $new_id ) && 'ai' === $image_mode ) {
            $should_queue_ai = true;
            $prompt_override = $api_prompt ? $api_prompt : $this->get_ai_prompt( $post_id, $new_id );
            $pending_key = 'rcp_ai_gen_pending_' . $new_id;
            if ( ! get_transient( $pending_key ) ) {
                set_transient( $pending_key, 1, 10 * MINUTE_IN_SECONDS );
                $ai_result = $this->generate_ai_image_for_post( $new_id, $post_id, $prompt_override );
                if ( is_wp_error( $ai_result ) ) {
                    $this->record_ai_image_log( [
                        'time'        => current_time( 'timestamp', true ),
                        'post_id'     => $new_id,
                        'rcp_post_id' => $post_id,
                        'status'      => 'error',
                        'message'     => $ai_result->get_error_message(),
                    ] );
                    delete_transient( $pending_key );
                }
            }
        } elseif ( $should_queue_ai && ! has_post_thumbnail( $new_id ) ) {
            if ( '' === $prompt_override ) {
                $prompt_override = $api_prompt ? $api_prompt : $this->get_ai_prompt( $post_id, $new_id );
            }
            $pending_key = 'rcp_ai_gen_pending_' . $new_id;
            if ( ! get_transient( $pending_key ) ) {
                set_transient( $pending_key, 1, 10 * MINUTE_IN_SECONDS );
                RCP_API_Job_Queue::schedule_single( 'ai_generate', [
                    'post_id'     => $new_id,
                    'rcp_post_id' => $post_id,
                    'prompt'      => $prompt_override,
                ], time() + 60 );
            }
        }

        if ( ! empty( $data['categories'] ) ) {
            $mapping = get_option( self::OPTION_CATEGORY_MAPPING, [] );
            if ( ! is_array( $mapping ) ) {
                $mapping = [];
            }
            $map     = [];
            foreach ( (array) $mapping as $row ) {
                if ( empty( $row['api'] ) || empty( $row['wp'] ) ) {
                    continue;
                }
                $map[ strtolower( trim( $row['api'] ) ) ] = (int) $row['wp'];
            }

            $cat_ids = [];
            foreach ( (array) $data['categories'] as $cat ) {
                if ( empty( $cat['name'] ) ) {
                    continue;
                }

                $name = $cat['name'];
                $key  = strtolower( trim( $name ) );
                if ( isset( $map[ $key ] ) ) {
                    $cat_ids[] = $map[ $key ];
                    continue;
                }

                $term = term_exists( $name, 'category' );
                if ( ! $term ) {
                    $term = wp_insert_term( $name, 'category' );
                }
                if ( ! is_wp_error( $term ) ) {
                    $cat_ids[] = intval( $term['term_id'] );
                }
            }
            $cat_ids = array_unique( array_filter( $cat_ids ) );
            if ( $cat_ids ) {
                wp_set_post_categories( $new_id, $cat_ids );
            }
        }

        add_post_meta( $new_id, '_rcp_original_post_id', $post_id, true );
        update_post_meta( $new_id, '_rcp_featured_image_mode', sanitize_key( $image_mode ) );

        $prompt_meta_key = '_rcp_ai_prompt';
        $api_prompt      = $data['ai_image_generation_prompt'] ?? '';
        $api_prompt      = is_string( $api_prompt ) ? trim( wp_strip_all_tags( $api_prompt ) ) : '';
        if ( '' !== $api_prompt ) {
            update_post_meta( $new_id, $prompt_meta_key, $api_prompt );
        } else {
            delete_post_meta( $new_id, $prompt_meta_key );
        }

        $edit_link  = get_edit_post_link( $new_id, '' );
        if ( 'publish' === $post_data['post_status'] ) {
            $view_link  = get_permalink( $new_id );
            $view_label = __( 'View Post', 'rcp-api-wp-integration' );
        } else {
            $view_link  = get_preview_post_link( $new_id );
            $view_label = __( 'View Preview', 'rcp-api-wp-integration' );
        }

        return [
            'new_id'     => $new_id,
            'edit_link'  => $edit_link,
            'view_link'  => $view_link,
            'view_label' => $view_label,
        ];
    }

    public function handle_auto_import( $cutoff = '', $limit_initial = 0, $force_import = false ) {
        // Check for concurrent runs (unless forced)
        if ( ! $force_import ) {
            $lock_key = 'rcp_import_running';
            $lock_value = get_transient( $lock_key );
            $now = current_time( 'timestamp', true );
            
            if ( $lock_value ) {
                $lock_time = is_numeric( $lock_value ) ? (int) $lock_value : 0;
                $lock_age = $now - $lock_time;
                
                if ( $lock_age < 5 * MINUTE_IN_SECONDS ) {
                    // Import is already running
                    $this->record_import_log( [
                        'time'      => $now,
                        'published' => [],
                        'drafted'   => [],
                        'errors'    => ['Import already running (started ' . $lock_age . ' seconds ago)'],
                        'success'   => false,
                    ] );
                    return;
                }
            }
            
            // Set lock
            set_transient( $lock_key, $now, 5 * MINUTE_IN_SECONDS );
        }
        
        // Update heartbeat to track cron health
        update_option( self::OPTION_LAST_HEARTBEAT, current_time( 'timestamp', true ) );
        
        // Start with detailed logging
        $start_time = microtime( true );
        $log_debug = [];
        $log_debug[] = "=== AUTO IMPORT STARTED ===";
        $log_debug[] = "Time: " . date( 'Y-m-d H:i:s', current_time( 'timestamp', true ) ) . " UTC";
        $log_debug[] = "Force import: " . ( $force_import ? 'YES' : 'NO' );
        $log_debug[] = "Cutoff: " . ( $cutoff ?: 'none' );
        $log_debug[] = "Limit: " . ( $limit_initial ?: 'none' );
        
        // Check for stuck imports (safety mechanism)
        if ( ! $force_import ) {
            $last_success = get_option( self::OPTION_LAST_SUCCESS, 0 );
            $last_import = get_option( self::OPTION_LAST_IMPORT, 0 );
            $now = current_time( 'timestamp', true );
            
            // If last 5 imports failed and it's been over 2 hours since success
            if ( $last_success && $last_import && $last_success < $last_import && 
                 $now - $last_success > 2 * HOUR_IN_SECONDS ) {
                
                $log_debug[] = "SAFETY: Detected stuck imports - adjusting cutoffs";
                
                // Adjust cutoffs to be more permissive
                $cutoffs = get_option( self::OPTION_CATEGORY_CUTOFFS, [] );
                if ( is_array( $cutoffs ) && ! empty( $cutoffs ) ) {
                    foreach ( $cutoffs as $cat => &$time ) {
                        // Move cutoff back by 1 hour
                        $time = $time - HOUR_IN_SECONDS;
                    }
                    update_option( self::OPTION_CATEGORY_CUTOFFS, $cutoffs );
                    $log_debug[] = "SAFETY: Moved all cutoffs back by 1 hour";
                }
            }
        }
        
        $this->clear_feed_cache();
        $automation = get_option( self::OPTION_CATEGORY_AUTOMATION, [] );
        $authors    = get_option( self::OPTION_CATEGORY_AUTHORS, [] );

        if ( ! is_array( $automation ) ) { $automation = []; }
        if ( ! is_array( $authors ) ) { $authors = []; }
        
        // Validate automation settings
        $validated_automation = [];
        foreach ( $automation as $cat => $mode ) {
            if ( is_string( $cat ) && in_array( $mode, [ 'publish', 'draft', 'none' ], true ) ) {
                $validated_automation[ $cat ] = $mode;
            }
        }
        $automation = $validated_automation;

        // Legacy options have been deprecated as of version 0.6.0
        // Users should migrate to the new category automation table
        
        // Log automation settings
        $log_debug[] = "Automation settings:";
        foreach ( $automation as $cat => $mode ) {
            $normalized = $this->normalize_category_name( $cat );
            $log_debug[] = "  - '{$cat}' (normalized: '{$normalized}'): {$mode}";
        }

        $normalized_automation_map = $this->get_normalized_automation_map( $automation, $authors );

        // Check if we have any automation settings
        if ( empty( $automation ) ) {
            $log_debug[] = "WARNING: No category automation configured";
            $log_debug[] = "Raw automation option value: " . json_encode( get_option( self::OPTION_CATEGORY_AUTOMATION, 'NOT_SET' ) );

            // Check if there are any configured categories at all
            $has_any_automation = false;
            foreach ( $automation as $cat => $mode ) {
                if ( $mode !== 'none' ) {
                    $has_any_automation = true;
                    break;
                }
            }
            
            if ( ! $has_any_automation ) {
                $log_debug[] = "INFO: No categories set to auto-publish or auto-draft. Configure automation settings to enable automatic imports.";
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( '[RCP Import] No automation configured - exiting' );
                    error_log( '[RCP Import] Automation settings: ' . json_encode( $automation ) );
                }
                
                $this->record_import_log( [
                    'time'      => current_time( 'timestamp', true ),
                    'published' => [],
                    'drafted'   => [],
                    'errors'    => [],
                    'info'      => ['No categories configured for automation. Visit Settings > Category Automation to enable automatic imports.'],
                    'success'   => true, // Not an error, just no work to do
                    'debug'     => [
                        'automation_settings' => $automation,
                        'force_import' => $force_import,
                        'cutoff' => $cutoff,
                        'limit_initial' => $limit_initial,
                        'debug_messages' => $log_debug,
                    ]
                ] );
                return; // No automation configured
            }
        }

        if ( empty( $normalized_automation_map ) ) {
            $log_debug[] = "INFO: Automation rules exist but none are set to auto-publish or auto-draft.";
            $this->record_import_log( [
                'time'      => current_time( 'timestamp', true ),
                'published' => [],
                'drafted'   => [],
                'errors'    => [],
                'info'      => ['Categories selected for automation are all set to "None". Update automation rules to import posts automatically.'],
                'success'   => true,
                'debug'     => [
                    'automation_settings' => $automation,
                    'force_import'        => $force_import,
                    'cutoff'              => $cutoff,
                    'limit_initial'       => $limit_initial,
                    'debug_messages'      => $log_debug,
                ],
            ] );
            if ( ! $force_import ) {
                delete_transient( 'rcp_import_running' );
            }
            return;
        }

        $cutoffs = get_option( self::OPTION_CATEGORY_CUTOFFS, [] );
        if ( ! is_array( $cutoffs ) ) { $cutoffs = []; }

        $normalized_cutoffs_map = $this->get_normalized_cutoffs( $cutoffs );
        
        // Use UTC timestamp consistently
        $now_utc = current_time( 'timestamp', true );
        $max_age  = (int) apply_filters( 'rcp_api_max_import_age', 36 * HOUR_IN_SECONDS );
        $limit_ts = $now_utc - $max_age;
        
        // Find the earliest cutoff time, but not older than 36 hours
        $after_ts = $cutoffs ? max( min( array_map( 'intval', $cutoffs ) ), $limit_ts ) : $limit_ts;
        
        $after  = $after_ts ? gmdate( 'Y-m-d H:i:s', $after_ts ) : '';
        $limit  = ( ! $after && $limit_initial ) ? (int) $limit_initial : 0;
        $count  = 0;
        $page   = 1;
        $log_published = [];
        $log_drafts    = [];
        $log_errors    = [];
        $imported_categories = [];
        $posts_processed = 0;
        $posts_skipped = 0;

        // Log current cutoffs
        $log_debug[] = "Current cutoffs:";
        foreach ( $cutoffs as $cat => $time ) {
            $log_debug[] = "  - '{$cat}': " . date( 'Y-m-d H:i:s', $time ) . " UTC (will import posts newer than " . date( 'Y-m-d H:i:s', $time - 60 ) . " UTC)";
        }
        $log_debug[] = "API query starting from: " . ( $after ?: 'beginning' );
        $log_debug[] = "Maximum post age: 36 hours (posts older than " . date( 'Y-m-d H:i:s', $limit_ts ) . " UTC will be skipped)";
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( '[RCP Import] Starting automation import' );
            error_log( '[RCP Import] Cutoffs: ' . json_encode( $cutoffs ) );
            error_log( '[RCP Import] After timestamp: ' . $after );
            error_log( '[RCP Import] Limit timestamp: ' . date( 'Y-m-d H:i:s', $limit_ts ) );
        }
        
        do {
            $args = [ 'per_page' => (int) apply_filters( 'rcp_api_posts_per_page', self::POSTS_PER_PAGE ), 'page' => $page ];
            if ( $after ) {
                $args['after'] = is_numeric( $after ) ? gmdate( 'Y-m-d H:i:s', (int) $after ) : $after;
            }
            $log_debug[] = "Fetching page {$page} with args: " . json_encode( $args );
            // Bypass cache for automation to ensure fresh data
            $feed = $this->get_feed( $args, true );
            if ( is_wp_error( $feed ) ) {
                $error_code = $feed->get_error_code();
                $error_msg = $feed->get_error_message();
                
                // Provide more specific error messages
                if ( $error_code === 'missing_creds' ) {
                    $error_msg = 'Authentication failed: API credentials are missing or invalid. Please check your API Connection settings.';
                } elseif ( $error_code === 'http_request_failed' ) {
                    $error_msg = 'Network error: Unable to connect to RCP API. ' . $error_msg;
                } elseif ( $error_code === 'api_error' ) {
                    $error_msg = 'API returned an error: ' . $error_msg;
                }
                
                $log_errors[] = 'Page ' . $page . ' - ' . $error_msg;
                $log_debug[] = 'API Error on page ' . $page . ' (code: ' . $error_code . '): ' . $error_msg;
                
                // For API errors, try to continue with next page after a brief delay
                // unless it's an authentication error
                if ( in_array( $error_code, [ 'missing_creds', 'api_error' ], true ) ) {
                    $log_debug[] = "Stopping import due to critical error";
                    break; // Stop completely for auth errors
                } else {
                    $log_debug[] = "Will retry next page after brief delay";
                    sleep( 1 ); // Brief delay before trying next page
                    continue; // Try next page
                }
            }
            
            // Build a lookup of already imported posts to avoid N+1 queries
            $imported_lookup = [];
            if ( isset( $feed['posts'] ) && is_array( $feed['posts'] ) ) {
                $page_ids = [];
                foreach ( $feed['posts'] as $p ) {
                    if ( isset( $p['id'] ) ) $page_ids[] = (int) $p['id'];
                }
                if ( $page_ids ) {
                    global $wpdb;
                    $placeholders = implode( ',', array_fill( 0, count( $page_ids ), '%d' ) );
                    $sql = $wpdb->prepare(
                        "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value IN ($placeholders)",
                        array_merge( [ '_rcp_original_post_id' ], $page_ids )
                    );
                    $found = $wpdb->get_col( $sql );
                    foreach ( (array) $found as $val ) {
                        $imported_lookup[ (int) $val ] = true;
                    }
                }
            }
            
            // Validate feed response structure
            if ( ! isset( $feed['posts'] ) || ! is_array( $feed['posts'] ) ) {
                $log_errors[] = 'Invalid feed response structure on page ' . $page;
                $log_debug[] = "Invalid feed structure: " . json_encode( array_keys( $feed ) );
                continue;
            }
            
            $log_debug[] = "Page {$page}: Found " . count( $feed['posts'] ) . " posts, total pages: " . ( $feed['total_pages'] ?? 'unknown' );
            
            // Filter out posts with empty blog_post_html BEFORE processing
            if ( ! empty( $feed['posts'] ) ) {
                $posts_before_filter = count( $feed['posts'] );
                $feed['posts'] = array_filter( $feed['posts'], function( $p ) {
                    $blog_title = trim( $p['blog_post_title'] ?? '' );
                    $blog_html = trim( $p['blog_post_html'] ?? '' );
                    return ! empty( $blog_title ) && ! empty( $blog_html );
                } );
                $feed['posts'] = array_values( $feed['posts'] );
                $posts_filtered = $posts_before_filter - count( $feed['posts'] );
                
                if ( $posts_filtered > 0 ) {
                    $log_debug[] = "Filtered out {$posts_filtered} posts with missing title or content";
                }
            }
            
            // Critical: Log when NO posts are found
            if ( empty( $feed['posts'] ) ) {
                $log_debug[] = "WARNING: No posts returned from API for the given time range (after content filtering)";
                $log_debug[] = "This could mean:";
                $log_debug[] = "  1. No new posts have been published since " . ( $after ?: 'the beginning' );
                $log_debug[] = "  2. The API query parameters are filtering out all posts";
                $log_debug[] = "  3. There's an issue with the API response";
                $log_debug[] = "  4. All posts were filtered out due to missing title or content";
                
                $this->debug_log( 'No posts returned from API - query after: ' . $after );
            }
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[RCP Import] Processing page ' . $page . ' with ' . count( $feed['posts'] ) . ' posts' );
            }
            
            foreach ( (array) $feed['posts'] as $post ) {
                $pid = isset( $post['id'] ) ? (int) $post['id'] : 0;
                if ( ! $pid ) {
                    $log_debug[] = "Post with no ID found - skipping";
                    continue;
                }
                $posts_processed++;
                $published = $post['published'] ?? ( $post['date'] ?? '' );
                $published_ts = $published ? strtotime( $published ) : 0;
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( "[RCP Import] Processing post {$pid}: '{$post['title']}' published at {$published}" );
                }
                
                // Skip posts older than 36 hours (using UTC)
                if ( $published_ts && $published_ts < $limit_ts ) {
                    $posts_skipped++;
                    $log_debug[] = "Post {$pid}: Skipped - too old ({$published})";
                    continue;
                }
                
                // Skip posts newer than cutoff (for partial imports)
                if ( $cutoff && $published_ts && $published_ts > strtotime( $cutoff ) ) {
                    $posts_skipped++;
                    $log_debug[] = "Post {$pid}: Skipped - newer than cutoff ({$published})";
                    continue;
                }
                
                // Check if post already exists (batched lookup)
                if ( isset( $imported_lookup[ $pid ] ) ) {
                    $posts_skipped++;
                    $log_debug[] = "Post {$pid}: Skipped - already exists";
                    continue;
                }
                
                // Check if post was previously deleted (only for automatic imports)
                if ( ! $force_import ) {
                    $deleted_posts = get_option( self::OPTION_DELETED_POSTS, [] );
                    if ( is_array( $deleted_posts ) && isset( $deleted_posts[ $pid ] ) ) {
                        $posts_skipped++;
                        $deletion_date = date( 'Y-m-d H:i:s', $deleted_posts[ $pid ] );
                        $log_debug[] = "Post {$pid}: Skipped - previously deleted on {$deletion_date}";
                        continue;
                    }
                }

                // Content requirements are already checked in the filter above
                // No need to check again here

            $names = [];
            $raw_names = [];
            if ( ! empty( $post['categories'] ) ) {
                foreach ( (array) $post['categories'] as $c ) {
                    if ( ! empty( $c['name'] ) ) {
                        $raw_names[] = $c['name'];
                        $normalized = $this->normalize_category_name( $c['name'] );
                        if ( $normalized ) {
                            $names[] = $normalized;
                        }
                    }
                }
            }

            $status  = '';
            $author  = 0;
            $matched = [];
            $match = $this->evaluate_automation_match( $names, $raw_names, $normalized_automation_map );
            $status = $match['status'];
            $matched = $match['matched'];
            $author = $match['author'];
            $category_debug = $match['debug'];

            // Log category matching results
            if ( empty( $matched ) ) {
                $posts_skipped++;
                $log_debug[] = "Post {$pid}: Skipped - no matching automation rules";
                $log_debug[] = "  Categories: " . implode(', ', $raw_names);
                $log_debug[] = "  Available automation rules: " . json_encode( array_keys( $normalized_automation_map ) );
                foreach ( $category_debug as $debug_msg ) {
                    $log_debug[] = "  " . $debug_msg;
                }
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( "[RCP Import] Post {$pid} skipped - no category match" );
                    error_log( "[RCP Import] Post categories: " . json_encode( $raw_names ) );
                    error_log( "[RCP Import] Normalized categories: " . json_encode( $names ) );
                    error_log( "[RCP Import] Available automation: " . json_encode( $normalized_automation_map ) );
                }
            } else {
                $log_debug[] = "Post {$pid}: Matched categories: " . implode(', ', $matched) . " -> status: {$status}";
                foreach ( $category_debug as $debug_msg ) {
                    $log_debug[] = "  " . $debug_msg;
                }
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( "[RCP Import] Post {$pid} matched - will import as {$status}" );
                }
            }
            if ( $status ) {
                $skip = false;
                
                // Only apply cutoff restrictions if this is a scheduled import (not forced)
                // and we have cutoff times configured
                if ( ! $force_import && ! empty( $cutoffs ) ) {
                    $skip = true;
                    $cutoff_reasons = [];
                    foreach ( $matched as $cat ) {
                        $co = isset( $normalized_cutoffs_map[ $cat ] ) ? (int) $normalized_cutoffs_map[ $cat ] : 0;
                        // Allow posts that are newer than the cutoff time for this category
                        // If no cutoff exists for this category, allow the post
                        // Add 60-second grace period to handle edge cases and timestamp variations

                        $log_debug[] = "  - Checking cutoff for category '{$cat}'";
                        $log_debug[] = "    - Post published: " . date('Y-m-d H:i:s', $published_ts) . " UTC (timestamp: {$published_ts})";
                        
                        if ( ! $co ) {
                            $skip = false;
                            $cutoff_reasons[] = "Category '{$cat}': no cutoff set - will import";
                            $log_debug[] = "    - No cutoff for this category - WILL IMPORT";
                            break;
                        } else {
                            $log_debug[] = "    - Cutoff time: " . date('Y-m-d H:i:s', $co) . " UTC (timestamp: {$co})";
                            $log_debug[] = "    - Grace cutoff: " . date('Y-m-d H:i:s', $co - 60) . " UTC (cutoff - 60s)";
                            
                            if ( ! $published_ts ) {
                                $skip = false;
                                $cutoff_reasons[] = "Category '{$cat}': no published timestamp - will import";
                                $log_debug[] = "    - No published timestamp - WILL IMPORT";
                                break;
                            } elseif ( $published_ts >= ($co - 60) ) {
                                $skip = false;
                                $time_diff = $published_ts - ($co - 60);
                                $cutoff_reasons[] = "Category '{$cat}': post newer than grace cutoff by {$time_diff} seconds - will import";
                                $log_debug[] = "    - Post is " . $time_diff . " seconds newer than grace cutoff - WILL IMPORT";
                                break;
                            } else {
                                $time_diff = ($co - 60) - $published_ts;
                                $cutoff_reasons[] = "Category '{$cat}': post older than grace cutoff by {$time_diff} seconds - skipping";
                                $log_debug[] = "    - Post is " . $time_diff . " seconds older than grace cutoff - SKIP";
                            }
                        }
                    }
                    if ( $skip ) {
                        $posts_skipped++;
                        $log_debug[] = "Post {$pid}: Skipped by cutoff logic - " . implode('; ', $cutoff_reasons);
                    } else {
                        $log_debug[] = "Post {$pid}: Allowed by cutoff logic - " . implode('; ', $cutoff_reasons);
                    }
                } else {
                    // For forced imports or when no cutoffs exist, don't skip based on cutoffs
                    $skip = false;
                    $log_debug[] = "Post {$pid}: Cutoff logic bypassed (force_import: " . ($force_import ? 'true' : 'false') . ", cutoffs_empty: " . (empty($cutoffs) ? 'true' : 'false') . ")";
                }
                
                if ( $skip ) {
                    continue;
                }
                // Validate required post data before import
                if ( ! is_array( $post ) || empty( $post['id'] ) ) {
                    $log_errors[] = 'Post data validation failed - missing required fields';
                    continue;
                }
                
                $result = $this->import_post( $pid, $status, $published, $author );
                if ( is_wp_error( $result ) ) {
                    $error_code = $result->get_error_code();
                    $error_msg = $result->get_error_message();
                    
                    // Add more context to error messages
                    if ( $error_code === 'import_failed' ) {
                        $log_errors[] = "Post {$pid}: Import failed - {$error_msg} (status: {$status}, author: {$author})";
                    } else {
                        $log_errors[] = "Post {$pid}: {$error_msg} (error code: {$error_code})";
                    }
                    $log_debug[] = "Post {$pid}: Import error - code: {$error_code}, message: {$error_msg}, status: {$status}, author: {$author}";
                } else {
                    $log_debug[] = "Post {$pid}: Successfully imported (WP ID: {$result['new_id']})";
                    if ( 'publish' === $status ) {
                        $log_published[] = $result['new_id'];
                    } else {
                        $log_drafts[] = $result['new_id'];
                    }
                    // Track which categories had posts imported + latest timestamps
                    foreach ( $matched as $cat ) {
                        if ( ! isset( $imported_categories[ $cat ] ) ) {
                            $imported_categories[ $cat ] = [
                                'count'  => 0,
                                'latest' => 0,
                            ];
                        }
                        $imported_categories[ $cat ]['count']++;
                        $latest_reference = $published_ts ?: $now_utc;
                        if ( $latest_reference > $imported_categories[ $cat ]['latest'] ) {
                            $imported_categories[ $cat ]['latest'] = $latest_reference;
                        }
                    }
                }
                $count++;
                if ( $limit && $count >= $limit ) {
                    break 2;
                }
            }
        }
            $page++;
            
            // Safety check to prevent infinite loops
            if ( $page > 50 ) {
                $log_errors[] = 'Import stopped - maximum pages limit reached (50)';
                break;
            }
        } while ( $page <= (int) $feed['total_pages'] );

        // Only update cutoffs if posts were actually imported
        if ( $count > 0 ) {
            // Normalize existing cutoffs
            $normalized_cutoffs = [];
            foreach ( $cutoffs as $cat => $time ) {
                $normalized = $this->normalize_category_name( $cat );
                if ( $normalized ) {
                    $normalized_cutoffs[ $normalized ] = $time;
                }
            }
            
            // Only update cutoffs for categories that had posts imported
            foreach ( $imported_categories as $cat => $meta ) {
                if ( empty( $meta['count'] ) ) {
                    continue;
                }

                $latest_ts = isset( $meta['latest'] ) && $meta['latest'] ? (int) $meta['latest'] : $now_utc;
                $grace     = (int) apply_filters( 'rcp_api_cutoff_grace', 2 * HOUR_IN_SECONDS, $cat, $meta );
                $grace     = max( MINUTE_IN_SECONDS, $grace );

                $proposed  = $latest_ts - $grace;
                $proposed  = max( $proposed, $limit_ts );

                // Never allow cutoff to be in the future
                if ( $proposed > $now_utc ) {
                    $proposed = $now_utc;
                }

                // Keep the most conservative (oldest) cutoff to avoid missing posts
                if ( isset( $normalized_cutoffs[ $cat ] ) ) {
                    $normalized_cutoffs[ $cat ] = min( $normalized_cutoffs[ $cat ], $proposed );
                } else {
                    $normalized_cutoffs[ $cat ] = $proposed;
                }
            }

            update_option( self::OPTION_CATEGORY_CUTOFFS, $normalized_cutoffs );
        }
        update_option( self::OPTION_LAST_IMPORT, $now_utc );
        
        // Track successful imports separately
        if ( $count > 0 || empty( $log_errors ) ) {
            update_option( self::OPTION_LAST_SUCCESS, $now_utc );
        }
        
        // Add final summary to debug log
        $execution_time = microtime( true ) - $start_time;
        $memory_peak = memory_get_peak_usage( true );
        $memory_used = memory_get_usage( true );
        
        $log_debug[] = "=== AUTO IMPORT COMPLETED ===";
        $log_debug[] = "Total execution time: " . number_format( $execution_time, 2 ) . " seconds";
        $log_debug[] = "Memory peak: " . number_format( $memory_peak / 1024 / 1024, 2 ) . " MB";
        $log_debug[] = "Memory used: " . number_format( $memory_used / 1024 / 1024, 2 ) . " MB";
        $log_debug[] = "Posts processed: {$posts_processed}";
        $log_debug[] = "Posts skipped: {$posts_skipped}";
        $log_debug[] = "Posts imported: {$count}";
        $log_debug[] = "Published: " . count( $log_published );
        $log_debug[] = "Drafted: " . count( $log_drafts );
        $log_debug[] = "Errors: " . count( $log_errors );
        
        // Calculate performance metrics
        $posts_per_second = $posts_processed > 0 && $execution_time > 0 ? 
            round( $posts_processed / $execution_time, 2 ) : 0;
        $avg_time_per_post = $posts_processed > 0 ? 
            round( $execution_time / $posts_processed, 3 ) : 0;
        
        $log_debug[] = "Performance: {$posts_per_second} posts/second, {$avg_time_per_post} seconds/post";

        $this->record_import_log( [
            'time'      => $now_utc,
            'published' => $log_published,
            'drafted'   => $log_drafts,
            'errors'    => $log_errors,
            'success'   => $count > 0 || empty( $log_errors ),
            'debug'     => [
                'import_type' => $force_import ? 'force_auto' : 'scheduled',
                'automation_settings' => $automation,
                'cutoff_settings' => $cutoffs,
                'posts_processed' => $posts_processed,
                'posts_skipped' => $posts_skipped,
                'posts_imported' => $count,
                'force_import' => $force_import,
                'after_timestamp' => $after,
                'limit_timestamp' => date('Y-m-d H:i:s', $limit_ts),
                'execution_time' => $execution_time,
                'memory_peak_mb' => round( $memory_peak / 1024 / 1024, 2 ),
                'memory_used_mb' => round( $memory_used / 1024 / 1024, 2 ),
                'posts_per_second' => $posts_per_second,
                'seconds_per_post' => $avg_time_per_post,
                'debug_messages' => $log_debug,
            ]
        ] );
        
        // Clear the import lock (unless this was a forced import)
        if ( ! $force_import ) {
            delete_transient( 'rcp_import_running' );
        }
    }

    /**
     * Run a test import to show what would be imported without actually importing.
     * 
     * @param bool $ignore_cutoffs Whether to ignore category cutoffs.
     * @return array|WP_Error Test results or error.
     */
    private function run_test_import( $ignore_cutoffs = false ) {
        $automation = get_option( self::OPTION_CATEGORY_AUTOMATION, [] );
        $authors    = get_option( self::OPTION_CATEGORY_AUTHORS, [] );

        if ( ! is_array( $automation ) ) { $automation = []; }
        if ( ! is_array( $authors ) ) { $authors = []; }

        if ( empty( $automation ) ) {
            return new WP_Error( 'no_automation', __( 'No category automation configured', 'rcp-api-wp-integration' ) );
        }

        $normalized_automation = $this->get_normalized_automation_map( $automation, $authors );
        if ( empty( $normalized_automation ) ) {
            return new WP_Error( 'no_active_rules', __( 'No active automation rules configured', 'rcp-api-wp-integration' ) );
        }

        $cutoffs_raw = $ignore_cutoffs ? [] : get_option( self::OPTION_CATEGORY_CUTOFFS, [] );
        if ( ! is_array( $cutoffs_raw ) ) { $cutoffs_raw = []; }
        $normalized_cutoffs = $ignore_cutoffs ? [] : $this->get_normalized_cutoffs( $cutoffs_raw );

        $now_utc  = current_time( 'timestamp', true );
        $max_age  = (int) apply_filters( 'rcp_api_max_import_age', 36 * HOUR_IN_SECONDS );
        $limit_ts = $now_utc - $max_age;

        $after_ts = ! empty( $normalized_cutoffs ) ? max( min( $normalized_cutoffs ), $limit_ts ) : $limit_ts;
        $after    = $after_ts ? gmdate( 'Y-m-d H:i:s', $after_ts ) : '';

        $per_page     = (int) apply_filters( 'rcp_api_posts_per_page', self::POSTS_PER_PAGE );
        $max_pages    = max( 1, (int) apply_filters( 'rcp_api_preview_max_pages', 3 ) );
        $max_results  = max( 5, (int) apply_filters( 'rcp_api_preview_max_results', 20 ) );

        $deleted_posts = get_option( self::OPTION_DELETED_POSTS, [] );
        if ( ! is_array( $deleted_posts ) ) { $deleted_posts = []; }

        $results = [
            'total_posts'  => 0,
            'would_import' => 0,
            'skipped'      => 0,
            'posts'        => [],
            'skip_reasons' => [],
        ];

        $page        = 1;
        $total_pages = 1;

        while ( $page <= $max_pages ) {
            $params = [ 'per_page' => $per_page, 'page' => $page ];
            if ( $after ) {
                $params['after'] = $after;
            }

            $feed = $this->get_feed( $params, true );
            if ( is_wp_error( $feed ) ) {
                return $feed;
            }

            if ( ! isset( $feed['posts'] ) || ! is_array( $feed['posts'] ) ) {
                return new WP_Error( 'invalid_feed', __( 'Invalid feed response', 'rcp-api-wp-integration' ) );
            }

            $posts = $feed['posts'];
            $results['total_posts'] += count( $posts );
            $total_pages = isset( $feed['total_pages'] ) ? max( 1, (int) $feed['total_pages'] ) : 1;

            if ( empty( $posts ) ) {
                if ( $page >= $total_pages ) {
                    break;
                }
                $page++;
                continue;
            }

            $page_ids = [];
            foreach ( $posts as $post ) {
                if ( isset( $post['id'] ) ) {
                    $page_ids[] = (int) $post['id'];
                }
            }

            $imported_lookup = [];
            if ( $page_ids ) {
                global $wpdb;
                $placeholders = implode( ',', array_fill( 0, count( $page_ids ), '%d' ) );
                $sql = $wpdb->prepare(
                    "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value IN ($placeholders)",
                    array_merge( [ '_rcp_original_post_id' ], $page_ids )
                );
                $rows = $wpdb->get_results( $sql, ARRAY_A );
                foreach ( (array) $rows as $row ) {
                    $imported_lookup[ (int) $row['meta_value'] ] = (int) $row['post_id'];
                }
            }

            foreach ( $posts as $post ) {
                if ( $results['would_import'] >= $max_results ) {
                    break;
                }

                $pid = isset( $post['id'] ) ? (int) $post['id'] : 0;
                if ( ! $pid ) {
                    continue;
                }

                $published    = $post['published'] ?? ( $post['date'] ?? '' );
                $published_ts = $published ? strtotime( $published ) : 0;

                if ( $published_ts && $published_ts < $limit_ts ) {
                    $results['skipped']++;
                    if ( count( $results['skip_reasons'] ) < $max_results ) {
                        $results['skip_reasons'][] = "Post {$pid}: Too old ({$published})";
                    }
                    continue;
                }

                if ( isset( $imported_lookup[ $pid ] ) ) {
                    $results['skipped']++;
                    if ( count( $results['skip_reasons'] ) < $max_results ) {
                        $results['skip_reasons'][] = "Post {$pid}: Already exists (WP ID: " . $imported_lookup[ $pid ] . ')';
                    }
                    continue;
                }

                if ( isset( $deleted_posts[ $pid ] ) ) {
                    $results['skipped']++;
                    if ( count( $results['skip_reasons'] ) < $max_results ) {
                        $deletion_date = date( 'Y-m-d H:i:s', $deleted_posts[ $pid ] );
                        $results['skip_reasons'][] = "Post {$pid}: Previously deleted on {$deletion_date}";
                    }
                    continue;
                }

                $blog_title = trim( $post['blog_post_title'] ?? '' );
                $blog_html  = trim( $post['blog_post_html'] ?? '' );

                if ( ! $this->check_content_requirements( $blog_title, $blog_html ) ) {
                    $results['skipped']++;
                    if ( count( $results['skip_reasons'] ) < $max_results ) {
                        $missing = [];
                        if ( empty( $blog_title ) ) { $missing[] = 'title'; }
                        if ( empty( $blog_html ) ) { $missing[] = 'blog_post_html'; }
                        $results['skip_reasons'][] = "Post {$pid}: Missing required " . implode( ' and ', $missing );
                    }
                    continue;
                }

                $names     = [];
                $raw_names = [];
                if ( ! empty( $post['categories'] ) ) {
                    foreach ( (array) $post['categories'] as $category ) {
                        if ( empty( $category['name'] ) ) {
                            continue;
                        }
                        $raw_names[] = $category['name'];
                        $normalized  = $this->normalize_category_name( $category['name'] );
                        if ( $normalized ) {
                            $names[] = $normalized;
                        }
                    }
                }

                $match   = $this->evaluate_automation_match( $names, $raw_names, $normalized_automation );
                $status  = $match['status'];
                $matched = $match['matched'];

                if ( empty( $matched ) || ! $status ) {
                    $results['skipped']++;
                    if ( count( $results['skip_reasons'] ) < $max_results ) {
                        $results['skip_reasons'][] = "Post {$pid}: No matching automation rules (categories: " . implode( ', ', $raw_names ) . ')';
                    }
                    continue;
                }

                if ( ! $ignore_cutoffs && ! empty( $normalized_cutoffs ) ) {
                    $skip_by_cutoff = true;
                    foreach ( $matched as $cat_name ) {
                        $co = isset( $normalized_cutoffs[ $cat_name ] ) ? (int) $normalized_cutoffs[ $cat_name ] : 0;
                        if ( ! $co || ! $published_ts || $published_ts >= ( $co - 60 ) ) {
                            $skip_by_cutoff = false;
                            break;
                        }
                    }

                    if ( $skip_by_cutoff ) {
                        $results['skipped']++;
                        if ( count( $results['skip_reasons'] ) < $max_results ) {
                            $results['skip_reasons'][] = "Post {$pid}: Blocked by cutoff time";
                        }
                        continue;
                    }
                }

                $results['would_import']++;
                if ( count( $results['posts'] ) < $max_results ) {
                    $results['posts'][] = [
                        'id'            => $pid,
                        'title'         => $blog_title,
                        'published'     => $published,
                        'categories'    => $raw_names,
                        'import_status' => $status,
                    ];
                }
            }

            if ( $results['would_import'] >= $max_results || $page >= $total_pages ) {
                break;
            }

            $page++;
        }

        return $results;
    }
    
    /**
     * Get comprehensive diagnostics data for troubleshooting.
     * 
     * @return array Diagnostics data.
     */
    private function get_diagnostics_data() {
        $next_scheduled = wp_next_scheduled( self::CRON_HOOK );
        $last_import = get_option( self::OPTION_LAST_IMPORT, 0 );
        $last_success = get_option( self::OPTION_LAST_SUCCESS, 0 );
        $last_heartbeat = get_option( self::OPTION_LAST_HEARTBEAT, 0 );
        $freq = get_option( self::OPTION_CRON_FREQUENCY, 'hourly' );
        $automation = get_option( self::OPTION_CATEGORY_AUTOMATION, [] );
        $authors = get_option( self::OPTION_CATEGORY_AUTHORS, [] );
        $cutoffs = get_option( self::OPTION_CATEGORY_CUTOFFS, [] );
        $image_modes_option = get_option( self::OPTION_CATEGORY_IMAGE_MODES, [] );
        
        // Normalize all data for consistency
        $normalized_automation = [];
        $normalized_authors = [];
        $normalized_cutoffs = [];
        $normalized_image_modes = [];
        
        foreach ( $automation as $cat => $mode ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $normalized_automation[ $cat ] = $mode;
            }
        }
        
        foreach ( $authors as $cat => $author_id ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $normalized_authors[ $cat ] = $author_id;
            }
        }
        
        foreach ( $cutoffs as $cat => $time ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $normalized_cutoffs[ $cat ] = $time;
            }
        }

        if ( is_array( $image_modes_option ) ) {
            foreach ( $image_modes_option as $cat => $mode ) {
                $normalized = $this->normalize_category_name( $cat );
                if ( $normalized ) {
                    $normalized_image_modes[ $cat ] = $mode;
                }
            }
        }
        
        // Get cron health status
        $schedules = wp_get_schedules();
        $interval = isset( $schedules[ $freq ] ) ? (int) $schedules[ $freq ]['interval'] : HOUR_IN_SECONDS;
        $now = current_time( 'timestamp', true );
        $cron_healthy = true;
        $cron_status = 'OK';
        
        if ( ! $next_scheduled ) {
            $cron_healthy = false;
            $cron_status = 'Not scheduled';
        } elseif ( $next_scheduled < $now - $interval ) {
            $cron_healthy = false;
            $cron_status = 'Overdue by ' . human_time_diff( $next_scheduled, $now );
        }
        
        // Check heartbeat health
        if ( $last_heartbeat ) {
            $heartbeat_age = $now - $last_heartbeat;
            if ( $heartbeat_age > $interval * 2 ) {
                $cron_healthy = false;
                $cron_status .= ' (Heartbeat stale: ' . human_time_diff( $last_heartbeat, $now ) . ' ago)';
            }
        }
        
        // Get recent import log entries
        $log = get_option( self::OPTION_IMPORT_LOG, [] );
        $recent_log = array_slice( $log, -10 ); // Last 10 entries
        
        // Get deleted posts count
        $deleted_posts = get_option( self::OPTION_DELETED_POSTS, [] );
        $deleted_posts_count = is_array( $deleted_posts ) ? count( $deleted_posts ) : 0;
        
        // Check import lock status
        $lock_key = 'rcp_import_running';
        $lock_value = get_transient( $lock_key );
        $import_locked = false;
        $lock_status = 'Not locked';
        
        if ( $lock_value ) {
            $import_locked = true;
            $lock_time = is_numeric( $lock_value ) ? (int) $lock_value : 0;
            $lock_age = $now - $lock_time;
            
            if ( $lock_age < 5 * MINUTE_IN_SECONDS ) {
                $lock_status = 'Import running (started ' . human_time_diff( $lock_time, $now ) . ' ago)';
            } else {
                $lock_status = 'Stale lock detected (' . human_time_diff( $lock_time, $now ) . ' old)';
            }
        }
        
        // Rate limit status
        $rl_ajax   = class_exists( 'RCP_API_Rate_Limiter' ) ? RCP_API_Rate_Limiter::get_status( 'fetch_feed', null, 'ajax' ) : [];
        $rl_import = class_exists( 'RCP_API_Rate_Limiter' ) ? RCP_API_Rate_Limiter::get_status( 'import_post', null, 'import' ) : [];

        // Cache statistics
        $cache_stats = class_exists( 'RCP_API_Cache_Manager' ) ? RCP_API_Cache_Manager::get_statistics() : [];

        return [
            'cron_frequency' => $freq,
            'cron_interval' => $interval,
            'next_scheduled' => $next_scheduled,
            'last_import' => $last_import,
            'last_success' => $last_success,
            'last_heartbeat' => $last_heartbeat,
            'cron_healthy' => $cron_healthy,
            'cron_status' => $cron_status,
            'automation_settings' => $normalized_automation,
            'category_authors' => $normalized_authors,
            'category_cutoffs' => $normalized_cutoffs,
            'image_mode_settings' => $normalized_image_modes,
            'recent_log' => array_reverse( $recent_log ),
            'api_connected' => $this->check_connection() === true,
            'deleted_posts_count' => $deleted_posts_count,
            'now_utc' => $now,
            'import_locked' => $import_locked,
            'lock_status' => $lock_status,
            'rate_limits' => [ 'ajax' => $rl_ajax, 'import' => $rl_import ],
            'cache' => $cache_stats,
        ];
    }
    
    private function record_import_log( array $entry ) {
        // Add debugging info
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'RCP API: Recording import log entry - ' . json_encode( [
                'time' => $entry['time'] ?? 'no time',
                'published_count' => count( $entry['published'] ?? [] ),
                'drafted_count' => count( $entry['drafted'] ?? [] ),
                'error_count' => count( $entry['errors'] ?? [] ),
                'source' => $entry['debug']['import_type'] ?? 'unknown'
            ] ) );
        }
        
        $log = get_option( self::OPTION_IMPORT_LOG, [] );
        if ( ! is_array( $log ) ) {
            $log = [];
        }

        // Clean up old entries
        $cutoff = current_time( 'timestamp', true ) - self::LOG_RETENTION_HOURS * HOUR_IN_SECONDS;
        $log = array_filter( $log, function( $row ) use ( $cutoff ) {
            return isset( $row['time'] ) && (int) $row['time'] >= $cutoff;
        } );

        // Add new entry
        $log[] = $entry;
        
        // Sort by time, newest first
        usort( $log, function( $a, $b ) {
            return ( $b['time'] ?? 0 ) - ( $a['time'] ?? 0 );
        } );
        
        // Delete the option first to ensure it gets updated
        delete_option( self::OPTION_IMPORT_LOG );
        
        // Force update with autoload disabled for better performance
        $updated = update_option( self::OPTION_IMPORT_LOG, array_values( $log ), false );
        
        // Verify the update
        if ( $updated || get_option( self::OPTION_IMPORT_LOG ) !== false ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'RCP API: Import log successfully updated' );
            }
            // Also write a summary row to DB history for scalability
            $summary_meta = $entry['debug'] ?? [];
            $published_count = count( (array) ( $entry['published'] ?? [] ) );
            $drafted_count   = count( (array) ( $entry['drafted'] ?? [] ) );
            $errors_count    = count( (array) ( $entry['errors'] ?? [] ) );
            $summary_meta['published_count'] = $published_count;
            $summary_meta['drafted_count']   = $drafted_count;
            $summary_meta['errors_count']    = $errors_count;
            if ( class_exists( 'RCP_API_Database_Manager' ) ) {
                RCP_API_Database_Manager::insert_import_history( [
                    'import_type' => $summary_meta['import_type'] ?? 'automation',
                    'status'      => ! empty( $entry['success'] ) ? 'success' : 'error',
                    'meta'        => $summary_meta,
                    'imported_at' => isset( $entry['time'] ) ? gmdate( 'Y-m-d H:i:s', (int) $entry['time'] ) : current_time( 'mysql' ),
                ] );
            }
            return true;
        } else {
            error_log( 'RCP API: Failed to update import log - ' . json_encode( $entry ) );
            return false;
        }
    }
    
    /**
     * Record a manual import to the log.
     * 
     * @param int    $post_id   The post ID that was imported.
     * @param string $status    The import status (publish/draft).
     * @param array  $result    The import result array.
     * @param string $source    The source of the import (manual/ajax).
     */
    private function record_manual_import( $post_id, $status, $result, $source = 'manual' ) {
        $new_id = isset( $result['new_id'] ) ? (int) $result['new_id'] : ( isset( $result['id'] ) ? (int) $result['id'] : 0 );

        $entry = [
            'time'      => current_time( 'timestamp', true ),
            'published' => ( 'publish' === $status && $new_id ) ? [ $new_id ] : [],
            'drafted'   => ( 'draft' === $status && $new_id ) ? [ $new_id ] : [],
            'errors'    => [],
            'success'   => true,
            'debug'     => [
                'import_type' => $source,
                'original_post_id' => $post_id,
                'wp_post_id' => $new_id,
                'status' => $status,
                'posts_processed' => 1,
                'posts_skipped' => 0,
                'posts_imported' => 1,
            ]
        ];
        
        $ok = $this->record_import_log( $entry );
        // Also write per-post history record for manual imports
        if ( class_exists( 'RCP_API_Database_Manager' ) ) {
            RCP_API_Database_Manager::insert_import_history( [
                'rcp_post_id' => (int) $post_id,
                'wp_post_id'  => (int) ( $result['new_id'] ?? 0 ),
                'import_type' => $source,
                'status'      => $status,
                'imported_at' => current_time( 'mysql' ),
                'meta'        => [ 'view_link' => $result['view_link'] ?? '', 'edit_link' => $result['edit_link'] ?? '' ],
            ] );
        }
        return $ok;
    }
    
    /**
     * Track when an imported post is deleted.
     * 
     * @param int $post_id The post ID being deleted.
     */
    public function track_deleted_post( $post_id ) {
        // Only track posts that were imported from RCP
        $rcp_id = get_post_meta( $post_id, '_rcp_original_post_id', true );
        if ( ! $rcp_id ) {
            return;
        }
        
        // Get the current list of deleted posts
        $deleted_posts = get_option( self::OPTION_DELETED_POSTS, [] );
        if ( ! is_array( $deleted_posts ) ) {
            $deleted_posts = [];
        }
        
        // Add this post with timestamp
        $deleted_posts[ $rcp_id ] = current_time( 'timestamp', true );
        
        // Clean up old entries (older than 90 days)
        $cutoff = current_time( 'timestamp', true ) - ( 90 * DAY_IN_SECONDS );
        $deleted_posts = array_filter( $deleted_posts, function( $time ) use ( $cutoff ) {
            return $time > $cutoff;
        });
        
        // Save the updated list
        update_option( self::OPTION_DELETED_POSTS, $deleted_posts, false );
        
        // Log the deletion for debugging
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "RCP API: Tracked deletion of imported post - RCP ID: {$rcp_id}, WP ID: {$post_id}" );
        }
    }
    
    /**
     * Check if post content meets requirements.
     * 
     * @param string $title Post title.
     * @param string $content Post content.
     * @return bool True if content meets requirements.
     */
    private function check_content_requirements( $title, $content ) {
        // Always require both title and content
        $trimmed_title = trim( (string) $title );
        $trimmed_content = trim( (string) $content );
        
        $has_title = ! empty( $trimmed_title );
        $has_content = ! empty( $trimmed_content );
        
        // Debug logging when content is rejected
        if ( ! $has_title || ! $has_content ) {
            $missing = [];
            if ( ! $has_title ) $missing[] = 'title';
            if ( ! $has_content ) $missing[] = 'content (blog_post_html)';
            
            $this->debug_log( sprintf(
                'Content requirements not met - missing: %s (title length: %d, content length: %d)',
                implode( ' and ', $missing ),
                strlen( $trimmed_title ),
                strlen( $trimmed_content )
            ) );
        }
        
        return $has_title && $has_content;
    }
    
    /**
     * Export debug log as a downloadable text file.
     */
    private function export_debug_log() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'Permission denied', 'rcp-api-wp-integration' ) );
        }
        
        $filename = 'rcp-api-debug-log-' . date( 'Y-m-d-His' ) . '.txt';
        
        header( 'Content-Type: text/plain' );
        header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
        header( 'Cache-Control: no-cache, no-store, must-revalidate' );
        header( 'Pragma: no-cache' );
        header( 'Expires: 0' );
        
        echo "RCP API WordPress Integration - Debug Log Export\n";
        echo "================================================\n";
        echo "Export Date: " . date( 'Y-m-d H:i:s' ) . " " . get_option( 'timezone_string' ) . "\n";
        echo "Plugin Version: " . self::VERSION . "\n";
        echo "WordPress Version: " . get_bloginfo( 'version' ) . "\n";
        echo "PHP Version: " . phpversion() . "\n";
        echo "Debug Mode: " . ( get_option( self::OPTION_DEBUG_MODE, false ) ? 'Enabled' : 'Disabled' ) . "\n";
        echo "WP_DEBUG: " . ( defined( 'WP_DEBUG' ) && WP_DEBUG ? 'Enabled' : 'Disabled' ) . "\n";
        echo "\n";
        
        // Current settings
        echo "Current Settings\n";
        echo "================\n";
        $automation = get_option( self::OPTION_CATEGORY_AUTOMATION, [] );
        $cutoffs = get_option( self::OPTION_CATEGORY_CUTOFFS, [] );
        
        echo "Category Automation:\n";
        foreach ( $automation as $cat => $mode ) {
            echo "  - {$cat}: {$mode}\n";
        }
        
        echo "\nCategory Cutoffs:\n";
        foreach ( $cutoffs as $cat => $time ) {
            echo "  - {$cat}: " . date( 'Y-m-d H:i:s', $time ) . " UTC\n";
        }
        
        echo "\nCron Status:\n";
        $next = wp_next_scheduled( self::CRON_HOOK );
        echo "  - Next scheduled: " . ( $next ? date( 'Y-m-d H:i:s', $next ) . " UTC" : 'Not scheduled' ) . "\n";
        echo "  - Frequency: " . get_option( self::OPTION_CRON_FREQUENCY, 'hourly' ) . "\n";
        
        // Import logs
        echo "\n\nImport Log History\n";
        echo "==================\n";
        
        $log = get_option( self::OPTION_IMPORT_LOG, [] );
        if ( ! empty( $log ) ) {
            // Show all logs, newest first
            $log = array_reverse( $log );
            
            foreach ( $log as $entry ) {
                echo "\n";
                echo "Time: " . date( 'Y-m-d H:i:s', $entry['time'] ) . " UTC\n";
                echo "Success: " . ( $entry['success'] ? 'Yes' : 'No' ) . "\n";
                echo "Published: " . count( $entry['published'] ?? [] ) . " posts\n";
                echo "Drafted: " . count( $entry['drafted'] ?? [] ) . " posts\n";
                echo "Errors: " . count( $entry['errors'] ?? [] ) . "\n";
                
                if ( ! empty( $entry['errors'] ) ) {
                    echo "Error Details:\n";
                    foreach ( $entry['errors'] as $error ) {
                        echo "  - " . $error . "\n";
                    }
                }
                
                if ( ! empty( $entry['debug'] ) ) {
                    echo "\nDebug Information:\n";
                    if ( isset( $entry['debug']['execution_time'] ) ) {
                        echo "  - Execution time: " . number_format( $entry['debug']['execution_time'], 2 ) . " seconds\n";
                    }
                    if ( isset( $entry['debug']['memory_peak_mb'] ) ) {
                        echo "  - Memory peak: " . $entry['debug']['memory_peak_mb'] . " MB\n";
                    }
                    if ( isset( $entry['debug']['posts_processed'] ) ) {
                        echo "  - Posts processed: " . $entry['debug']['posts_processed'] . "\n";
                    }
                    if ( isset( $entry['debug']['posts_skipped'] ) ) {
                        echo "  - Posts skipped: " . $entry['debug']['posts_skipped'] . "\n";
                    }
                    
                    if ( ! empty( $entry['debug']['debug_messages'] ) ) {
                        echo "\nDetailed Debug Log:\n";
                        foreach ( $entry['debug']['debug_messages'] as $msg ) {
                            echo "  " . $msg . "\n";
                        }
                    }
                }
                
                echo str_repeat( '-', 80 ) . "\n";
            }
        } else {
            echo "No import logs found.\n";
        }
        
        // WordPress debug.log tail (if accessible)
        $debug_log_path = WP_CONTENT_DIR . '/debug.log';
        if ( file_exists( $debug_log_path ) && is_readable( $debug_log_path ) ) {
            echo "\n\nRecent WordPress Debug Log Entries (RCP API)\n";
            echo "=============================================\n";
            
            $lines = file( $debug_log_path );
            $rcp_lines = [];
            
            // Get last 100 RCP-related lines
            foreach ( $lines as $line ) {
                if ( strpos( $line, '[RCP' ) !== false ) {
                    $rcp_lines[] = $line;
                    if ( count( $rcp_lines ) > 100 ) {
                        array_shift( $rcp_lines );
                    }
                }
            }
            
            if ( ! empty( $rcp_lines ) ) {
                foreach ( $rcp_lines as $line ) {
                    echo $line;
                }
            } else {
                echo "No RCP API entries found in debug.log\n";
            }
        }
    }
    
    /**
     * Check if debug logging is enabled.
     * 
     * @return bool True if debug logging should be used.
     */
    private function is_debug_enabled() {
        return get_option( self::OPTION_DEBUG_MODE, false ) && defined( 'WP_DEBUG' ) && WP_DEBUG;
    }
    
    /**
     * Log debug message if debug mode is enabled.
     * 
     * @param string $message Message to log.
     * @param mixed  $data    Optional data to log.
     */
    private function debug_log( $message, $data = null ) {
        if ( $this->is_debug_enabled() ) {
            error_log( '[RCP API] ' . $message );
            if ( $data !== null ) {
                error_log( '[RCP API] Data: ' . ( is_string( $data ) ? $data : json_encode( $data ) ) );
            }
        }
    }
    
    /**
     * Normalize category name for consistent matching.
     * 
     * @param string $name Category name to normalize.
     * @return string Normalized category name.
     */
    private function normalize_category_name( $name ) {
        if ( ! is_string( $name ) ) {
            return '';
        }

        // Decode HTML entities first
        $name = html_entity_decode( $name, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
        
        // Trim whitespace and invisible characters
        $name = trim( $name, " \t\n\r\0\x0B\xC2\xA0" );
        
        // Convert to lowercase
        $name = strtolower( $name );
        
        // Replace non-breaking spaces with regular spaces
        $name = str_replace( "\xC2\xA0", ' ', $name );
        
        // Remove multiple spaces
        $name = preg_replace( '/\s+/', ' ', $name );
        
        // Remove or replace special characters
        // Keep alphanumeric, spaces, hyphens, and underscores
        $name = preg_replace( '/[^\w\s\-]/', '', $name );
        
        // Final trim
        $name = trim( $name );
        
        return $name;
    }

    /**
     * Build a normalized automation map keyed by normalized category names.
     *
     * @param array $automation Raw automation settings.
     * @param array $authors    Optional author assignments keyed by category.
     * @return array Normalized map of category => [ mode, original, author ].
     */
    private function get_normalized_automation_map( array $automation, array $authors = [] ) {
        $map = [];

        foreach ( $automation as $cat => $mode ) {
            if ( ! in_array( $mode, [ 'publish', 'draft' ], true ) ) {
                continue;
            }

            $normalized = $this->normalize_category_name( $cat );
            if ( ! $normalized ) {
                continue;
            }

            $author = 0;
            if ( isset( $authors[ $cat ] ) ) {
                $author = (int) $authors[ $cat ];
            } else {
                $lower = strtolower( $cat );
                if ( isset( $authors[ $lower ] ) ) {
                    $author = (int) $authors[ $lower ];
                } elseif ( isset( $authors[ $normalized ] ) ) {
                    $author = (int) $authors[ $normalized ];
                }
            }

            $map[ $normalized ] = [
                'mode'     => $mode,
                'original' => $cat,
                'author'   => $author,
            ];
        }

        return $map;
    }

    /**
     * Build a normalized image mode map for per-category settings.
     *
     * @return array
     */
    private function get_normalized_image_mode_map() {
        $modes = get_option( self::OPTION_CATEGORY_IMAGE_MODES, [] );
        if ( ! is_array( $modes ) ) {
            $modes = [];
        }

        $map = [];
        foreach ( $modes as $cat => $mode ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( ! $normalized ) {
                continue;
            }

            if ( ! in_array( $mode, [ 'import', 'fallback', 'ai', 'none' ], true ) ) {
                continue;
            }

            $map[ $normalized ] = [
                'mode'     => $mode,
                'original' => $cat,
                'source'   => 'category',
            ];
        }

        // Legacy per-category checkbox support.
        $legacy = get_option( self::OPTION_FEATURED_IMAGE_CATS, [] );
        if ( is_array( $legacy ) ) {
            foreach ( $legacy as $cat ) {
                $normalized = $this->normalize_category_name( $cat );
                if ( ! $normalized || isset( $map[ $normalized ] ) ) {
                    continue;
                }
                $map[ $normalized ] = [
                    'mode'     => 'import',
                    'original' => $cat,
                    'source'   => 'legacy_checkbox',
                ];
            }
        }

        return $map;
    }

    /**
     * Resolve the featured image behaviour for a post based on its categories.
     *
     * @param array $categories Raw categories array from the API.
     * @return string One of none|import|ai|fallback.
     */
    private function resolve_category_image_mode( array $categories ) {
        $mode_map = $this->get_normalized_image_mode_map();

        $names = [];
        foreach ( (array) $categories as $cat ) {
            if ( empty( $cat['name'] ) ) {
                continue;
            }
            $normalized = $this->normalize_category_name( $cat['name'] );
            if ( $normalized ) {
                $names[] = $normalized;
            }
        }

        $priority = [
            'none'     => 1,
            'import'   => 2,
            'fallback' => 3,
            'ai'       => 4,
        ];

        $selected     = 'none';
        $selected_src = 'default';
        $selected_pri = 0;

        foreach ( $names as $name ) {
            if ( isset( $mode_map[ $name ] ) ) {
                $mode = $mode_map[ $name ]['mode'];
                $pri  = isset( $priority[ $mode ] ) ? (int) $priority[ $mode ] : 0;
                if ( $pri > $selected_pri ) {
                    $selected     = $mode;
                    $selected_pri = $pri;
                    $selected_src = $mode_map[ $name ]['source'];
                }
            }
        }

        if ( 0 === $selected_pri ) {
            // Fall back to global defaults.
            if ( get_option( self::OPTION_IMPORT_FEATURED, false ) ) {
                $selected     = 'import';
                $selected_src = 'global_import';
            } elseif ( get_option( self::OPTION_AI_AUTOGEN, false ) ) {
                $selected     = 'fallback';
                $selected_src = 'global_ai';
            }
        }

        /**
         * Filter the resolved image mode for a given post.
         *
         * @param string $selected  Resolved mode.
         * @param array  $names     Normalized category names for the post.
         * @param array  $mode_map  Normalized image mode map.
         * @param string $source    Where the selection came from.
         */
        return apply_filters( 'rcp_api_resolved_image_mode', $selected, $names, $mode_map, $selected_src );
    }

    /**
     * Normalize cutoff map keys for consistent lookups.
     *
     * @param array $cutoffs Raw cutoff map.
     * @return array Normalized cutoff map.
     */
    private function get_normalized_cutoffs( array $cutoffs ) {
        $map = [];

        foreach ( $cutoffs as $cat => $time ) {
            $normalized = $this->normalize_category_name( $cat );
            if ( $normalized ) {
                $map[ $normalized ] = (int) $time;
            }
        }

        return $map;
    }

    /**
     * Evaluate automation rules against a list of normalized category names.
     *
     * @param array $names      Normalized category names from the post.
     * @param array $raw_names  Raw category names from the post.
     * @param array $automation Normalized automation map.
     * @return array {
     *   @type string $status  Selected status (publish|draft|'' ).
     *   @type array  $matched Matched normalized category names.
     *   @type int    $author  Author ID to assign (0 = unchanged).
     *   @type array  $debug   Debug messages describing the matching process.
     * }
     */
    private function evaluate_automation_match( array $names, array $raw_names, array $automation ) {
        $status  = '';
        $matched = [];
        $author  = 0;
        $debug   = [];

        foreach ( $names as $idx => $name ) {
            $raw_label = $raw_names[ $idx ] ?? $name;
            $debug[]   = "Checking '{$raw_label}' (normalized: '{$name}')";

            if ( ! isset( $automation[ $name ] ) ) {
                $debug[] = '  - No match found';
                continue;
            }

            $rule = $automation[ $name ];
            $mode = $rule['mode'];
            $debug[] = "  - Matched! Mode: {$mode}";

            if ( 'publish' === $mode ) {
                $matched[] = $name;
                $status    = 'publish';
                $author    = (int) $rule['author'];
                break;
            }

            if ( 'draft' === $mode ) {
                $matched[] = $name;
                if ( 'publish' !== $status ) {
                    $status = 'draft';
                    $author = (int) $rule['author'];
                }
                continue;
            }

            $debug[] = '  - Mode set to none; ignoring category';
        }

        return [
            'status'  => $status,
            'matched' => $matched,
            'author'  => $author,
            'debug'   => $debug,
        ];
    }

    /**
     * Handle health check AJAX request.
     */
    public function ajax_health_check() {
        try {
            // Validate request
            RCP_API_Validator::validate_ajax_request( 'health_check', 'manage_options' );
            
            // Gather health check data
            $health = [
                'status' => 'healthy',
                'timestamp' => current_time( 'timestamp', true ),
                'checks' => [],
            ];
            
            // Check API connection
            $api_check = $this->check_connection();
            $health['checks']['api_connection'] = [
                'status' => $api_check === true ? 'pass' : 'fail',
                'message' => $api_check === true ? 'Connected' : $api_check,
            ];
            
            // Check database
            $db_stats = RCP_API_Database_Manager::get_statistics();
            $health['checks']['database'] = [
                'status' => 'pass',
                'tables' => $db_stats,
            ];
            
            // Check job queue
            $queue_status = RCP_API_Job_Queue::get_status();
            $health['checks']['job_queue'] = [
                'status' => $queue_status['available'] ? 'pass' : 'warning',
                'details' => $queue_status,
            ];
            
            // Check cron
            $next_cron = wp_next_scheduled( self::CRON_HOOK );
            $health['checks']['cron'] = [
                'status' => $next_cron ? 'pass' : 'fail',
                'next_run' => $next_cron,
            ];
            
            // Check performance
            $perf_data = RCP_API_Performance_Monitor::get_system_health();
            $health['checks']['performance'] = [
                'status' => 'pass',
                'metrics' => $perf_data,
            ];
            
            // Check for errors
            $recent_errors = RCP_API_Error_Handler::get_logs( [
                'severity' => 'critical',
                'since' => time() - HOUR_IN_SECONDS,
                'limit' => 5,
            ] );
            $health['checks']['errors'] = [
                'status' => empty( $recent_errors ) ? 'pass' : 'warning',
                'recent_critical' => count( $recent_errors ),
            ];
            
            // Overall status
            $has_failures = false;
            $has_warnings = false;
            foreach ( $health['checks'] as $check ) {
                if ( $check['status'] === 'fail' ) {
                    $has_failures = true;
                } elseif ( $check['status'] === 'warning' ) {
                    $has_warnings = true;
                }
            }
            
            if ( $has_failures ) {
                $health['status'] = 'unhealthy';
            } elseif ( $has_warnings ) {
                $health['status'] = 'degraded';
            }
            
            wp_send_json_success( $health );
            
        } catch ( Exception $e ) {
            wp_send_json_error( [
                'status' => 'error',
                'message' => $e->getMessage(),
            ], 500 );
        }
    }
    
    /**
     * Run scheduled cleanup tasks.
     */
    public function scheduled_cleanup() {
        try {
            // Clean up old rate limit data
            RCP_API_Rate_Limiter::cleanup( 2 * HOUR_IN_SECONDS );
            
            // Clean up old database records
            RCP_API_Database_Manager::cleanup_old_data( 90 );
            
            // Clean up old performance metrics
            RCP_API_Performance_Monitor::clear_metrics();
            
            // Clean up old error logs
            RCP_API_Error_Handler::clear_logs();
            
            // Clean up orphaned transients
            $this->cleanup_orphaned_transients();
            
        } catch ( Exception $e ) {
            RCP_API_Error_Handler::log( 
                'Scheduled cleanup failed: ' . $e->getMessage(), 
                'error' 
            );
        }
    }
    
    /**
     * Clean up orphaned transients.
     */
    private function cleanup_orphaned_transients() {
        global $wpdb;
        
        // Delete expired transients
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_timeout_rcp_%' 
             AND option_value < UNIX_TIMESTAMP()"
        );
        
        // Delete orphaned transient data
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_rcp_%' 
             AND option_name NOT IN (
                SELECT CONCAT('_transient_', SUBSTRING(option_name, 20)) 
                FROM (
                    SELECT option_name 
                    FROM {$wpdb->options} 
                    WHERE option_name LIKE '_transient_timeout_rcp_%'
                ) AS t
             )"
        );
    }

}

    new RCP_API_WP_Integration();

    register_activation_hook( __FILE__, [ 'RCP_API_WP_Integration', 'activate' ] );
    register_deactivation_hook( __FILE__, [ 'RCP_API_WP_Integration', 'deactivate' ] );

} // End class_exists check

// Set up private plugin updates.
require_once plugin_dir_path( __FILE__ ) . 'lib/plugin-update-checker.php';
use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

PucFactory::buildUpdateChecker(
    'https://app.radiocontentpro.com/api-plugin/plugin-update.json',
    __FILE__,
    'rcp-api-wp-integration-v2'
);
