<?php
/**
 * RCP API Input Validator
 *
 * Provides comprehensive input validation for the RCP API WordPress Integration plugin.
 *
 * @package RCP_API_WP_Integration
 * @since 0.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Input validation class for RCP API plugin.
 */
class RCP_API_Validator {
    
    /**
     * Validation rules for different input types.
     *
     * @var array
     */
    private static $rules = [
        'post_id' => [
            'type' => 'integer',
            'min' => 1,
            'max' => PHP_INT_MAX,
        ],
        'page' => [
            'type' => 'integer',
            'min' => 1,
            'max' => 1000,
        ],
        'status' => [
            'type' => 'enum',
            'values' => [ 'publish', 'draft' ],
        ],
        'post_date' => [
            'type' => 'datetime',
            'format' => 'Y-m-d H:i:s',
            'max_future' => 365 * DAY_IN_SECONDS,
            'max_past' => 5 * 365 * DAY_IN_SECONDS,
        ],
        'categories' => [
            'type' => 'array',
            'max_items' => 100,
            'item_type' => 'string',
            'item_max_length' => 200,
        ],
        'bulk_import' => [
            'type' => 'boolean',
        ],
        'total_posts' => [
            'type' => 'integer',
            'min' => 0,
            'max' => 1000,
        ],
        'nonce' => [
            'type' => 'string',
            'max_length' => 255,
            'pattern' => '/^[a-zA-Z0-9_-]+$/',
        ],
    ];
    
    /**
     * Validate input data against defined rules.
     *
     * @param string $field Field name to validate.
     * @param mixed  $value Value to validate.
     * @return mixed Sanitized value.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    public static function validate( $field, $value ) {
        if ( ! isset( self::$rules[ $field ] ) ) {
            throw new RCP_API_Validation_Exception( 
                sprintf( __( 'Unknown validation field: %s', 'rcp-api-wp-integration' ), $field ) 
            );
        }
        
        $rule = self::$rules[ $field ];
        
        switch ( $rule['type'] ) {
            case 'integer':
                return self::validate_integer( $value, $rule );
                
            case 'string':
                return self::validate_string( $value, $rule );
                
            case 'enum':
                return self::validate_enum( $value, $rule );
                
            case 'datetime':
                return self::validate_datetime( $value, $rule );
                
            case 'array':
                return self::validate_array( $value, $rule );
                
            case 'boolean':
                return self::validate_boolean( $value );
                
            default:
                throw new RCP_API_Validation_Exception( 
                    sprintf( __( 'Unknown validation type: %s', 'rcp-api-wp-integration' ), $rule['type'] ) 
                );
        }
    }
    
    /**
     * Validate integer input.
     *
     * @param mixed $value Value to validate.
     * @param array $rule  Validation rule.
     * @return int Validated integer.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    private static function validate_integer( $value, $rule ) {
        if ( ! is_numeric( $value ) ) {
            throw new RCP_API_Validation_Exception( __( 'Value must be numeric', 'rcp-api-wp-integration' ) );
        }
        
        $int_value = intval( $value );
        
        if ( isset( $rule['min'] ) && $int_value < $rule['min'] ) {
            throw new RCP_API_Validation_Exception( 
                sprintf( __( 'Value must be at least %d', 'rcp-api-wp-integration' ), $rule['min'] ) 
            );
        }
        
        if ( isset( $rule['max'] ) && $int_value > $rule['max'] ) {
            throw new RCP_API_Validation_Exception( 
                sprintf( __( 'Value must not exceed %d', 'rcp-api-wp-integration' ), $rule['max'] ) 
            );
        }
        
        return $int_value;
    }
    
    /**
     * Validate string input.
     *
     * @param mixed $value Value to validate.
     * @param array $rule  Validation rule.
     * @return string Validated string.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    private static function validate_string( $value, $rule ) {
        if ( ! is_string( $value ) ) {
            $value = strval( $value );
        }
        
        $string_value = sanitize_text_field( $value );
        
        if ( isset( $rule['max_length'] ) && strlen( $string_value ) > $rule['max_length'] ) {
            throw new RCP_API_Validation_Exception( 
                sprintf( __( 'Value must not exceed %d characters', 'rcp-api-wp-integration' ), $rule['max_length'] ) 
            );
        }
        
        if ( isset( $rule['pattern'] ) && ! preg_match( $rule['pattern'], $string_value ) ) {
            throw new RCP_API_Validation_Exception( __( 'Value contains invalid characters', 'rcp-api-wp-integration' ) );
        }
        
        return $string_value;
    }
    
    /**
     * Validate enum input.
     *
     * @param mixed $value Value to validate.
     * @param array $rule  Validation rule.
     * @return string Validated enum value.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    private static function validate_enum( $value, $rule ) {
        $string_value = sanitize_text_field( $value );
        
        if ( ! in_array( $string_value, $rule['values'], true ) ) {
            throw new RCP_API_Validation_Exception( 
                sprintf( 
                    __( 'Value must be one of: %s', 'rcp-api-wp-integration' ), 
                    implode( ', ', $rule['values'] ) 
                ) 
            );
        }
        
        return $string_value;
    }
    
    /**
     * Validate datetime input.
     *
     * @param mixed $value Value to validate.
     * @param array $rule  Validation rule.
     * @return string Validated datetime string.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    private static function validate_datetime( $value, $rule ) {
        if ( empty( $value ) ) {
            return '';
        }
        
        $string_value = sanitize_text_field( $value );
        $timestamp = strtotime( $string_value );
        
        if ( false === $timestamp ) {
            throw new RCP_API_Validation_Exception( __( 'Invalid date format', 'rcp-api-wp-integration' ) );
        }
        
        $now = current_time( 'timestamp' );
        
        if ( isset( $rule['max_future'] ) ) {
            $max_future = $now + $rule['max_future'];
            if ( $timestamp > $max_future ) {
                throw new RCP_API_Validation_Exception( 
                    __( 'Date cannot be more than 1 year in the future', 'rcp-api-wp-integration' ) 
                );
            }
        }
        
        if ( isset( $rule['max_past'] ) ) {
            $max_past = $now - $rule['max_past'];
            if ( $timestamp < $max_past ) {
                throw new RCP_API_Validation_Exception( 
                    __( 'Date cannot be more than 5 years in the past', 'rcp-api-wp-integration' ) 
                );
            }
        }
        
        return gmdate( $rule['format'], $timestamp );
    }
    
    /**
     * Validate array input.
     *
     * @param mixed $value Value to validate.
     * @param array $rule  Validation rule.
     * @return array Validated array.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    private static function validate_array( $value, $rule ) {
        if ( ! is_array( $value ) ) {
            // Try to parse comma-separated string
            if ( is_string( $value ) ) {
                $value = array_map( 'trim', explode( ',', $value ) );
            } else {
                throw new RCP_API_Validation_Exception( __( 'Value must be an array', 'rcp-api-wp-integration' ) );
            }
        }
        
        if ( isset( $rule['max_items'] ) && count( $value ) > $rule['max_items'] ) {
            throw new RCP_API_Validation_Exception( 
                sprintf( __( 'Array must not exceed %d items', 'rcp-api-wp-integration' ), $rule['max_items'] ) 
            );
        }
        
        $validated = [];
        foreach ( $value as $item ) {
            if ( isset( $rule['item_type'] ) && $rule['item_type'] === 'string' ) {
                $item = sanitize_text_field( $item );
                
                if ( isset( $rule['item_max_length'] ) && strlen( $item ) > $rule['item_max_length'] ) {
                    throw new RCP_API_Validation_Exception( 
                        sprintf( __( 'Array items must not exceed %d characters', 'rcp-api-wp-integration' ), $rule['item_max_length'] ) 
                    );
                }
            }
            $validated[] = $item;
        }
        
        return $validated;
    }
    
    /**
     * Validate boolean input.
     *
     * @param mixed $value Value to validate.
     * @return bool Validated boolean.
     */
    private static function validate_boolean( $value ) {
        return filter_var( $value, FILTER_VALIDATE_BOOLEAN );
    }
    
    /**
     * Validate AJAX request with nonce and capabilities.
     *
     * @param string $action     AJAX action name.
     * @param string $capability Required capability.
     * @return bool True if valid.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    public static function validate_ajax_request( $action, $capability ) {
        // Check nonce
        if ( ! isset( $_REQUEST['nonce'] ) ) {
            throw new RCP_API_Validation_Exception( __( 'Security nonce missing', 'rcp-api-wp-integration' ), 403 );
        }
        
        $nonce = self::validate( 'nonce', $_REQUEST['nonce'] );
        
        if ( ! wp_verify_nonce( $nonce, 'rcp_feed_nonce' ) ) {
            throw new RCP_API_Validation_Exception( __( 'Security check failed', 'rcp-api-wp-integration' ), 403 );
        }
        
        // Check capabilities
        if ( ! current_user_can( $capability ) ) {
            throw new RCP_API_Validation_Exception( __( 'Permission denied', 'rcp-api-wp-integration' ), 403 );
        }
        
        return true;
    }
    
    /**
     * Determine whether a host matches one of the allowed hosts.
     * Accepts exact matches or subdomains (e.g. cdn.example.com matches example.com).
     *
     * @param string $host    Parsed host from the URL.
     * @param string $allowed Allowed host pattern.
     * @return bool
     */
    private static function host_matches_allowed( $host, $allowed ) {
        if ( '' === $host || '' === $allowed ) {
            return false;
        }

        if ( $host === $allowed ) {
            return true;
        }

        $suffix = '.' . $allowed;
        return strlen( $host ) > strlen( $suffix ) && substr( $host, -strlen( $suffix ) ) === $suffix;
    }

    /**
     * Sanitize and validate URL for downloads.
     *
     * @param string          $url          URL to validate.
     * @param string|string[] $allowed_host Allowed hostname(s) (optional).
     * @return string Validated URL.
     * @throws RCP_API_Validation_Exception If validation fails.
     */
    public static function validate_url( $url, $allowed_host = '' ) {
        if ( empty( $url ) || ! is_string( $url ) ) {
            throw new RCP_API_Validation_Exception( __( 'Invalid URL provided', 'rcp-api-wp-integration' ) );
        }
        
        $url = esc_url_raw( $url );
        
        if ( ! wp_http_validate_url( $url ) ) {
            throw new RCP_API_Validation_Exception( __( 'URL failed validation', 'rcp-api-wp-integration' ) );
        }
        
        $parsed = wp_parse_url( $url );
        
        if ( ! $parsed || empty( $parsed['scheme'] ) || empty( $parsed['host'] ) ) {
            throw new RCP_API_Validation_Exception( __( 'Invalid URL format', 'rcp-api-wp-integration' ) );
        }
        
        // Check scheme
        if ( ! in_array( $parsed['scheme'], [ 'http', 'https' ], true ) ) {
            throw new RCP_API_Validation_Exception( __( 'Invalid URL scheme', 'rcp-api-wp-integration' ) );
        }
        
        // Check allowed hosts if specified (supports strict matches and subdomains)
        if ( ! empty( $allowed_host ) ) {
            $allowed_hosts = array_filter( array_map( 'strtolower', (array) $allowed_host ) );
            $host          = strtolower( $parsed['host'] );
            $matches       = false;

            foreach ( $allowed_hosts as $allowed ) {
                if ( self::host_matches_allowed( $host, $allowed ) ) {
                    $matches = true;
                    break;
                }
            }

            if ( ! $matches ) {
                throw new RCP_API_Validation_Exception(
                    sprintf(
                        __( 'URL must be from an approved domain (%s)', 'rcp-api-wp-integration' ),
                        implode( ', ', $allowed_hosts )
                    )
                );
            }
        }

        // Additional SSRF protection - check for private IPs
        $ip = gethostbyname( $parsed['host'] );
        if ( $ip !== $parsed['host'] ) {
            if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) === false ) {
                throw new RCP_API_Validation_Exception( __( 'URL points to private or reserved IP range', 'rcp-api-wp-integration' ) );
            }
        }
        
        return $url;
    }
}

/**
 * Custom validation exception class.
 */
class RCP_API_Validation_Exception extends Exception {
    
    /**
     * HTTP status code.
     *
     * @var int
     */
    protected $http_status_code = 400;
    
    /**
     * Constructor.
     *
     * @param string $message    Error message.
     * @param int    $code       HTTP status code.
     * @param Exception $previous Previous exception.
     */
    public function __construct( $message = '', $code = 400, Exception $previous = null ) {
        $this->http_status_code = $code;
        parent::__construct( $message, $code, $previous );
    }
    
    /**
     * Get HTTP status code.
     *
     * @return int
     */
    public function get_http_status_code() {
        return $this->http_status_code;
    }
}
