<?php
/**
 * RCP API Rate Limiter
 *
 * Provides enhanced rate limiting functionality for the RCP API WordPress Integration plugin.
 *
 * @package RCP_API_WP_Integration
 * @since 0.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Rate limiting class for RCP API plugin.
 */
class RCP_API_Rate_Limiter {
    
    /**
     * Default rate limit configurations.
     *
     * @var array
     */
    private static $default_limits = [
        'global' => [
            'requests' => 60,
            'window' => 60, // seconds
        ],
        'ajax' => [
            'requests' => 30,
            'window' => 60,
        ],
        'import' => [
            'requests' => 10,
            'window' => 300, // 5 minutes
        ],
        'api' => [
            'requests' => 100,
            'window' => 300,
        ],
    ];
    
    /**
     * Check if an action is rate limited.
     *
     * @param string $action     Action to check.
     * @param string $identifier Unique identifier (user ID, IP, etc.).
     * @param string $type       Rate limit type (global, ajax, import, api).
     * @return bool|WP_Error True if allowed, WP_Error if rate limited.
     */
    public static function check( $action, $identifier = null, $type = 'global' ) {
        if ( null === $identifier ) {
            $identifier = self::get_identifier();
        }
        
        $limits = self::get_limits( $type );
        $key = self::get_cache_key( $action, $identifier, $type );
        
        // Get current request data
        $data = get_transient( $key );
        if ( false === $data ) {
            $data = [
                'count' => 0,
                'window_start' => time(),
                'requests' => [],
            ];
        }
        
        // Clean old requests outside current window
        $current_time = time();
        $window_start = $current_time - $limits['window'];
        $data['requests'] = array_filter( $data['requests'], function( $timestamp ) use ( $window_start ) {
            return $timestamp >= $window_start;
        } );
        
        // Check if rate limit exceeded
        if ( count( $data['requests'] ) >= $limits['requests'] ) {
            $retry_after = min( $data['requests'] ) + $limits['window'] - $current_time;
            
            return new WP_Error( 
                'rate_limit_exceeded', 
                sprintf( 
                    __( 'Rate limit exceeded. Please try again in %d seconds.', 'rcp-api-wp-integration' ), 
                    $retry_after 
                ),
                [ 'retry_after' => $retry_after ]
            );
        }
        
        // Add current request
        $data['requests'][] = $current_time;
        $data['count'] = count( $data['requests'] );
        
        // Store updated data
        set_transient( $key, $data, $limits['window'] );
        
        return true;
    }
    
    /**
     * Increment rate limit counter.
     *
     * @param string $action     Action to increment.
     * @param string $identifier Unique identifier.
     * @param string $type       Rate limit type.
     * @return bool True on success.
     */
    public static function increment( $action, $identifier = null, $type = 'global' ) {
        // Check is already incrementing, just verify it's not exceeded
        return ! is_wp_error( self::check( $action, $identifier, $type ) );
    }
    
    /**
     * Reset rate limit for an action.
     *
     * @param string $action     Action to reset.
     * @param string $identifier Unique identifier.
     * @param string $type       Rate limit type.
     * @return bool True on success.
     */
    public static function reset( $action, $identifier = null, $type = 'global' ) {
        if ( null === $identifier ) {
            $identifier = self::get_identifier();
        }
        
        $key = self::get_cache_key( $action, $identifier, $type );
        delete_transient( $key );
        
        return true;
    }
    
    /**
     * Get remaining requests for an action.
     *
     * @param string $action     Action to check.
     * @param string $identifier Unique identifier.
     * @param string $type       Rate limit type.
     * @return array Rate limit status.
     */
    public static function get_status( $action, $identifier = null, $type = 'global' ) {
        if ( null === $identifier ) {
            $identifier = self::get_identifier();
        }
        
        $limits = self::get_limits( $type );
        $key = self::get_cache_key( $action, $identifier, $type );
        
        $data = get_transient( $key );
        if ( false === $data ) {
            return [
                'limit' => $limits['requests'],
                'remaining' => $limits['requests'],
                'reset' => time() + $limits['window'],
                'retry_after' => null,
            ];
        }
        
        // Clean old requests
        $current_time = time();
        $window_start = $current_time - $limits['window'];
        $data['requests'] = array_filter( $data['requests'], function( $timestamp ) use ( $window_start ) {
            return $timestamp >= $window_start;
        } );
        
        $used = count( $data['requests'] );
        $remaining = max( 0, $limits['requests'] - $used );
        $reset = $remaining > 0 ? time() + $limits['window'] : min( $data['requests'] ) + $limits['window'];
        
        return [
            'limit' => $limits['requests'],
            'remaining' => $remaining,
            'reset' => $reset,
            'retry_after' => $remaining === 0 ? $reset - $current_time : null,
        ];
    }
    
    /**
     * Get rate limit configuration.
     *
     * @param string $type Rate limit type.
     * @return array Rate limit configuration.
     */
    private static function get_limits( $type ) {
        $limits = self::$default_limits[ $type ] ?? self::$default_limits['global'];
        
        // Allow filtering of rate limits
        $limits = apply_filters( 'rcp_api_rate_limits', $limits, $type );
        $limits = apply_filters( "rcp_api_rate_limit_{$type}", $limits );
        
        // Ensure valid structure
        $limits = wp_parse_args( $limits, [
            'requests' => 60,
            'window' => 60,
        ] );
        
        return $limits;
    }
    
    /**
     * Get cache key for rate limiting.
     *
     * @param string $action     Action name.
     * @param string $identifier Unique identifier.
     * @param string $type       Rate limit type.
     * @return string Cache key.
     */
    private static function get_cache_key( $action, $identifier, $type ) {
        return 'rcp_rate_limit_' . md5( "{$type}_{$action}_{$identifier}" );
    }
    
    /**
     * Get identifier for rate limiting.
     *
     * @return string Identifier (user ID or IP address).
     */
    private static function get_identifier() {
        $user_id = get_current_user_id();
        
        if ( $user_id > 0 ) {
            return 'user_' . $user_id;
        }
        
        // Fall back to IP address for non-logged-in users
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        
        // Handle proxies
        if ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
            $ips = explode( ',', $_SERVER['HTTP_X_FORWARDED_FOR'] );
            $ip = trim( $ips[0] );
        } elseif ( ! empty( $_SERVER['HTTP_X_REAL_IP'] ) ) {
            $ip = $_SERVER['HTTP_X_REAL_IP'];
        }
        
        return 'ip_' . $ip;
    }
    
    /**
     * Add rate limit headers to response.
     *
     * @param string $action     Action name.
     * @param string $identifier Unique identifier.
     * @param string $type       Rate limit type.
     */
    public static function add_headers( $action, $identifier = null, $type = 'global' ) {
        $status = self::get_status( $action, $identifier, $type );
        
        header( 'X-RateLimit-Limit: ' . $status['limit'] );
        header( 'X-RateLimit-Remaining: ' . $status['remaining'] );
        header( 'X-RateLimit-Reset: ' . $status['reset'] );
        
        if ( $status['retry_after'] !== null ) {
            header( 'Retry-After: ' . $status['retry_after'] );
        }
    }
    
    /**
     * Clean up expired rate limit data.
     *
     * @param int $older_than Clean entries older than this many seconds.
     * @return int Number of entries cleaned.
     */
    public static function cleanup( $older_than = 3600 ) {
        global $wpdb;
        
        $older_than_time = time() - $older_than;
        
        // Properly escape table name
        $table = esc_sql( $wpdb->options );
        
        // Clean transients older than specified time
        $like_pattern = $wpdb->esc_like( '_transient_timeout_rcp_rate_limit_' ) . '%';
        $count = $wpdb->query( $wpdb->prepare(
            "DELETE FROM `$table` 
             WHERE option_name LIKE %s 
             AND option_value < %s",
            $like_pattern,
            $older_than_time
        ) );
        
        // Also delete the corresponding transient data
        $data_like_pattern = $wpdb->esc_like( '_transient_rcp_rate_limit_' ) . '%';
        $timeout_like_pattern = $wpdb->esc_like( '_transient_timeout_rcp_rate_limit_' ) . '%';
        
        $wpdb->query( $wpdb->prepare(
            "DELETE FROM `$table` 
             WHERE option_name LIKE %s 
             AND option_name NOT IN (
                SELECT CONCAT('_transient_', SUBSTRING(option_name, 20)) 
                FROM (
                    SELECT option_name FROM `$table` WHERE option_name LIKE %s
                ) AS t
             )",
            $data_like_pattern,
            $timeout_like_pattern
        ) );
        
        return $count;
    }
    
    /**
     * Check if IP is whitelisted from rate limiting.
     *
     * @param string $ip IP address to check.
     * @return bool True if whitelisted.
     */
    public static function is_whitelisted( $ip = null ) {
        if ( null === $ip ) {
            $identifier = self::get_identifier();
            if ( strpos( $identifier, 'ip_' ) === 0 ) {
                $ip = substr( $identifier, 3 );
            }
        }
        
        // Allow filtering of whitelisted IPs
        $whitelisted_ips = apply_filters( 'rcp_api_rate_limit_whitelist', [] );
        
        if ( in_array( $ip, $whitelisted_ips, true ) ) {
            return true;
        }
        
        // Check for IP ranges
        foreach ( $whitelisted_ips as $whitelist_entry ) {
            if ( strpos( $whitelist_entry, '/' ) !== false ) {
                // CIDR notation
                list( $subnet, $mask ) = explode( '/', $whitelist_entry );
                if ( self::ip_in_range( $ip, $subnet, $mask ) ) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Check if IP is in CIDR range.
     *
     * @param string $ip     IP to check.
     * @param string $subnet Subnet address.
     * @param int    $mask   Subnet mask.
     * @return bool True if in range.
     */
    private static function ip_in_range( $ip, $subnet, $mask ) {
        $subnet = ip2long( $subnet );
        $ip = ip2long( $ip );
        $mask = -1 << ( 32 - $mask );
        $subnet &= $mask;
        
        return ( $ip & $mask ) == $subnet;
    }
}