<?php
/**
 * RCP API Performance Monitor
 *
 * Provides performance monitoring and metrics for the RCP API WordPress Integration plugin.
 *
 * @package RCP_API_WP_Integration
 * @since 0.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Performance monitoring class for RCP API plugin.
 */
class RCP_API_Performance_Monitor {
    
    /**
     * Performance metrics storage.
     *
     * @var array
     */
    private static $metrics = [];
    
    /**
     * Start time for current operation.
     *
     * @var float
     */
    private static $start_time;
    
    /**
     * Memory usage at start.
     *
     * @var int
     */
    private static $start_memory;
    
    /**
     * Active timers.
     *
     * @var array
     */
    private static $timers = [];
    
    /**
     * Start monitoring an operation.
     *
     * @param string $operation Operation name.
     */
    public static function start( $operation = 'default' ) {
        self::$timers[ $operation ] = [
            'start_time' => microtime( true ),
            'start_memory' => memory_get_usage(),
            'queries_start' => get_num_queries(),
        ];
    }
    
    /**
     * End monitoring an operation.
     *
     * @param string $operation Operation name.
     * @param array  $metadata  Additional metadata to store.
     * @return array Performance metrics.
     */
    public static function end( $operation = 'default', $metadata = [] ) {
        if ( ! isset( self::$timers[ $operation ] ) ) {
            return [];
        }
        
        $timer = self::$timers[ $operation ];
        $end_time = microtime( true );
        $end_memory = memory_get_usage();
        $queries_end = get_num_queries();
        
        $metrics = [
            'operation' => $operation,
            'duration' => round( ( $end_time - $timer['start_time'] ) * 1000, 2 ), // milliseconds
            'memory_used' => $end_memory - $timer['start_memory'],
            'memory_peak' => memory_get_peak_usage(),
            'queries' => $queries_end - $timer['queries_start'],
            'timestamp' => current_time( 'timestamp', true ),
            'metadata' => $metadata,
        ];
        
        // Store metrics
        self::store_metrics( $operation, $metrics );
        
        // Clean up timer
        unset( self::$timers[ $operation ] );
        
        return $metrics;
    }
    
    /**
     * Record a specific metric.
     *
     * @param string $metric_name Metric name.
     * @param mixed  $value      Metric value.
     * @param string $operation  Operation context.
     */
    public static function record( $metric_name, $value, $operation = 'general' ) {
        $metrics = [
            'metric' => $metric_name,
            'value' => $value,
            'operation' => $operation,
            'timestamp' => current_time( 'timestamp', true ),
        ];
        
        self::store_metrics( $operation . '_' . $metric_name, $metrics );
    }
    
    /**
     * Store metrics data.
     *
     * @param string $key     Metrics key.
     * @param array  $metrics Metrics data.
     */
    private static function store_metrics( $key, $metrics ) {
        // Store in memory for current request
        self::$metrics[ $key ] = $metrics;
        
        // Store persistent metrics if enabled
        if ( apply_filters( 'rcp_api_store_performance_metrics', false ) ) {
            $stored_metrics = get_option( 'rcp_performance_metrics', [] );
            
            // Keep only last 100 entries per key
            if ( ! isset( $stored_metrics[ $key ] ) ) {
                $stored_metrics[ $key ] = [];
            }
            
            array_unshift( $stored_metrics[ $key ], $metrics );
            $stored_metrics[ $key ] = array_slice( $stored_metrics[ $key ], 0, 100 );
            
            update_option( 'rcp_performance_metrics', $stored_metrics, false );
        }
        
        // Log to error log if debug is enabled
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'RCP_DEBUG_PERFORMANCE' ) && RCP_DEBUG_PERFORMANCE ) {
            error_log( sprintf(
                '[RCP Performance] %s: %s',
                $key,
                json_encode( $metrics )
            ) );
        }
    }
    
    /**
     * Get performance metrics.
     *
     * @param string $operation Operation name (optional).
     * @param int    $limit     Number of entries to return.
     * @return array Performance metrics.
     */
    public static function get_metrics( $operation = null, $limit = 50 ) {
        $stored_metrics = get_option( 'rcp_performance_metrics', [] );
        
        if ( null !== $operation ) {
            return isset( $stored_metrics[ $operation ] ) 
                ? array_slice( $stored_metrics[ $operation ], 0, $limit )
                : [];
        }
        
        // Return all metrics
        $all_metrics = [];
        foreach ( $stored_metrics as $key => $metrics ) {
            foreach ( array_slice( $metrics, 0, $limit ) as $metric ) {
                $metric['key'] = $key;
                $all_metrics[] = $metric;
            }
        }
        
        // Sort by timestamp
        usort( $all_metrics, function( $a, $b ) {
            return ( $b['timestamp'] ?? 0 ) - ( $a['timestamp'] ?? 0 );
        } );
        
        return array_slice( $all_metrics, 0, $limit );
    }
    
    /**
     * Get average metrics for an operation.
     *
     * @param string $operation Operation name.
     * @param int    $samples   Number of samples to average.
     * @return array Average metrics.
     */
    public static function get_averages( $operation, $samples = 10 ) {
        $metrics = self::get_metrics( $operation, $samples );
        
        if ( empty( $metrics ) ) {
            return [];
        }
        
        $totals = [
            'duration' => 0,
            'memory_used' => 0,
            'queries' => 0,
            'count' => 0,
        ];
        
        foreach ( $metrics as $metric ) {
            if ( isset( $metric['duration'] ) ) {
                $totals['duration'] += $metric['duration'];
                $totals['count']++;
            }
            if ( isset( $metric['memory_used'] ) ) {
                $totals['memory_used'] += $metric['memory_used'];
            }
            if ( isset( $metric['queries'] ) ) {
                $totals['queries'] += $metric['queries'];
            }
        }
        
        if ( $totals['count'] === 0 ) {
            return [];
        }
        
        return [
            'avg_duration' => round( $totals['duration'] / $totals['count'], 2 ),
            'avg_memory' => round( $totals['memory_used'] / $totals['count'] ),
            'avg_queries' => round( $totals['queries'] / $totals['count'], 1 ),
            'samples' => $totals['count'],
        ];
    }
    
    /**
     * Monitor database queries.
     *
     * @param callable $callback Function to monitor.
     * @param string   $label    Query label.
     * @return mixed Callback result.
     */
    public static function monitor_queries( $callback, $label = 'Query' ) {
        global $wpdb;
        
        $start_queries = get_num_queries();
        $queries_before = $wpdb->queries ?? [];
        
        $result = call_user_func( $callback );
        
        $end_queries = get_num_queries();
        $queries_after = $wpdb->queries ?? [];
        
        $new_queries = array_slice( $queries_after, count( $queries_before ) );
        
        if ( $end_queries > $start_queries ) {
            self::record( 'database_queries', [
                'count' => $end_queries - $start_queries,
                'queries' => $new_queries,
            ], $label );
        }
        
        return $result;
    }
    
    /**
     * Get system health metrics.
     *
     * @return array System health data.
     */
    public static function get_system_health() {
        global $wpdb;
        
        return [
            'php' => [
                'version' => phpversion(),
                'memory_limit' => ini_get( 'memory_limit' ),
                'max_execution_time' => ini_get( 'max_execution_time' ),
                'current_memory' => size_format( memory_get_usage() ),
                'peak_memory' => size_format( memory_get_peak_usage() ),
            ],
            'wordpress' => [
                'version' => get_bloginfo( 'version' ),
                'debug' => defined( 'WP_DEBUG' ) && WP_DEBUG,
                'cache' => wp_using_ext_object_cache(),
                'multisite' => is_multisite(),
            ],
            'database' => [
                'version' => $wpdb->db_version(),
                'charset' => $wpdb->charset,
                'collate' => $wpdb->collate,
                'queries' => get_num_queries(),
            ],
            'server' => [
                'software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                'php_sapi' => php_sapi_name(),
                'max_upload_size' => size_format( wp_max_upload_size() ),
            ],
        ];
    }
    
    /**
     * Generate performance report.
     *
     * @param array $operations Operations to include in report.
     * @return array Performance report.
     */
    public static function generate_report( $operations = [] ) {
        $report = [
            'generated' => current_time( 'mysql' ),
            'system' => self::get_system_health(),
            'operations' => [],
        ];
        
        if ( empty( $operations ) ) {
            // Get all available operations
            $stored_metrics = get_option( 'rcp_performance_metrics', [] );
            $operations = array_keys( $stored_metrics );
        }
        
        foreach ( $operations as $operation ) {
            $averages = self::get_averages( $operation, 50 );
            if ( ! empty( $averages ) ) {
                $report['operations'][ $operation ] = $averages;
            }
        }
        
        return $report;
    }
    
    /**
     * Clear stored metrics.
     *
     * @param string $operation Specific operation to clear (optional).
     * @return bool True on success.
     */
    public static function clear_metrics( $operation = null ) {
        if ( null === $operation ) {
            delete_option( 'rcp_performance_metrics' );
        } else {
            $stored_metrics = get_option( 'rcp_performance_metrics', [] );
            unset( $stored_metrics[ $operation ] );
            update_option( 'rcp_performance_metrics', $stored_metrics, false );
        }
        
        return true;
    }
    
    /**
     * Add performance data to debug output.
     *
     * @param array $debug_data Existing debug data.
     * @return array Enhanced debug data.
     */
    public static function add_to_debug( $debug_data ) {
        if ( ! is_array( $debug_data ) ) {
            $debug_data = [];
        }
        
        // Add current request metrics
        if ( ! empty( self::$metrics ) ) {
            $debug_data['current_request_metrics'] = self::$metrics;
        }
        
        // Add active timers
        if ( ! empty( self::$timers ) ) {
            $active_timers = [];
            foreach ( self::$timers as $operation => $timer ) {
                $active_timers[ $operation ] = [
                    'elapsed' => round( ( microtime( true ) - $timer['start_time'] ) * 1000, 2 ) . 'ms',
                    'memory' => size_format( memory_get_usage() - $timer['start_memory'] ),
                ];
            }
            $debug_data['active_operations'] = $active_timers;
        }
        
        return $debug_data;
    }
}