<?php
/**
 * RCP API Error Handler
 *
 * Provides structured error handling for the RCP API WordPress Integration plugin.
 *
 * @package RCP_API_WP_Integration
 * @since 0.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Error handling class for RCP API plugin.
 */
class RCP_API_Error_Handler {
    
    /**
     * Error log option name.
     *
     * @var string
     */
    const ERROR_LOG_OPTION = 'rcp_api_error_log';
    
    /**
     * Maximum error log entries.
     *
     * @var int
     */
    const MAX_LOG_ENTRIES = 100;
    
    /**
     * Queued critical error emails option name.
     *
     * @var string
     */
    const EMAIL_QUEUE_OPTION = 'rcp_api_error_email_queue';
    
    /**
     * Error severity levels.
     *
     * @var array
     */
    const SEVERITY_LEVELS = [
        'critical' => 1,
        'error' => 2,
        'warning' => 3,
        'notice' => 4,
        'info' => 5,
        'debug' => 6,
    ];
    
    /**
     * Initialize error handler.
     */
    public static function init() {
        // Set custom error handler for plugin errors
        set_error_handler( [ __CLASS__, 'handle_error' ], E_ALL );
        
        // Set custom exception handler
        set_exception_handler( [ __CLASS__, 'handle_exception' ] );
        
        // Register shutdown function for fatal errors
        register_shutdown_function( [ __CLASS__, 'handle_shutdown' ] );

        // Attempt to flush any queued critical error emails when WordPress is fully loaded
        if ( function_exists( 'add_action' ) ) {
            add_action( 'init', [ __CLASS__, 'flush_email_queue' ] );
        }
    }
    
    /**
     * Handle PHP errors.
     *
     * @param int    $errno   Error number.
     * @param string $errstr  Error message.
     * @param string $errfile Error file.
     * @param int    $errline Error line.
     * @return bool True to prevent default error handler.
     */
    public static function handle_error( $errno, $errstr, $errfile, $errline ) {
        // Check if error is from our plugin
        if ( ! self::is_plugin_file( $errfile ) ) {
            return false; // Let default handler process it
        }
        
        // Convert error number to severity
        $severity = self::get_severity_from_errno( $errno );
        
        // Create error context
        $context = [
            'file' => $errfile,
            'line' => $errline,
            'errno' => $errno,
            'type' => self::get_error_type( $errno ),
        ];
        
        // Log the error
        self::log( $errstr, $severity, $context );
        
        // Don't execute PHP internal error handler for our errors
        return true;
    }
    
    /**
     * Handle uncaught exceptions.
     *
     * @param Exception|Throwable $exception The exception.
     */
    public static function handle_exception( $exception ) {
        // Check if exception is from our plugin
        if ( ! self::is_plugin_file( $exception->getFile() ) ) {
            // Re-throw if not from our plugin
            throw $exception;
        }
        
        $context = [
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'type' => get_class( $exception ),
        ];
        
        self::log( $exception->getMessage(), 'critical', $context );
        
        // If this is an AJAX request, send error response
        if ( wp_doing_ajax() ) {
            $status_code = 500;
            if ( $exception instanceof RCP_API_Validation_Exception ) {
                $status_code = $exception->get_http_status_code();
            }
            
            wp_send_json_error( 
                __( 'An error occurred processing your request.', 'rcp-api-wp-integration' ), 
                $status_code 
            );
        }
    }
    
    /**
     * Handle fatal errors on shutdown.
     */
    public static function handle_shutdown() {
        $error = error_get_last();
        
        if ( null === $error ) {
            return;
        }
        
        // Check for fatal errors
        $fatal_types = [ E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR ];
        if ( ! in_array( $error['type'], $fatal_types, true ) ) {
            return;
        }
        
        // Check if error is from our plugin
        if ( ! self::is_plugin_file( $error['file'] ) ) {
            return;
        }
        
        $context = [
            'file' => $error['file'],
            'line' => $error['line'],
            'type' => self::get_error_type( $error['type'] ),
        ];
        
        self::log( $error['message'], 'critical', $context );
    }
    
    /**
     * Log an error.
     *
     * @param string $message  Error message.
     * @param string $severity Severity level.
     * @param array  $context  Additional context.
     * @return bool True on success.
     */
    public static function log( $message, $severity = 'error', $context = [] ) {
        // Validate severity
        if ( ! isset( self::SEVERITY_LEVELS[ $severity ] ) ) {
            $severity = 'error';
        }
        
        // Get current log
        $log = get_option( self::ERROR_LOG_OPTION, [] );
        if ( ! is_array( $log ) ) {
            $log = [];
        }
        
        // Create log entry
        $entry = [
            'timestamp' => current_time( 'timestamp', true ),
            'severity' => $severity,
            'message' => $message,
            'context' => $context,
            'user_id' => get_current_user_id(),
            'url' => $_SERVER['REQUEST_URI'] ?? '',
            'method' => $_SERVER['REQUEST_METHOD'] ?? '',
        ];
        
        // Add to log
        array_unshift( $log, $entry );
        
        // Trim log to maximum entries
        $log = array_slice( $log, 0, self::MAX_LOG_ENTRIES );
        
        // Save log
        update_option( self::ERROR_LOG_OPTION, $log, false );
        
        // Also log to WordPress debug log if enabled
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
            error_log( sprintf(
                '[RCP API %s] %s %s',
                strtoupper( $severity ),
                $message,
                ! empty( $context ) ? json_encode( $context ) : ''
            ) );
        }
        
        // Send notification for critical errors
        if ( 'critical' === $severity ) {
            self::notify_admin( $message, $context );
        }
        
        return true;
    }
    
    /**
     * Get error logs.
     *
     * @param array $filters Filters to apply.
     * @return array Error log entries.
     */
    public static function get_logs( $filters = [] ) {
        $log = get_option( self::ERROR_LOG_OPTION, [] );
        if ( ! is_array( $log ) ) {
            return [];
        }
        
        // Apply filters
        if ( ! empty( $filters['severity'] ) ) {
            $log = array_filter( $log, function( $entry ) use ( $filters ) {
                return $entry['severity'] === $filters['severity'];
            } );
        }
        
        if ( ! empty( $filters['since'] ) ) {
            $since = is_numeric( $filters['since'] ) ? $filters['since'] : strtotime( $filters['since'] );
            $log = array_filter( $log, function( $entry ) use ( $since ) {
                return $entry['timestamp'] >= $since;
            } );
        }
        
        if ( ! empty( $filters['limit'] ) ) {
            $log = array_slice( $log, 0, (int) $filters['limit'] );
        }
        
        return $log;
    }
    
    /**
     * Clear error logs.
     *
     * @param string $severity Clear only specific severity (optional).
     * @return bool True on success.
     */
    public static function clear_logs( $severity = '' ) {
        if ( empty( $severity ) ) {
            delete_option( self::ERROR_LOG_OPTION );
            return true;
        }
        
        $log = get_option( self::ERROR_LOG_OPTION, [] );
        if ( ! is_array( $log ) ) {
            return true;
        }
        
        $log = array_filter( $log, function( $entry ) use ( $severity ) {
            return $entry['severity'] !== $severity;
        } );
        
        update_option( self::ERROR_LOG_OPTION, array_values( $log ), false );
        return true;
    }
    
    /**
     * Check if file belongs to our plugin.
     *
     * @param string $file File path.
     * @return bool True if plugin file.
     */
    private static function is_plugin_file( $file ) {
        $plugin_dir = RCP_API_PLUGIN_DIR;
        return 0 === strpos( $file, $plugin_dir );
    }
    
    /**
     * Get severity from error number.
     *
     * @param int $errno Error number.
     * @return string Severity level.
     */
    private static function get_severity_from_errno( $errno ) {
        switch ( $errno ) {
            case E_ERROR:
            case E_PARSE:
            case E_CORE_ERROR:
            case E_COMPILE_ERROR:
            case E_USER_ERROR:
                return 'critical';
                
            case E_WARNING:
            case E_CORE_WARNING:
            case E_COMPILE_WARNING:
            case E_USER_WARNING:
                return 'warning';
                
            case E_NOTICE:
            case E_USER_NOTICE:
                return 'notice';
                
            case E_STRICT:
            case E_DEPRECATED:
            case E_USER_DEPRECATED:
                return 'info';
                
            default:
                return 'error';
        }
    }
    
    /**
     * Get error type string.
     *
     * @param int $errno Error number.
     * @return string Error type.
     */
    private static function get_error_type( $errno ) {
        $types = [
            E_ERROR => 'Fatal Error',
            E_WARNING => 'Warning',
            E_PARSE => 'Parse Error',
            E_NOTICE => 'Notice',
            E_CORE_ERROR => 'Core Error',
            E_CORE_WARNING => 'Core Warning',
            E_COMPILE_ERROR => 'Compile Error',
            E_COMPILE_WARNING => 'Compile Warning',
            E_USER_ERROR => 'User Error',
            E_USER_WARNING => 'User Warning',
            E_USER_NOTICE => 'User Notice',
            E_STRICT => 'Strict Notice',
            E_DEPRECATED => 'Deprecated',
            E_USER_DEPRECATED => 'User Deprecated',
        ];
        
        return $types[ $errno ] ?? 'Unknown Error';
    }
    
    /**
     * Notify admin of critical errors.
     *
     * @param string $message Error message.
     * @param array  $context Error context.
     */
    private static function notify_admin( $message, $context ) {
        // Check if notifications are enabled
        if ( ! apply_filters( 'rcp_api_notify_critical_errors', true ) ) {
            return;
        }
        
        // Get admin email
        $admin_email = get_option( 'admin_email' );
        if ( ! $admin_email ) {
            return;
        }
        
        // Prepare email
        $subject = sprintf(
            __( '[%s] Critical Error in RCP API Integration', 'rcp-api-wp-integration' ),
            get_bloginfo( 'name' )
        );
        
        $body = sprintf(
            __( "A critical error has occurred in the RCP API WordPress Integration plugin.\n\nError: %s\n\nDetails:\n%s\n\nTime: %s\nURL: %s", 'rcp-api-wp-integration' ),
            $message,
            print_r( $context, true ),
            current_time( 'mysql' ),
            home_url( $_SERVER['REQUEST_URI'] ?? '' )
        );
        
        // Send email (throttled to prevent spam)
        $last_sent_key = 'rcp_api_last_error_email';
        $last_sent = get_transient( $last_sent_key );
        
        if ( ! $last_sent ) {
            if ( function_exists( 'wp_mail' ) ) {
                wp_mail( $admin_email, $subject, $body );
                set_transient( $last_sent_key, time(), HOUR_IN_SECONDS );
            } else {
                // Queue the email to send later when pluggable functions are available
                $queue = get_option( self::EMAIL_QUEUE_OPTION, [] );
                if ( ! is_array( $queue ) ) { $queue = []; }
                $queue[] = [
                    'to'      => $admin_email,
                    'subject' => $subject,
                    'body'    => $body,
                    'time'    => time(),
                ];
                // Keep queue from growing unbounded
                $queue = array_slice( $queue, -20 );
                update_option( self::EMAIL_QUEUE_OPTION, $queue, false );
                // Always set throttle so we don't spam once mail becomes available
                set_transient( $last_sent_key, time(), HOUR_IN_SECONDS );
                // Log fallback
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( '[RCP API] Queued critical error email (wp_mail unavailable yet)' );
                }
            }
        }
    }

    /**
     * Send queued critical error emails if possible.
     */
    public static function flush_email_queue() {
        if ( ! function_exists( 'wp_mail' ) ) {
            return;
        }
        $queue = get_option( self::EMAIL_QUEUE_OPTION, [] );
        if ( empty( $queue ) || ! is_array( $queue ) ) {
            return;
        }
        foreach ( $queue as $email ) {
            $to      = $email['to'] ?? get_option( 'admin_email' );
            $subject = $email['subject'] ?? '[RCP API] Critical Error';
            $body    = $email['body'] ?? '';
            if ( $to && $body ) {
                wp_mail( $to, $subject, $body );
            }
        }
        delete_option( self::EMAIL_QUEUE_OPTION );
    }
    
    /**
     * Format error for display.
     *
     * @param array $entry Error log entry.
     * @return string Formatted error.
     */
    public static function format_error( $entry ) {
        $output = sprintf(
            '[%s] %s: %s',
            gmdate( 'Y-m-d H:i:s', $entry['timestamp'] ),
            strtoupper( $entry['severity'] ),
            $entry['message']
        );
        
        if ( ! empty( $entry['context']['file'] ) ) {
            $output .= sprintf(
                ' in %s on line %d',
                $entry['context']['file'],
                $entry['context']['line'] ?? 0
            );
        }
        
        return $output;
    }
}
