<?php
/**
 * RCP API Database Manager
 *
 * Handles database operations and migrations for the RCP API WordPress Integration plugin.
 *
 * @package RCP_API_WP_Integration
 * @since 0.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Database management class for RCP API plugin.
 */
class RCP_API_Database_Manager {
    
    /**
     * Database version option name.
     *
     * @var string
     */
    const DB_VERSION_OPTION = 'rcp_api_db_version';
    
    /**
     * Current database version.
     *
     * @var string
     */
    const CURRENT_VERSION = '1.0.0';
    
    /**
     * Table names.
     *
     * @var array
     */
    private static $tables = [
        'import_queue' => 'rcp_import_queue',
        'import_history' => 'rcp_import_history',
        'error_log' => 'rcp_error_log',
    ];
    
    /**
     * Initialize database manager.
     */
    public static function init() {
        // Check database version on admin init
        add_action( 'admin_init', [ __CLASS__, 'check_database' ] );
    }
    
    /**
     * Check database and run migrations if needed.
     */
    public static function check_database() {
        $current_version = get_option( self::DB_VERSION_OPTION, '0' );
        
        if ( version_compare( $current_version, self::CURRENT_VERSION, '<' ) ) {
            self::migrate();
        }
    }
    
    /**
     * Get table name with prefix.
     *
     * @param string $table Table key.
     * @return string Full table name.
     */
    public static function get_table_name( $table ) {
        global $wpdb;
        
        if ( ! isset( self::$tables[ $table ] ) ) {
            return false;
        }
        
        return $wpdb->prefix . self::$tables[ $table ];
    }
    
    /**
     * Run database migrations.
     */
    private static function migrate() {
        global $wpdb;
        
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        
        $charset_collate = $wpdb->get_charset_collate();
        $current_version = get_option( self::DB_VERSION_OPTION, '0' );
        
        // Migration for version 1.0.0
        if ( version_compare( $current_version, '1.0.0', '<' ) ) {
            self::migrate_1_0_0( $charset_collate );
        }
        
        // Update database version
        update_option( self::DB_VERSION_OPTION, self::CURRENT_VERSION );
        
        // Run optimization
        self::optimize_tables();
    }
    
    /**
     * Migration for version 1.0.0.
     *
     * @param string $charset_collate Charset collation.
     */
    private static function migrate_1_0_0( $charset_collate ) {
        global $wpdb;
        
        // Import queue table
        $table_name = self::get_table_name( 'import_queue' );
        $sql = "CREATE TABLE $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            post_id bigint(20) unsigned NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            priority int(11) NOT NULL DEFAULT 0,
            category varchar(255) DEFAULT NULL,
            import_type varchar(50) NOT NULL DEFAULT 'manual',
            scheduled_at datetime DEFAULT NULL,
            started_at datetime DEFAULT NULL,
            completed_at datetime DEFAULT NULL,
            retry_count int(11) NOT NULL DEFAULT 0,
            error_message text DEFAULT NULL,
            meta longtext DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY status (status),
            KEY priority_scheduled (priority, scheduled_at),
            KEY category (category)
        ) $charset_collate;";
        
        dbDelta( $sql );
        
        // Import history table
        $table_name = self::get_table_name( 'import_history' );
        $sql = "CREATE TABLE $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            rcp_post_id bigint(20) unsigned NOT NULL,
            wp_post_id bigint(20) unsigned DEFAULT NULL,
            import_type varchar(50) NOT NULL,
            status varchar(20) NOT NULL,
            category varchar(255) DEFAULT NULL,
            author_id bigint(20) unsigned DEFAULT NULL,
            duration float DEFAULT NULL,
            memory_used bigint(20) DEFAULT NULL,
            error_message text DEFAULT NULL,
            meta longtext DEFAULT NULL,
            imported_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY rcp_post_id (rcp_post_id),
            KEY wp_post_id (wp_post_id),
            KEY imported_at (imported_at),
            KEY status_imported (status, imported_at)
        ) $charset_collate;";
        
        dbDelta( $sql );
        
        // Error log table
        $table_name = self::get_table_name( 'error_log' );
        $sql = "CREATE TABLE $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            severity varchar(20) NOT NULL DEFAULT 'error',
            message text NOT NULL,
            context longtext DEFAULT NULL,
            user_id bigint(20) unsigned DEFAULT NULL,
            url varchar(255) DEFAULT NULL,
            method varchar(10) DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY severity_created (severity, created_at),
            KEY user_id (user_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        dbDelta( $sql );
        
        // Add indexes to existing WordPress tables for better performance
        self::add_custom_indexes();
    }
    
    /**
     * Add custom indexes to WordPress tables.
     */
    private static function add_custom_indexes() {
        global $wpdb;
        
        // Add index for _rcp_original_post_id meta queries
        $index_exists = $wpdb->get_var(
            $wpdb->prepare(
                "SHOW INDEX FROM {$wpdb->postmeta} WHERE Key_name = %s",
                'rcp_original_post_id_idx'
            )
        );
        
        if ( ! $index_exists ) {
            $wpdb->query(
                "ALTER TABLE {$wpdb->postmeta} 
                 ADD INDEX rcp_original_post_id_idx (meta_key(20), meta_value(20))"
            );
        }
    }
    
    /**
     * Optimize database tables.
     */
    private static function optimize_tables() {
        global $wpdb;
        
        foreach ( self::$tables as $table ) {
            $table_name = self::get_table_name( $table );
            if ( self::table_exists( $table_name ) ) {
                $wpdb->query( "OPTIMIZE TABLE $table_name" );
            }
        }
    }
    
    /**
     * Check if table exists.
     *
     * @param string $table_name Table name.
     * @return bool True if exists.
     */
    private static function table_exists( $table_name ) {
        global $wpdb;
        
        $query = $wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $table_name
        );
        
        return $wpdb->get_var( $query ) === $table_name;
    }
    
    /**
     * Insert import queue item.
     *
     * @param array $data Queue item data.
     * @return int|false Insert ID or false on failure.
     */
    public static function insert_queue_item( $data ) {
        global $wpdb;
        
        $defaults = [
            'status' => 'pending',
            'priority' => 0,
            'import_type' => 'manual',
            'retry_count' => 0,
            'created_at' => current_time( 'mysql' ),
        ];
        
        $data = wp_parse_args( $data, $defaults );
        
        // Serialize meta data if present
        if ( isset( $data['meta'] ) && is_array( $data['meta'] ) ) {
            $data['meta'] = maybe_serialize( $data['meta'] );
        }
        
        $table_name = self::get_table_name( 'import_queue' );
        $result = $wpdb->insert( $table_name, $data );
        
        return $result ? $wpdb->insert_id : false;
    }
    
    /**
     * Get queue items.
     *
     * @param array $args Query arguments.
     * @return array Queue items.
     */
    public static function get_queue_items( $args = [] ) {
        global $wpdb;
        
        $defaults = [
            'status' => '',
            'category' => '',
            'limit' => 10,
            'offset' => 0,
            'orderby' => 'priority',
            'order' => 'DESC',
        ];
        
        $args = wp_parse_args( $args, $defaults );
        
        $table_name = self::get_table_name( 'import_queue' );
        $where = [];
        $values = [];
        
        if ( ! empty( $args['status'] ) ) {
            $where[] = 'status = %s';
            $values[] = $args['status'];
        }
        
        if ( ! empty( $args['category'] ) ) {
            $where[] = 'category = %s';
            $values[] = $args['category'];
        }
        
        $where_clause = ! empty( $where ) ? 'WHERE ' . implode( ' AND ', $where ) : '';
        
        $query = "SELECT * FROM $table_name $where_clause 
                  ORDER BY {$args['orderby']} {$args['order']} 
                  LIMIT %d OFFSET %d";
        
        $values[] = $args['limit'];
        $values[] = $args['offset'];
        
        $results = $wpdb->get_results(
            $wpdb->prepare( $query, $values ),
            ARRAY_A
        );
        
        // Unserialize meta data
        foreach ( $results as &$item ) {
            if ( ! empty( $item['meta'] ) ) {
                $item['meta'] = maybe_unserialize( $item['meta'] );
            }
        }
        
        return $results;
    }
    
    /**
     * Update queue item.
     *
     * @param int   $id   Item ID.
     * @param array $data Data to update.
     * @return bool True on success.
     */
    public static function update_queue_item( $id, $data ) {
        global $wpdb;
        
        // Serialize meta data if present
        if ( isset( $data['meta'] ) && is_array( $data['meta'] ) ) {
            $data['meta'] = maybe_serialize( $data['meta'] );
        }
        
        $table_name = self::get_table_name( 'import_queue' );
        $result = $wpdb->update(
            $table_name,
            $data,
            [ 'id' => $id ],
            null,
            [ '%d' ]
        );
        
        return false !== $result;
    }
    
    /**
     * Delete queue item.
     *
     * @param int $id Item ID.
     * @return bool True on success.
     */
    public static function delete_queue_item( $id ) {
        global $wpdb;
        
        $table_name = self::get_table_name( 'import_queue' );
        $result = $wpdb->delete(
            $table_name,
            [ 'id' => $id ],
            [ '%d' ]
        );
        
        return false !== $result;
    }
    
    /**
     * Log import history.
     *
     * @param array $data History data.
     * @return int|false Insert ID or false on failure.
     */
    public static function log_import_history( $data ) {
        global $wpdb;
        
        $defaults = [
            'imported_at' => current_time( 'mysql' ),
        ];
        
        $data = wp_parse_args( $data, $defaults );
        
        // Serialize meta data if present
        if ( isset( $data['meta'] ) && is_array( $data['meta'] ) ) {
            $data['meta'] = maybe_serialize( $data['meta'] );
        }
        
        $table_name = self::get_table_name( 'import_history' );
        $result = $wpdb->insert( $table_name, $data );
        
        return $result ? $wpdb->insert_id : false;
    }
    
    /**
     * Get import history.
     *
     * @param array $args Query arguments.
     * @return array History items.
     */
    public static function get_import_history( $args = [] ) {
        global $wpdb;
        
        $defaults = [
            'status' => '',
            'category' => '',
            'date_from' => '',
            'date_to' => '',
            'limit' => 50,
            'offset' => 0,
        ];
        
        $args = wp_parse_args( $args, $defaults );
        
        $table_name = self::get_table_name( 'import_history' );
        $where = [];
        $values = [];
        
        if ( ! empty( $args['status'] ) ) {
            $where[] = 'status = %s';
            $values[] = $args['status'];
        }
        
        if ( ! empty( $args['category'] ) ) {
            $where[] = 'category = %s';
            $values[] = $args['category'];
        }
        
        if ( ! empty( $args['date_from'] ) ) {
            $where[] = 'imported_at >= %s';
            $values[] = $args['date_from'];
        }
        
        if ( ! empty( $args['date_to'] ) ) {
            $where[] = 'imported_at <= %s';
            $values[] = $args['date_to'];
        }
        
        $where_clause = ! empty( $where ) ? 'WHERE ' . implode( ' AND ', $where ) : '';
        
        $query = "SELECT * FROM $table_name $where_clause 
                  ORDER BY imported_at DESC 
                  LIMIT %d OFFSET %d";
        
        $values[] = $args['limit'];
        $values[] = $args['offset'];
        
        $results = $wpdb->get_results(
            $wpdb->prepare( $query, $values ),
            ARRAY_A
        );
        
        // Unserialize meta data
        foreach ( $results as &$item ) {
            if ( ! empty( $item['meta'] ) ) {
                $item['meta'] = maybe_unserialize( $item['meta'] );
            }
        }
        
        return $results;
    }
    
    /**
     * Clean up old data.
     *
     * @param int $days_to_keep Number of days to keep.
     * @return array Cleanup results.
     */
    public static function cleanup_old_data( $days_to_keep = 90 ) {
        global $wpdb;
        
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$days_to_keep} days" ) );
        $results = [];
        
        // Clean import history
        $table_name = self::get_table_name( 'import_history' );
        $deleted = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $table_name WHERE imported_at < %s",
                $cutoff_date
            )
        );
        $results['import_history'] = $deleted;
        
        // Clean error logs
        $table_name = self::get_table_name( 'error_log' );
        $deleted = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $table_name WHERE created_at < %s",
                $cutoff_date
            )
        );
        $results['error_log'] = $deleted;
        
        // Clean old queue items
        $table_name = self::get_table_name( 'import_queue' );
        $deleted = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM $table_name 
                 WHERE status IN ('completed', 'failed') 
                 AND completed_at < %s",
                $cutoff_date
            )
        );
        $results['import_queue'] = $deleted;
        
        return $results;
    }
    
    /**
     * Get database statistics.
     *
     * @return array Statistics.
     */
    public static function get_statistics() {
        global $wpdb;
        
        $stats = [];
        
        foreach ( self::$tables as $key => $table ) {
            $table_name = self::get_table_name( $key );
            if ( self::table_exists( $table_name ) ) {
                $stats[ $key ] = [
                    'count' => $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" ),
                    'size' => $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT ROUND(((data_length + index_length) / 1024 / 1024), 2) 
                             FROM information_schema.TABLES 
                             WHERE table_schema = %s 
                             AND table_name = %s",
                            DB_NAME,
                            $table_name
                        )
                    ),
                ];
            }
        }
        
        return $stats;
    }

    /**
     * Insert a record into import history table.
     *
     * @param array $data Row data.
     * @return int|false Insert ID or false on failure.
     */
    public static function insert_import_history( $data ) {
        global $wpdb;
        $table = self::get_table_name( 'import_history' );
        if ( ! $table ) { return false; }

        $defaults = [
            'rcp_post_id' => 0,
            'wp_post_id'  => null,
            'import_type' => 'manual',
            'status'      => 'success',
            'category'    => null,
            'author_id'   => null,
            'duration'    => null,
            'memory_used' => null,
            'error_message' => null,
            'meta'        => null,
            'imported_at' => current_time( 'mysql' ),
        ];
        $row = wp_parse_args( $data, $defaults );
        if ( is_array( $row['meta'] ) ) {
            $row['meta'] = maybe_serialize( $row['meta'] );
        }

        $ok = $wpdb->insert( $table, $row );
        return $ok ? $wpdb->insert_id : false;
    }
    
    /**
     * Drop all plugin tables.
     *
     * @return bool True on success.
     */
    public static function drop_tables() {
        global $wpdb;
        
        foreach ( self::$tables as $table ) {
            $table_name = self::get_table_name( $table );
            $wpdb->query( "DROP TABLE IF EXISTS $table_name" );
        }
        
        delete_option( self::DB_VERSION_OPTION );
        
        return true;
    }
}
