<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

if ( defined( 'WP_CLI' ) && WP_CLI ) {

    class RCP_API_CLI_Import {
        /**
         * Run the auto import immediately.
         *
         * ## OPTIONS
         *
         * [--force]
         * : Force import ignoring cutoffs for a larger initial backfill.
         */
        public function run( $args, $assoc_args ) {
            $force = isset( $assoc_args['force'] );
            $integration = new RCP_API_WP_Integration();
            $integration->handle_auto_import( '', 0, $force );
            \WP_CLI::success( 'RCP import executed' . ( $force ? ' (forced)' : '' ) . '.' );
        }

        /**
         * Show basic diagnostics.
         */
        public function diagnose( $args, $assoc_args ) {
            $next = wp_next_scheduled( RCP_API_WP_Integration::CRON_HOOK );
            $last_import  = get_option( RCP_API_WP_Integration::OPTION_LAST_IMPORT, 0 );
            $last_success = get_option( RCP_API_WP_Integration::OPTION_LAST_SUCCESS, 0 );
            $queue        = RCP_API_Job_Queue::get_status();

            \WP_CLI::line( 'Cron:' );
            \WP_CLI::line( '  Next scheduled: ' . ( $next ? gmdate( 'Y-m-d H:i:s', $next ) . ' UTC' : 'not scheduled' ) );
            \WP_CLI::line( 'Last Import: ' . ( $last_import ? gmdate( 'Y-m-d H:i:s', $last_import ) . ' UTC' : 'n/a' ) );
            \WP_CLI::line( 'Last Success: ' . ( $last_success ? gmdate( 'Y-m-d H:i:s', $last_success ) . ' UTC' : 'n/a' ) );
            \WP_CLI::line( 'Queue: ' . json_encode( $queue ) );
        }
    }

    class RCP_API_CLI_Log {
        /**
         * Show the last N log entries (options-based, and DB if present).
         *
         * ## OPTIONS
         *
         * [--count=<count>]
         * : Number of entries to show.
         * ---
         * default: 10
         * ---
         */
        public function tail( $args, $assoc_args ) {
            $count = isset( $assoc_args['count'] ) ? (int) $assoc_args['count'] : 10;

            $opt_log = get_option( RCP_API_WP_Integration::OPTION_IMPORT_LOG, [] );
            if ( is_array( $opt_log ) ) {
                \WP_CLI::line( 'Option Log:' );
                $entries = array_slice( array_reverse( $opt_log ), 0, $count );
                foreach ( $entries as $e ) {
                    \WP_CLI::line( sprintf( '- %s | success=%s | published=%d | drafted=%d | errors=%d',
                        isset( $e['time'] ) ? gmdate( 'Y-m-d H:i:s', $e['time'] ) : 'n/a',
                        ! empty( $e['success'] ) ? 'yes' : 'no',
                        count( $e['published'] ?? [] ),
                        count( $e['drafted'] ?? [] ),
                        count( $e['errors'] ?? [] )
                    ) );
                }
            }

            // DB history (if table exists)
            $history = RCP_API_Database_Manager::get_import_history( [ 'limit' => $count ] );
            if ( ! empty( $history ) ) {
                \WP_CLI::line( 'DB History:' );
                foreach ( $history as $row ) {
                    \WP_CLI::line( sprintf( '- %s | type=%s | status=%s | rcp_id=%s | wp_id=%s',
                        $row['imported_at'] ?? 'n/a',
                        $row['import_type'] ?? 'n/a',
                        $row['status'] ?? 'n/a',
                        $row['rcp_post_id'] ?? '-',
                        $row['wp_post_id'] ?? '-'
                    ) );
                }
            }
        }
    }

    class RCP_API_CLI_Cache {
        /**
         * Flush plugin caches (object cache + transients).
         */
        public function flush( $args, $assoc_args ) {
            if ( class_exists( 'RCP_API_Cache_Manager' ) ) {
                RCP_API_Cache_Manager::flush_all();
                \WP_CLI::success( 'RCP caches flushed.' );
            } else {
                \WP_CLI::warning( 'Cache manager not available.' );
            }
        }
    }

    // Register commands
    \WP_CLI::add_command( 'rcp import', 'RCP_API_CLI_Import' );
    \WP_CLI::add_command( 'rcp log', 'RCP_API_CLI_Log' );
    \WP_CLI::add_command( 'rcp cache', 'RCP_API_CLI_Cache' );

    class RCP_API_CLI_AI {
        /**
         * Show AI credits balance from the primary site.
         */
        public function balance( $args, $assoc_args ) {
            $integration = new RCP_API_WP_Integration();
            $res = $integration->get_credits_balance();
            if ( is_wp_error( $res ) ) {
                \WP_CLI::error( 'Balance fetch failed: ' . $res->get_error_message() );
            } else {
                $balance = isset( $res['balance'] ) ? (int) $res['balance'] : -1;
                $available = isset( $res['available_balance'] ) ? (int) $res['available_balance'] : $balance;
                \WP_CLI::success( sprintf( 'Balance: %d (available: %d)', $balance, $available ) );
            }
        }

        /**
         * Queue AI image generation.
         *
         * ## OPTIONS
         *
         * [--post=<id>]
         * : A specific WP post ID to generate for.
         *
         * [--missing-thumbnails]
         * : Queue for posts (type=post) missing featured images.
         *
         * [--since=<date>]
         * : Only consider posts published after this date (e.g., 2024-09-01).
         *
         * [--dry-run]
         * : Do not queue; just print matches.
         */
        public function generate( $args, $assoc_args ) {
            $integration = new RCP_API_WP_Integration();
            $dry = isset( $assoc_args['dry-run'] );
            if ( isset( $assoc_args['post'] ) ) {
                $post_id = (int) $assoc_args['post'];
                if ( $dry ) {
                    \WP_CLI::line( "Would queue AI generation for post #$post_id" );
                    return;
                }
                RCP_API_Job_Queue::schedule_async( 'ai_generate', [ 'post_id' => $post_id, 'rcp_post_id' => 0 ] );
                \WP_CLI::success( "Queued AI generation for post #$post_id" );
                return;
            }

            if ( isset( $assoc_args['missing-thumbnails'] ) ) {
                $query = [
                    'post_type'      => 'post',
                    'post_status'    => 'any',
                    'posts_per_page' => -1,
                    'meta_key'       => '_rcp_original_post_id',
                    'meta_compare'   => 'EXISTS',
                ];
                if ( isset( $assoc_args['since'] ) ) {
                    $date = sanitize_text_field( $assoc_args['since'] );
                    $query['date_query'] = [ [ 'after' => $date ] ];
                }
                $ids = get_posts( array_merge( $query, [ 'fields' => 'ids' ] ) );
                $target = [];
                foreach ( (array) $ids as $pid ) {
                    if ( ! has_post_thumbnail( $pid ) ) {
                        $target[] = (int) $pid;
                    }
                }
                if ( $dry ) {
                    \WP_CLI::line( 'Would queue ' . count( $target ) . ' posts.' );
                    return;
                }
                foreach ( $target as $pid ) {
                    RCP_API_Job_Queue::schedule_async( 'ai_generate', [ 'post_id' => $pid, 'rcp_post_id' => 0 ] );
                }
                \WP_CLI::success( 'Queued ' . count( $target ) . ' posts for AI image generation.' );
                return;
            }

            \WP_CLI::error( 'Provide --post=<id> or --missing-thumbnails.' );
        }

        /**
         * Show recent AI image jobs from local log.
         *
         * ## OPTIONS
         * [--status=<status>]
         * : Filter by status (queued|success|failed|timeout)
         * [--limit=<n>]
         * : Number of rows to display (default 20)
         */
        public function jobs( $args, $assoc_args ) {
            $status = isset( $assoc_args['status'] ) ? sanitize_text_field( $assoc_args['status'] ) : '';
            $limit  = isset( $assoc_args['limit'] ) ? max( 1, (int) $assoc_args['limit'] ) : 20;
            $log = get_option( 'rcp_ai_image_log', [] );
            if ( ! is_array( $log ) ) { $log = []; }
            $rows = array_reverse( array_slice( $log, -$limit ) );
            foreach ( $rows as $row ) {
                if ( $status && ( (string) ( $row['status'] ?? '' ) !== $status ) ) { continue; }
                $time = isset( $row['time'] ) ? gmdate( 'Y-m-d H:i:s', (int) $row['time'] ) : 'n/a';
                $pid = (int) ( $row['post_id'] ?? 0 );
                $st  = (string) ( $row['status'] ?? '-' );
                \WP_CLI::line( sprintf( '%s | post=%d | status=%s', $time, $pid, $st ) );
            }
        }
    }

    \WP_CLI::add_command( 'rcp ai', 'RCP_API_CLI_AI' );
}
