<?php
/**
 * RCP API Cache Manager
 *
 * Provides advanced caching functionality for the RCP API WordPress Integration plugin.
 *
 * @package RCP_API_WP_Integration
 * @since 0.8.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Cache management class for RCP API plugin.
 */
class RCP_API_Cache_Manager {
    
    /**
     * Cache groups.
     *
     * @var array
     */
    private static $cache_groups = [
        'api_responses' => 'rcp_api',
        'categories' => 'rcp_categories',
        'posts' => 'rcp_posts',
        'settings' => 'rcp_settings',
    ];
    
    /**
     * Default cache TTL in seconds.
     *
     * @var int
     */
    private static $default_ttl = 300; // 5 minutes
    
    /**
     * Initialize cache manager.
     */
    public static function init() {
        // Add cache invalidation hooks
        add_action( 'rcp_api_post_imported', [ __CLASS__, 'invalidate_post_cache' ], 10, 2 );
        add_action( 'rcp_api_settings_updated', [ __CLASS__, 'invalidate_settings_cache' ] );
        add_action( 'updated_option', [ __CLASS__, 'maybe_invalidate_cache' ], 10, 3 );
        
        // Add cache warming
        add_action( 'rcp_api_after_import', [ __CLASS__, 'warm_cache' ] );
    }
    
    /**
     * Get item from cache.
     *
     * @param string $key   Cache key.
     * @param string $group Cache group.
     * @return mixed|false Cached data or false if not found.
     */
    public static function get( $key, $group = 'api_responses' ) {
        $cache_key = self::get_cache_key( $key );
        $cache_group = self::get_cache_group( $group );
        
        // Try object cache first
        $cached = wp_cache_get( $cache_key, $cache_group );
        if ( false !== $cached ) {
            self::record_hit( $key, $group, 'object' );
            return $cached;
        }
        
        // Fall back to transient cache
        $transient_key = self::get_transient_key( $key, $group );
        $cached = get_transient( $transient_key );
        if ( false !== $cached ) {
            // Promote to object cache
            wp_cache_set( $cache_key, $cached, $cache_group, self::get_ttl( $group ) );
            self::record_hit( $key, $group, 'transient' );
            return $cached;
        }
        
        self::record_miss( $key, $group );
        return false;
    }
    
    /**
     * Set item in cache.
     *
     * @param string $key   Cache key.
     * @param mixed  $data  Data to cache.
     * @param string $group Cache group.
     * @param int    $ttl   Time to live in seconds.
     * @return bool True on success.
     */
    public static function set( $key, $data, $group = 'api_responses', $ttl = null ) {
        $cache_key = self::get_cache_key( $key );
        $cache_group = self::get_cache_group( $group );
        $ttl = $ttl ?: self::get_ttl( $group );
        
        // Set in object cache
        $object_result = wp_cache_set( $cache_key, $data, $cache_group, $ttl );
        
        // Also set in transient cache as fallback
        $transient_key = self::get_transient_key( $key, $group );
        $transient_result = set_transient( $transient_key, $data, $ttl );
        
        return $object_result || $transient_result;
    }
    
    /**
     * Delete item from cache.
     *
     * @param string $key   Cache key.
     * @param string $group Cache group.
     * @return bool True on success.
     */
    public static function delete( $key, $group = 'api_responses' ) {
        $cache_key = self::get_cache_key( $key );
        $cache_group = self::get_cache_group( $group );
        
        // Delete from object cache
        $object_result = wp_cache_delete( $cache_key, $cache_group );
        
        // Delete from transient cache
        $transient_key = self::get_transient_key( $key, $group );
        $transient_result = delete_transient( $transient_key );
        
        return $object_result || $transient_result;
    }
    
    /**
     * Flush cache group.
     *
     * @param string $group Cache group to flush.
     * @return bool True on success.
     */
    public static function flush_group( $group ) {
        $cache_group = self::get_cache_group( $group );
        
        // Flush object cache group
        if ( function_exists( 'wp_cache_flush_group' ) ) {
            wp_cache_flush_group( $cache_group );
        } else {
            // Fallback: increment group version to invalidate
            $version_key = 'rcp_cache_version_' . $group;
            $version = (int) get_option( $version_key, 0 );
            update_option( $version_key, $version + 1, false );
        }
        
        // Clear transients for this group
        self::clear_transients( $group );
        
        return true;
    }
    
    /**
     * Flush all plugin caches.
     *
     * @return bool True on success.
     */
    public static function flush_all() {
        foreach ( array_keys( self::$cache_groups ) as $group ) {
            self::flush_group( $group );
        }
        
        // Clear all plugin transients
        global $wpdb;
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_rcp_%' 
             OR option_name LIKE '_transient_timeout_rcp_%'"
        );
        
        return true;
    }
    
    /**
     * Get cache key with versioning.
     *
     * @param string $key Original key.
     * @return string Versioned cache key.
     */
    private static function get_cache_key( $key ) {
        return 'rcp_' . md5( $key );
    }
    
    /**
     * Get cache group name.
     *
     * @param string $group Group key.
     * @return string Full group name.
     */
    private static function get_cache_group( $group ) {
        if ( isset( self::$cache_groups[ $group ] ) ) {
            $base_group = self::$cache_groups[ $group ];
        } else {
            $base_group = 'rcp_' . $group;
        }
        
        // Add version to group for easy invalidation
        $version_key = 'rcp_cache_version_' . $group;
        $version = get_option( $version_key, 0 );
        
        return $base_group . '_v' . $version;
    }
    
    /**
     * Get transient key.
     *
     * @param string $key   Cache key.
     * @param string $group Cache group.
     * @return string Transient key.
     */
    private static function get_transient_key( $key, $group ) {
        return 'rcp_' . $group . '_' . md5( $key );
    }
    
    /**
     * Get TTL for cache group.
     *
     * @param string $group Cache group.
     * @return int TTL in seconds.
     */
    private static function get_ttl( $group ) {
        $ttl_map = [
            'api_responses' => RCP_CACHE_TTL,
            'categories' => 900, // 15 minutes
            'posts' => 600, // 10 minutes
            'settings' => 3600, // 1 hour
        ];
        
        $ttl = isset( $ttl_map[ $group ] ) ? $ttl_map[ $group ] : self::$default_ttl;
        
        return apply_filters( 'rcp_api_cache_ttl', $ttl, $group );
    }
    
    /**
     * Clear transients for a group.
     *
     * @param string $group Cache group.
     */
    private static function clear_transients( $group ) {
        global $wpdb;
        
        $prefix = 'rcp_' . $group . '_';
        
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} 
                 WHERE option_name LIKE %s 
                 OR option_name LIKE %s",
                '_transient_' . $prefix . '%',
                '_transient_timeout_' . $prefix . '%'
            )
        );
    }
    
    /**
     * Invalidate post cache.
     *
     * @param int   $post_id    WordPress post ID.
     * @param array $import_data Import data.
     */
    public static function invalidate_post_cache( $post_id, $import_data ) {
        // Clear specific post cache
        self::delete( 'post_' . $post_id, 'posts' );
        
        // Clear related category caches
        if ( ! empty( $import_data['categories'] ) ) {
            foreach ( $import_data['categories'] as $category ) {
                self::delete( 'category_' . sanitize_title( $category ), 'categories' );
            }
        }
        
        // Clear feed caches that might contain this post
        self::flush_group( 'api_responses' );
    }
    
    /**
     * Invalidate settings cache.
     */
    public static function invalidate_settings_cache() {
        self::flush_group( 'settings' );
        self::flush_group( 'categories' );
    }
    
    /**
     * Maybe invalidate cache on option update.
     *
     * @param string $option    Option name.
     * @param mixed  $old_value Old value.
     * @param mixed  $new_value New value.
     */
    public static function maybe_invalidate_cache( $option, $old_value, $new_value ) {
        $watched_options = [
            'rcp_feed_username',
            'rcp_feed_password',
            'rcp_category_mapping',
            'rcp_category_automation',
            'rcp_featured_image_cats',
        ];
        
        if ( in_array( $option, $watched_options, true ) ) {
            self::flush_all();
        }
    }
    
    /**
     * Warm cache after import.
     *
     * @param array $import_results Import results.
     */
    public static function warm_cache( $import_results ) {
        // Warm category cache
        $categories = self::get_all_categories();
        foreach ( $categories as $category ) {
            $cache_key = 'category_' . sanitize_title( $category );
            self::set( $cache_key, $category, 'categories' );
        }
        
        // Warm recent posts cache
        if ( ! empty( $import_results['imported_posts'] ) ) {
            foreach ( $import_results['imported_posts'] as $post_id ) {
                $post = get_post( $post_id );
                if ( $post ) {
                    self::set( 'post_' . $post_id, $post, 'posts' );
                }
            }
        }
    }
    
    /**
     * Get all categories for cache warming.
     *
     * @return array Categories.
     */
    private static function get_all_categories() {
        $integration = new RCP_API_WP_Integration();
        $categories = $integration->get_api_categories();
        
        return is_array( $categories ) ? $categories : [];
    }
    
    /**
     * Record cache hit.
     *
     * @param string $key   Cache key.
     * @param string $group Cache group.
     * @param string $type  Cache type (object/transient).
     */
    private static function record_hit( $key, $group, $type ) {
        if ( defined( 'RCP_DEBUG_CACHE' ) && RCP_DEBUG_CACHE ) {
            RCP_API_Performance_Monitor::record( 'cache_hit', [
                'key' => $key,
                'group' => $group,
                'type' => $type,
            ], 'cache' );
        }
    }
    
    /**
     * Record cache miss.
     *
     * @param string $key   Cache key.
     * @param string $group Cache group.
     */
    private static function record_miss( $key, $group ) {
        if ( defined( 'RCP_DEBUG_CACHE' ) && RCP_DEBUG_CACHE ) {
            RCP_API_Performance_Monitor::record( 'cache_miss', [
                'key' => $key,
                'group' => $group,
            ], 'cache' );
        }
    }
    
    /**
     * Get cache statistics.
     *
     * @return array Cache statistics.
     */
    public static function get_statistics() {
        global $wp_object_cache;
        
        $stats = [
            'object_cache' => [
                'enabled' => wp_using_ext_object_cache(),
                'type' => 'unknown',
            ],
            'groups' => [],
        ];
        
        // Try to determine object cache type
        if ( $stats['object_cache']['enabled'] ) {
            if ( class_exists( 'Redis' ) || class_exists( 'Predis\Client' ) ) {
                $stats['object_cache']['type'] = 'Redis';
            } elseif ( class_exists( 'Memcached' ) || class_exists( 'Memcache' ) ) {
                $stats['object_cache']['type'] = 'Memcached';
            }
        }
        
        // Get group statistics
        foreach ( self::$cache_groups as $key => $group ) {
            $version_key = 'rcp_cache_version_' . $key;
            $stats['groups'][ $key ] = [
                'version' => get_option( $version_key, 0 ),
            ];
        }
        
        return $stats;
    }
}