# RCP AI Image Generation — Implementation Plan (To‑Do)

This is the working to‑do plan to integrate AI image generation via Radio Content Pro’s centralized credits system. It reuses the existing Application Password + email that users already enter for the RCP API.

Status: Draft (v1)
Owners: Plugin team (secondary site) + Primary site API team

## Goals

- Allow users to generate AI images for imported posts a la carte or automatically on import (opt‑in, only if no featured image).
- Use centralized credits on the primary site for reservation/accounting and enforce model/size limits there.
- Keep UX simple, reliable, and scalable; reuse existing credentials; provide clear statuses and safe fallbacks.

## Primary‑Site Endpoints (per credits‑api‑reference.md)

Namespace: `/wp-json/rcp-ai-credits/v1`

- GET `/balance?user_id=` → `{ user_id, balance, available_balance }`
- GET `/action-cost?action_id=image_generation` → `{ action_id, cost }`
- POST `/reserve` → `{ reservation_id }`
  - Body: `{ user_id?, amount?, action_id }` (amount optional; inferred from action_id if omitted)
- POST `/confirm` → `{ success: true }`
  - Body: `{ reservation_id, context? }`
- POST `/cancel` → `{ success: true }`
  - Body: `{ reservation_id }`
- POST `/generation-log` → `{ generation_log_id }`
- POST `/generate-image` → `{ generation_log_id, result_type, result_url, credits_used, credits_confirmed }` (200 sync)
  - Body: `{ prompt, size?, quality?, model?, user_id?, post_id?, reservation_id?, context? }`
- GET `/job-status?generation_log_id=` → `{ status: success|failed|pending, generation_log_id, result_type?, result_url?, credits_used?, error_message?, created_at }`

Notes
- v1 may return 200 synchronously. If/when async is used, client polls by `generation_log_id`.
- The primary site enforces model/size/quality; the plugin sends only prompt + context.

## Phasing

- P0 (MVP): Per‑post generation, automation toggle (off by default), queue + attach, prompt fetching, error handling, minimal diagnostics.
- P1: Balance display, bulk actions, UI polish, robust retries/backoff, idempotency.
- P2: CLI, logs/diagnostics page, i18n coverage, docs.
- P3: Webhook optionality, analytics/log forwarding, cost display auto‑refresh.

---

## P0 — Core Functionality

- [x] Client: Credits API base and auth
  - [x] Derive credits base from content base: `.../wp-json/rcp/v1` → `.../wp-json/rcp-ai-credits/v1` with HTTPS only.
  - [x] Use Basic Auth with existing email + Application Password.
  - [x] Common request helper with timeouts, JSON encode/decode, error mapping.
  - Notes: Added `get_credits_api_base()` with HTTPS enforcement and `credits_request()` as a shared helper.

- [x] Prompt strategy
  - [x] Fetch default prompt from content API `ai_image_generation_prompt?post_id=RCP_ID`.
  - [x] Fallback prompt from WP post title + excerpt.
  - [x] Filters: `rcp_api_ai_prompt_endpoint`, `rcp_api_ai_prompt_param`, `rcp_api_default_image_prompt`.

- [x] Generate image call
  - [x] Implement `generate-image` POST with `{ prompt, context: { wp_post_id, rcp_post_id, source } }`.
  - [x] Handle 200 sync success: read `result_url`, `generation_log_id`.
  - [x] Handle future async: accept `{ generation_log_id }` and schedule polling.
  - [x] Sanitize/validate all response fields.

- [x] Poll job status
  - [x] Implement GET `job-status?generation_log_id=...`.
  - [x] States: `pending|running|success|failed` (treat unknown as `pending` once, then fail with backoff cap).
  - [x] On success: sideload image, attach featured, alt from prompt; ensure content‑type/size checks; avoid duplicates.

- [x] Job queue integration
  - [x] Action Scheduler tasks:
    - [x] `rcp_api_job_ai_generate` → call generate‑image, branch on 200 vs async.
    - [x] `rcp_api_job_ai_poll` → poll status with exponential backoff + cap.
  - [x] Idempotency: skip if post already has thumbnail; dedupe queued tasks per post (transient `rcp_ai_gen_pending_{post}`; cleared on success/failure).

- [x] UI (Feed screen)
  - [x] Add “Generate AI Image” button for already‑imported rows; nonce‑protected AJAX → queue `ai_generate`.
  - [x] After manual import, render same button on the row.
  - [x] Show “Queued for image generation” feedback on success.

- [x] Settings
  - [x] Add “AI Images” section in Settings with a single toggle:
        “Generate AI image on import only if no featured image” (default off; no per‑day cap).
  - [x] On import, if toggle on and post lacks featured image, schedule `ai_generate` with slight delay (allow original featured image import to finish first).

- [x] Security/Hardening
  - [x] Capability `manage_rcp_api` for settings & AJAX.
  - [x] Nonce on AJAX; validate IDs; sanitize payloads.
  - [x] SSRF protection + HEAD checks on image sideload (content‑type, size limit).
  - [x] HTTPS enforcement for credits base URL; reject plain HTTP.

- [x] Errors/UX
  - [x] Clear error messages for: missing creds, server error, insufficient credits (propagate message), validation failures.
  - [x] Gracefully degrade if endpoints unavailable; do not fatal.

Acceptance for P0
- Per‑post button queues job; generates and attaches an image on success.
- Automation (if enabled) runs on import only when no thumbnail exists.
- Errors surface in UI and logs; no fatal on activation/runtime.

---

## P1 — UX & Reliability

- [x] Balance display
  - [x] Fetch GET `/balance`; show in Settings “AI Images” card.
  - [x] Link “Buy credits” to the primary dashboard URL (config or filter if endpoint not exposed).
  - Notes: Uses `credits_request('GET','balance')`. Displays `Balance: X (available: Y)` and a filtered dashboard link (`rcp_ai_credits_dashboard_url`).

- [x] Bulk actions
  - [x] Add bulk “Generate AI Images” for imported posts on current page (button near Refresh).
  - [x] Respect dedupe + concurrency; appends “Queued for image generation” per row.
  - Notes: Uses the presence of per-row AI button to identify imported rows. Posts are queued with transient dedupe.

- [x] Retries/Backoff
  - [x] Exponential backoff on `job-status` polling with max attempts (filter `rcp_ai_poll_max_attempts`, default 10).
  - [x] Backoff on 429 with jitter; errors logged; pending flag cleared on failure/timeout.

- [x] Idempotency & dedupe
  - [x] Prevent duplicate generate requests with a transient `rcp_ai_gen_pending_{post}` and skip if thumbnail exists.
  - [x] Store `_rcp_ai_generation_log_id` post meta for in‑flight correlation; cleared on success.

- [x] Diagnostics (light)
  - [x] Admin “AI Images (Recent Activity)” card listing last entries (queued/success/failed/timeout).
  - [ ] Optional: Add one‑click retry for failures.

- [ ] Prompt polish
  - [ ] Optional negative prompt preset and style filters via hooks only (server enforces actual limits).

Acceptance for P1
- Settings shows balance; bulk generate works with dedupe; polling/retries stable; light diagnostics present.

---

## P2 — Tooling, i18n, docs, CLI

- [x] CLI commands
  - [x] `wp rcp ai balance`
  - [x] `wp rcp ai generate --post=<id>|--missing-thumbnails [--since=<date>] [--dry-run]`
  - [x] `wp rcp ai jobs --status=<state> [--limit=<n>]`
  - Notes: Added `get_credits_balance()` helper for CLI to fetch balance; generate queues async jobs; jobs reads recent local log.

- [x] Logs & diagnostics page
  - [x] Dedicated “AI Images” subpage with paginated log table: post, status, details, and Retry action for failed/timeout.
  - [x] Clear log action and status filter; pagination controls and per-page selector.
  - Notes: Attempts not tracked yet (can be added later); uses local option log.

- [x] Internationalization
  - [x] Wrapped new strings with the existing text domain.
  - [ ] Optional: Regenerate POT to include new strings (team task).

- [x] Documentation
  - [x] README/Overview: added AI Images feature overview, settings, and CLI usage.
  - [ ] Optional: Expand admin help text with credits usage policies.

- [x] Tests
  - [x] Unit: credits base URL enforcement + path replacement
  - [x] Unit: AI image log trim to 200 entries
  - [ ] Optional: add more (HTTP client mocks, polling transitions, sideload checks)

Acceptance for P2
- CLI usable; diagnostics page helpful; i18n complete; docs updated; tests passing locally and in CI.

---

## P3 — Enhancements (Optional)

- [ ] Webhook delivery (optional)
  - [ ] Support callback_url + HMAC signature (server opt‑in); validate timestamp + signature; queue finalize on webhook.

- [ ] Cost auto‑refresh
  - [ ] Use GET `/action-cost?action_id=image_generation` for display; cache; notify on changes.

- [ ] Analytics
  - [ ] Optionally POST `/generation-log` for richer analytics context (model, size, processing time) aligned to generation events.

- [ ] Advanced policy hooks
  - [ ] `rcp_api_ai_generation_allowed($allowed, $post_id)` to gate generation based on custom rules.

---

## Code Tasks (Mapping)

- [ ] Client helpers (PHP)
  - [ ] `rcp_ai_client_generate_image( $prompt, $context )`
  - [ ] `rcp_ai_client_job_status( $generation_log_id )`
  - [ ] `rcp_ai_client_balance()`

- [ ] Orchestration methods
  - [ ] `generate_ai_image_for_post( $wp_post_id, $rcp_post_id = 0, $prompt_override = '' )`
  - [ ] `poll_ai_job_status( $generation_log_id )` (align to new query param)
  - [ ] `attach_image_from_url( $post_id, $url, $alt )`

- [ ] Job queue
  - [ ] `rcp_api_job_ai_generate`: call generate, branch sync/async; store `generation_log_id` in meta/transient.
  - [ ] `rcp_api_job_ai_poll`: poll by `generation_log_id`, backoff, attach on success.

- [ ] AJAX/UI
  - [ ] `wp_ajax_rcp_generate_ai_image` handler: queue generate; returns `{ queued: true }`.
  - [ ] Feed row button and messages; bulk action binding.

- [ ] Settings
  - [ ] Option `rcp_ai_autogen_on_import` (bool; default false)
  - [ ] Balance display (read‑only) + “Buy credits” link.

- [ ] Security & validation
  - [ ] Capabilities + nonces + sanitize; SSRF & MIME checks.

---

## Server‑Side Expectations (Primary Site)

- [ ] Endpoints implemented exactly as in credits‑api‑reference.md.
- [ ] Enforce allowed model/size/quality; ignore/clamp client input.
- [ ] Reservation handling:
  - [ ] If `reservation_id` omitted on `generate-image`, do atomic reserve→generate→confirm.
  - [ ] Return `generation_log_id` and `result_url` on success; `pending` job state exposed by `job-status`.
- [ ] Rate limiting + error semantics stable (`reservation_required`, `insufficient_credits`, `db_error`, etc.).

---

## Tracking & Acceptance Criteria

- [ ] MVP: A user can import a post, click “Generate AI Image,” and get a featured image attached using credits.
- [ ] Automation: With the toggle enabled, posts that lack a featured image automatically get an AI image generation attempt.
- [ ] Safety: Image type and size validated; credentials reused; HTTPS enforced; no fatal paths.
- [ ] Reliability: Retries/backoff; dedupe; no duplicate attachments.
- [ ] UX: Clear status/feedback; minimal settings; no required model/size inputs.

---

## Open Questions

- Balance dashboard URL: expose from primary site for deep link (“Buy credits”) or provide static URL?
- ai_image_generation_prompt path/shape: confirm final path and whether it returns `{ prompt }` or raw text.
- Async policy: will v1 stay synchronous for now, or do we need to support `pending` flows immediately?
- Any constraints on allowed post types (only `post`, or also `page`/CPT)?

---

## Change Log (for this plan)

- v1: Initial plan aligned with updated endpoints: `generate-image` + `job-status` by `generation_log_id`.
