# RCP AI Credits Manager - API Reference

## Table of Contents
- [Public API Functions](#public-api-functions)
- [Credit Operations](#credit-operations)
- [Reservation System](#reservation-system)
- [Action Registration](#action-registration)
- [Hooks & Filters](#hooks--filters)
- [Database Schema](#database-schema)
- [Email System](#email-system)

## Public API Functions

### Core Credit Operations

#### `rcp_ai_credits_get_balance($user_id)`
Get the current credit balance for a user.

**Parameters:**
- `$user_id` (int) - WordPress user ID

**Returns:**
- (int) - Current credit balance

**Example:**
```php
$balance = rcp_ai_credits_get_balance(get_current_user_id());
echo "You have {$balance} credits remaining";
```

#### `rcp_ai_credits_can_perform_action($user_id, $action_id, $cost = null)`
Check if a user has enough credits to perform an action.

**Parameters:**
- `$user_id` (int) - WordPress user ID
- `$action_id` (string) - Registered action identifier
- `$cost` (int|null) - Optional custom cost, uses registered cost if null

**Returns:**
- (bool) - True if user has sufficient credits

**Example:**
```php
if (rcp_ai_credits_can_perform_action($user_id, 'gpt_completion', 10)) {
    // Proceed with AI operation
}
```

#### `rcp_ai_credits_deduct($user_id, $amount, $action_id, $context = array())`
Deduct credits from a user's balance.

**Parameters:**
- `$user_id` (int) - WordPress user ID
- `$amount` (int) - Number of credits to deduct
- `$action_id` (string) - Action identifier for tracking
- `$context` (array) - Additional context data

**Returns:**
- (bool|WP_Error) - True on success, WP_Error on failure

**Example:**
```php
$result = rcp_ai_credits_deduct(
    $user_id, 
    10, 
    'image_generation',
    array(
        'prompt' => 'A beautiful sunset',
        'model' => 'dall-e-3'
    )
);

if (is_wp_error($result)) {
    echo $result->get_error_message();
}
```

#### `rcp_ai_credits_add($user_id, $amount, $source = 'manual')`
Add credits to a user's balance.

**Parameters:**
- `$user_id` (int) - WordPress user ID
- `$amount` (int) - Number of credits to add
- `$source` (string) - Source identifier ('manual', 'purchase', 'monthly', 'auto_refill', 'admin')

**Returns:**
- (bool|WP_Error) - True on success, WP_Error on failure

**Example:**
```php
// Add 100 credits from a purchase source
rcp_ai_credits_add($user_id, 100, 'purchase');
```

## Reservation System

For long-running operations, use the reservation system to prevent race conditions:

#### `rcp_ai_credits_reserve($user_id, $amount, $action_id)`
Reserve credits for a future operation. Returns a reservation ID.

**Parameters:**
- `$user_id` (int) - WordPress user ID
- `$amount` (int) - Number of credits to reserve
- `$action_id` (string) - Action identifier

**Returns:**
- (string|false) - Reservation ID on success, false on failure

#### `rcp_ai_credits_confirm_reservation($reservation_id, $additional_context = array())`
Confirm a reservation and deduct the reserved amount. You may pass additional context for logging.

**Parameters:**
- `$reservation_id` (string) - Reservation ID from reserve function
- `$additional_context` (array) - Optional metadata to include with confirmation

**Returns:**
- (bool) - Success status

#### `rcp_ai_credits_cancel_reservation($reservation_id)`
Cancel a reservation and release the reserved credits.

**Parameters:**
- `$reservation_id` (string) - Reservation ID

**Returns:**
- (bool) - Success status

**Complete Reservation Example:**
```php
// Reserve credits
$reservation_id = rcp_ai_credits_reserve($user_id, 50, 'complex_generation');

if ($reservation_id) {
    try {
        // Perform your AI operation
        $result = perform_ai_operation();
        
        // Confirm reservation (include optional context if useful)
        rcp_ai_credits_confirm_reservation($reservation_id, array(
            'job_id' => $result['id'] ?? null,
        ));
        
    } catch (Exception $e) {
        // Cancel if operation fails
        rcp_ai_credits_cancel_reservation($reservation_id);
        throw $e;
    }
}
```

## Action Registration

Register custom AI actions for credit consumption tracking:

#### `rcp_ai_credits_register_action($args)`
Register a new AI action type.

**Parameters:**
- `$args` (array) - Action configuration
  - `action_id` (string) - Unique identifier
  - `label` (string) - Display name
  - `cost` (int) - Default credit cost
  - `description` (string) - Optional description
  - `plugin` (string) - Source plugin name
  - `category` (string) - Optional category

**Returns:**
- (bool) - Registration success

**Example:**
```php
// Register during plugin initialization
add_action('init', function() {
    rcp_ai_credits_register_action(array(
        'action_id' => 'my_plugin_chat',
        'label' => 'AI Chat Response',
        'cost' => 5,
        'description' => 'Generate AI chat responses',
        'plugin' => 'my-plugin',
        'category' => 'text_generation'
    ));
});
```

#### `rcp_ai_credits_get_action_cost($action_id)`
Get the current cost for a registered action. This is the single source of truth for action costs.

```php
$cost = rcp_ai_credits_get_action_cost('my_plugin_chat');
```

#### `rcp_ai_credits_ensure_action_defaults($action_id, $defaults)`
Ensure an action exists with default values without overriding existing data.

```php
rcp_ai_credits_ensure_action_defaults('my_plugin_chat', array(
  'label' => 'AI Chat Response',
  'cost' => 5,
  'description' => 'Generate AI chat responses',
));
```

## Hooks & Filters

### Action Hooks

#### `rcp_ai_credits_after_deduction`
Fired after credits are successfully deducted.

**Parameters:**
- `$user_id` (int) - User ID
- `$amount` (int) - Credits deducted
- `$action_id` (string) - Action identifier
- `$new_balance` (int) - New credit balance

```php
add_action('rcp_ai_credits_after_deduction', function($user_id, $amount, $action_id, $new_balance) {
    // Log usage to external analytics
    my_analytics_track('credits_used', array(
        'user' => $user_id,
        'amount' => $amount,
        'action' => $action_id,
        'balance' => $new_balance
    ));
}, 10, 4);
```

#### `rcp_ai_credits_after_addition`
Fired after credits are added to a user's balance.

**Parameters:**
- `$user_id` (int) - User ID
- `$amount` (int) - Credits added
- `$source` (string) - Source of credits
- `$new_balance` (int) - New credit balance

#### `rcp_ai_credits_low_balance`
Fired when user's balance drops below threshold.

**Parameters:**
- `$user_id` (int) - User ID
- `$balance` (int) - Current balance
- `$threshold` (int) - Low balance threshold

#### `rcp_ai_credits_auto_refill_success`
Fired when auto-refill completes successfully.

**Parameters:**
- `$user_id` (int) - User ID
- `$refill_amount` (int) - Credits added
- `$price` (float) - Price charged

#### `rcp_ai_credits_auto_refill_failed`
Fired when auto-refill fails.

**Parameters:**
- `$user_id` (int) - User ID
- `$error_code` (string)
- `$error_message` (string)
- `$amount` (int) - Credits to be added
- `$price` (float) - Amount to charge

### Filter Hooks

#### `rcp_ai_credits_action_cost`
Filter the cost of an action before deduction.

**Parameters:**
- `$cost` (int) - Default cost
- `$action_id` (string) - Action identifier
- `$user_id` (int) - User ID
- `$context` (array) - Additional context

```php
add_filter('rcp_ai_credits_action_cost', function($cost, $action_id, $user_id, $context) {
    // 20% discount for premium members
    if (user_has_premium_membership($user_id)) {
        return ceil($cost * 0.8);
    }
    return $cost;
}, 10, 4);
```

#### `rcp_ai_credits_can_perform_action`
Filter the boolean result of whether a user can perform an action based on balance and cost.

**Parameters:**
- `$can_perform` (bool) - Default result
- `$user_id` (int) - User ID
- `$action_id` (string) - Action identifier

#### `rcp_ai_credits_refill_price`
Filter the calculated refill price for a given credit amount.

**Parameters:**
- `$base_price` (float) - Base price
- `$credits` (int) - Credits to be added

#### `rcp_ai_credits_max_amount`
Filter the maximum allowed amount for manual operations.

#### `rcp_ai_credits_valid_currencies`
Filter valid currencies for purchases.

#### `rcp_ai_credits_max_price`
Filter the maximum allowed price for purchases.

## Database Schema

### Core Tables

#### `wp_ai_credit_transactions`
```sql
CREATE TABLE wp_ai_credit_transactions (
    id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    user_id bigint(20) UNSIGNED NOT NULL,
    type enum('add','deduct','reserve','cancel') NOT NULL,
    amount int(11) NOT NULL,
    balance_after int(11) NOT NULL,
    source varchar(50),
    action_id varchar(100),
    metadata longtext,
    created_at datetime DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    KEY user_id (user_id),
    KEY created_at (created_at),
    KEY type_source (type, source)
)
```

#### `wp_ai_credit_buckets`
Credit buckets with expiration support:
```sql
CREATE TABLE wp_ai_credit_buckets (
    id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    user_id bigint(20) UNSIGNED NOT NULL,
    credits int(11) NOT NULL,
    source varchar(50) NOT NULL,
    expires_at datetime DEFAULT NULL,
    created_at datetime DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    KEY user_id (user_id),
    KEY expires_at (expires_at)
)
```

#### `wp_ai_credit_actions`
Registered AI actions:
```sql
CREATE TABLE wp_ai_credit_actions (
    id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    action_id varchar(100) NOT NULL,
    label varchar(255) NOT NULL,
    cost int(11) NOT NULL DEFAULT 1,
    description text,
    plugin varchar(100),
    category varchar(50),
    usage_count int(11) DEFAULT 0,
    created_at datetime DEFAULT CURRENT_TIMESTAMP,
    updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    UNIQUE KEY action_id (action_id)
)
```

#### `wp_ai_credit_reservations`
Active credit reservations:
```sql
CREATE TABLE wp_ai_credit_reservations (
    id varchar(32) NOT NULL,
    user_id bigint(20) UNSIGNED NOT NULL,
    amount int(11) NOT NULL,
    action_id varchar(100),
    status enum('pending','confirmed','cancelled') DEFAULT 'pending',
    expires_at datetime NOT NULL,
    created_at datetime DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    KEY user_id_status (user_id, status),
    KEY expires_at (expires_at)
)
```

## Email System

### Email Manager

The email system uses a singleton pattern with modern responsive templates.

#### Get Email Manager Instance
```php
$email_manager = RCP_AI_Credits_Email_Manager::get_instance();
```

#### Send Email Notification
```php
$email_manager->send('purchase_confirmation', $user_id, array(
    'credits' => 100,
    'amount' => 10.00,
    'invoice_id' => 'INV-2024-001',
    'receipt_url' => 'https://pay.stripe.com/receipts/...'
));
```

### Available Email Types
- `purchase_confirmation` - Sent after credit purchase
- `auto_refill_success` - Sent after successful auto-refill
- `auto_refill_failed` - Sent when auto-refill fails
- `low_balance` - Warning when balance is low
- `monthly_credits` - Monthly credit allocation notification
- `admin_gave_credits` - When admin manually adds credits
- `welcome_credits` - Welcome email with initial credits
- `admin_payment_failure` - Admin alert for payment failures
- `admin_daily_summary` - Daily usage summary for admins

### Email Template Variables

Built-in variables available in all templates:
- `{{site_name}}` - Site title
- `{{site_url}}` - Site URL
- `{{dashboard_url}}` - Credits dashboard URL
- `{{purchase_url}}` - Credit purchase URL
- `{{user_name}}` - Recipient's display name
- `{{user_email}}` - Recipient's email
- `{{balance}}` - Current credit balance

### Email Filters

#### `wp_mail_from_name`
The plugin sets this to use the site title (priority 999).

#### `wp_mail_from`
The plugin sets this to use the admin email (priority 999).

### Custom Email Template

To customize email appearance:

```php
// Filter email content before sending
add_filter('rcp_ai_credits_email_content', function($content, $type, $vars) {
    if ($type === 'purchase_confirmation') {
        // Modify content
    }
    return $content;
}, 10, 3);

// Filter email subject
add_filter('rcp_ai_credits_email_subject', function($subject, $type, $vars) {
    if ($type === 'low_balance') {
        return 'Warning: Your AI credits are running low!';
    }
    return $subject;
}, 10, 3);
```

## Error Handling

All API functions return `WP_Error` objects on failure. Always check for errors:

```php
$result = rcp_ai_credits_deduct($user_id, 10, 'test_action');

if (is_wp_error($result)) {
    $error_code = $result->get_error_code();
    $error_message = $result->get_error_message();
    
    switch($error_code) {
        case 'insufficient_credits':
            // Handle insufficient balance
            break;
        case 'invalid_user':
            // Handle invalid user
            break;
        case 'invalid_amount':
            // Handle invalid amount
            break;
        default:
            // Handle other errors
    }
}
```

## Best Practices

1. **Always use the reservation system** for operations that might fail after starting
2. **Register your actions on init** with priority 20 or later
3. **Check credit availability** before starting expensive operations
4. **Log important operations** using the audit trail
5. **Handle errors gracefully** and provide user feedback
6. **Use appropriate sources** when adding credits for tracking
7. **Include context data** for better analytics and debugging

## Integration Example

Complete example of integrating with the credit system:

```php
class My_AI_Plugin {
    
    public function __construct() {
        add_action('init', array($this, 'register_actions'), 20);
    }
    
    public function register_actions() {
        rcp_ai_credits_register_action(array(
            'action_id' => 'my_ai_generation',
            'label' => 'AI Content Generation',
            'cost' => 10,
            'plugin' => 'my-ai-plugin'
        ));
    }
    
    public function generate_content($prompt, $user_id = null) {
        $user_id = $user_id ?: get_current_user_id();
        
        // Check if user can perform action
        if (!rcp_ai_credits_can_perform_action($user_id, 'my_ai_generation')) {
            return new WP_Error('insufficient_credits', 'Not enough credits');
        }
        
        // Reserve credits
        $reservation_id = rcp_ai_credits_reserve($user_id, 10, 'my_ai_generation');
        
        if (!$reservation_id) {
            return new WP_Error('reservation_failed', 'Could not reserve credits');
        }
        
        try {
            // Perform AI operation
            $result = $this->call_ai_api($prompt);
            
            // Calculate actual usage (if variable)
            $tokens_used = $result['usage']['total_tokens'];
            $actual_credits = ceil($tokens_used / 100); // 1 credit per 100 tokens
            
            // Confirm reservation with actual usage
            rcp_ai_credits_confirm_reservation($reservation_id, $actual_credits);
            
            return $result['content'];
            
        } catch (Exception $e) {
            // Cancel reservation on failure
            rcp_ai_credits_cancel_reservation($reservation_id);
            return new WP_Error('api_error', $e->getMessage());
        }
    }
}
```

## Support

For issues or questions about the API:
1. Check the error logs at `wp-content/debug.log`
2. Review the audit trail in the admin dashboard
3. Contact support with transaction IDs and error messages

## REST API Endpoints (Remote Integrations)

When the “Enable REST API” setting is on (Settings → AI Credits → API), this plugin exposes versioned endpoints for external plugins and services.

### Authentication
- Use WordPress Application Passwords over HTTPS (Basic Auth). The authenticated user acts as the current user.

### Endpoints (namespace: `rcp-ai-credits/v1`)
- `GET /balance?user_id=`: Returns `{ user_id, balance, available_balance }`.
- `GET /action-cost?action_id=`: Returns `{ action_id, cost }`.
- `POST /reserve`: Body `{ user_id?, amount, action_id }` → `{ reservation_id }`.
- Note: `amount` is optional. If omitted, the server infers the current cost for the user from `action_id`.
- `POST /confirm`: Body `{ reservation_id, context? }` → `{ success: true }`.
- `POST /cancel`: Body `{ reservation_id }` → `{ success: true }`.
- `POST /generation-log`: Body `{ user_id?, post_id?, action_type?, action_subtype?, prompt?, model?, settings?, result_type?, result_url?, result_text?, credits_used?, processing_time?, status?, error_message? }` → `{ generation_log_id }`.
- `POST /generate-image`: Body `{ prompt, size?, quality?, model?, user_id?, post_id?, reservation_id?, context? }` → `{ generation_log_id, result_type, result_url, credits_used, credits_confirmed }`.
- `GET /job-status?generation_log_id=`: Returns `{ status, generation_log_id, result_type, result_url, credits_used, error_message, created_at }`.

### Permissions
- Self operations allowed for authenticated user.
- Operating on other users requires `manage_ai_credits` (mutations) or `view_ai_credits_analytics` (balance view).

### Examples
```bash
# Get balance (primary site)
curl -u 'user@example.com:APP_PASSWORD' \
  'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1/balance'

# Reserve 10 credits for image generation
curl -u 'user@example.com:APP_PASSWORD' \
  -H 'Content-Type: application/json' \
  -d '{"amount":10, "action_id":"generate_image"}' \
  'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1/reserve'

# Confirm reservation
curl -u 'user@example.com:APP_PASSWORD' \
  -H 'Content-Type: application/json' \
  -d '{"reservation_id":"RES_ID"}' \
  'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1/confirm'

# Log a generation (optional but recommended for UI linking)
curl -u 'user@example.com:APP_PASSWORD' \
  -H 'Content-Type: application/json' \
  -d '{"action_type":"image_generation","action_subtype":"generate_image","prompt":"A red vintage radio", "model":"gpt-image-1", "result_type":"image", "result_url":"https://.../ai-generated/2025/09/img.png", "credits_used":10 }' \
  'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1/generation-log'

# One-call image generation (server reserves/confirm internally if no reservation_id provided)
curl -u 'user@example.com:APP_PASSWORD' \
  -H 'Content-Type: application/json' \
  -d '{"prompt":"A red vintage radio on a wood table"}' \
  'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1/generate-image'

# Check job status (v1 returns immediate success, but this is forward-compatible)
curl -u 'user@example.com:APP_PASSWORD' \
  'https://app.radiocontentpro.com/wp-json/rcp-ai-credits/v1/job-status?generation_log_id=12345'
```

### Recommended Flow
- Balance → Reserve → Perform AI work → Create `generation-log` (captures model/prompt/result) → Confirm with `context` that includes `generation_log_id` (or Cancel on failure).
- Use canonical `action_id` values (e.g., `image_generation`) so analytics and labels align. The plugin ensures defaults for known actions.

Example confirm body with context linkage:
```json
{
  "reservation_id": "RES_ID",
  "context": {
    "generation_log_id": 12345,
    "post_id": 6789
  }
}
```

## Client Examples

### PHP (Secondary Plugin) using Application Passwords
```php
class RCP_AIC_Remote_Client {
  private $base; // e.g. https://primary-site.com
  private $email; // WP user email on primary site
  private $app_password; // Application Password

  public function __construct($base, $email, $app_password) {
    $this->base = rtrim($base, '/');
    $this->email = $email;
    $this->app_password = $app_password;
  }

  private function auth_header() {
    return 'Basic ' . base64_encode($this->email . ':' . $this->app_password);
  }

  private function request($method, $path, $body = null) {
    $args = array(
      'method' => $method,
      'headers' => array(
        'Authorization' => $this->auth_header(),
        'Content-Type' => 'application/json',
      ),
      'timeout' => 20,
    );
    if ($body !== null) {
      $args['body'] = wp_json_encode($body);
    }
    $res = wp_remote_request($this->base . $path, $args);
    if (is_wp_error($res)) return $res;
    $code = wp_remote_retrieve_response_code($res);
    $json = json_decode(wp_remote_retrieve_body($res), true);
    if ($code >= 200 && $code < 300) return $json;
    return new WP_Error('remote_error', $json['message'] ?? 'Remote error', array('status' => $code, 'data' => $json));
  }

  public function reserve($amount, $action_id = 'image_generation', $user_id = null) {
    $payload = array('amount' => (int)$amount, 'action_id' => $action_id);
    if ($user_id) $payload['user_id'] = (int)$user_id;
    return $this->request('POST', '/wp-json/rcp-ai-credits/v1/reserve', $payload);
  }

  public function confirm($reservation_id, $context = array()) {
    return $this->request('POST', '/wp-json/rcp-ai-credits/v1/confirm', array(
      'reservation_id' => $reservation_id,
      'context' => $context,
    ));
  }

  public function log_generation($data) {
    return $this->request('POST', '/wp-json/rcp-ai-credits/v1/generation-log', $data);
  }
}

// Usage
$client = new RCP_AIC_Remote_Client('https://app.radiocontentpro.com', 'user@example.com', 'APP_PASSWORD');
$reservation = $client->reserve(10, 'image_generation');
if (!is_wp_error($reservation)) {
  $gen = $client->log_generation(array(
    'action_type' => 'image_generation',
    'action_subtype' => 'generate_image',
    'prompt' => 'A red vintage radio',
    'model' => 'gpt-image-1',
    'result_type' => 'image',
    'result_url' => 'https://example.com/path/to/image.png',
    'credits_used' => 10,
  ));
  $gen_id = is_wp_error($gen) ? null : ($gen['generation_log_id'] ?? null);
  $client->confirm($reservation['reservation_id'], array('generation_log_id' => $gen_id));
}
```

### JavaScript (Node/fetch) Example
```js
const btoa = (str) => Buffer.from(str, 'utf8').toString('base64');
const auth = 'Basic ' + btoa(`${EMAIL}:${APP_PASSWORD}`);

async function reserve(amount) {
  const res = await fetch(`${BASE}/wp-json/rcp-ai-credits/v1/reserve`, {
    method: 'POST',
    headers: { 'Authorization': auth, 'Content-Type': 'application/json' },
    body: JSON.stringify({ amount, action_id: 'image_generation' })
  });
  if (!res.ok) throw new Error(await res.text());
  return res.json();
}
```

## Error Codes & Responses
- 400 `reservation_failed` / `confirm_failed` / `cancel_failed`: Validation or state error.
- 401 `rest_forbidden`: Not authenticated or invalid Application Password.
- 403 `rest_forbidden`: Authenticated but lacks required capability for target user.
- 404 `not_found`: Action ID unknown.
- 500 `db_error`: Database insert/update failed.

Confirm is single-use per reservation; a second confirm will return a 400 (treat as already consumed if your logs show success).

## Action IDs & Naming
- Use canonical `action_id` values for built-in analytics:
  - `image_generation`: AI image generation (default cost maintained by the plugin).
- Custom actions should be registered on the primary site (PHP) if you plan to use them widely.

## User Mapping Strategies
- Simple: Authenticate each user with their own Application Password on the primary site.
- Service account: Use one Application Password with `manage_ai_credits` and pass `user_id` in requests (store the primary-site user ID securely in your plugin settings).

## Security Notes
- Always use HTTPS; Application Passwords are Basic Auth credentials.
- Do not include PII in prompts or `context` unless necessary. Generation logs are stored for history and analytics.
- Limit the privileges of the Application Password user; prefer per-user credentials where possible.

## Operational Tips
- Timezone: Charts and date filters use the site’s WordPress timezone.
- Reservations: Default expiry is ~5 minutes; confirm or cancel promptly.
- Retries: If a network error occurs after confirm, check balance before retrying to avoid double-charging.
- Rate limiting: The settings screen includes an API rate limit value for future use; implement client-side backoff as good practice.

### Notes on AI Generation
- Built-in AJAX endpoints (nonce-protected) are for on-site UI; remote integrations should perform AI calls directly and use the REST API only for credit accounting.
## Additional Public Helpers

#### `rcp_ai_credits_get_balance_breakdown($user_id)`
Return an array with keys: `total`, `expiring_30d`, `permanent`, `monthly`, `next_expiration`.

#### `rcp_ai_credits_get_expiring_credits($user_id, $days = 30)`
Return an array of expiring credits within the window.

#### `rcp_ai_credits_get_dashboard_url()`
Return the front-end dashboard URL used by the plugin.

#### Payment Method Helpers
- `create_payment_method_setup($user_id = null)` → Returns Stripe Setup Intent client secret or false
- `can_purchase_credits($user_id)` → True if user has an attached payment method
- `rcp_ai_credits_is_free_member_needing_setup($user_id)` → Free member without payment method
- `rcp_ai_credits_get_user_status($user_id)` → Membership + payment summary array
