# RCP API WP Integration: Functional Reference

## 1. End-to-End Workflow
1. **Credential setup** – Admins store the Radio Content Pro API email and application password under options `rcp_feed_username` and `rcp_feed_password`. These credentials are injected into every outbound request via HTTP Basic auth.
2. **Feed discovery** – The Feed page (`feed_page()` + `script.js`) loads recent API posts by calling the AJAX action `rcp_fetch_feed`, which proxies to `get_feed()` and caches results in both object cache and a transient (`rcp_feed_cache<hash>`).
3. **Manual handling** – Editors filter posts, trigger `rcp_import_post`/`rcp_generate_ai_image`, or run bulk operations. Ajax handlers validate with `RCP_API_Validator`, enforce `manage_rcp_api`, and push long-running work to `RCP_API_Job_Queue` when needed.
4. **Automation loop** – Cron (`rcp_auto_import_hook`) or missed-run recovery (`maybe_run_missed_import()`) call `handle_auto_import()`. The importer enforces per-category “automation” rules, cutoffs, rate limits, and author overrides before forwarding each post to `import_post()`.
5. **Post assembly** – `import_post()` fetches enriched post data, sanitizes it, inserts a `post`, maps categories, sets meta, and resolves featured-image mode. Depending on the per-category mode, the plugin either imports the RCP image, generates an AI image immediately, or queues background jobs.
6. **Observability & cleanup** – Every run records structured summaries (`rcp_auto_import_log` + `rcp_import_history`), updates category cutoffs, surfaces diagnostics in the admin UI, and prunes caches, rate-limit buckets, and AI job logs via scheduled cleanup.

## 2. Configuration Surface
- **Core constants** – `POSTS_PER_PAGE` (30), `MAX_INITIAL_POSTS` (15), request timeout (`RCP_API_TIMEOUT`, default 30 s), feed cache TTL (`RCP_CACHE_TTL`, default 300 s), retry ceiling (`RCP_MAX_RETRIES`, 3), performance debugging toggle (`RCP_DEBUG_PERFORMANCE`).
- **Capability** – Every privileged operation checks `manage_rcp_api`; `maybe_upgrade()` ensures administrators receive the capability during upgrades.
- **Custom schedules** – `add_cron_schedules()` registers `rcp_15min` and `rcp_30min`. `maybe_schedule_auto_import()` keeps `rcp_auto_import_hook` aligned with the chosen frequency (`rcp_auto_import_frequency`).
- **Category persistence** – All category-related options store **normalized** keys (see `normalize_category_name()`): lowercase, trimmed, ASCII-only. This prevents duplicates caused by casing or stray whitespace and keeps automation settings stable even if API labels fluctuate.

### 2.1 Persistent Data Stores
| Store | Key | Purpose |
| --- | --- | --- |
| Options | `rcp_feed_username`, `rcp_feed_password` | API credentials retained even if the settings form submits empty fields (`sanitize_persistent_option()`). |
|  | `rcp_category_mapping` | Array of `{ api, wp }` rows mapping API category names to WordPress term IDs. |
|  | `rcp_category_automation` | Normalized map `{ slug => mode }` where mode ∈ {`publish`,`draft`,`none`}. Stored alongside `rcp_category_authors` for per-category author overrides. |
|  | `rcp_category_image_modes` | Normalized map controlling featured-image behaviour (`none`,`import`,`ai`,`fallback`). Legacy checkbox data (`rcp_featured_image_cats`) is folded in for backward compatibility. |
|  | `rcp_ai_autogen_on_import`, `rcp_import_featured_image` | Global fallbacks when no per-category mode is defined. |
|  | `rcp_ai_image_log` | Circular buffer (≤200 entries) of AI image attempts. |
|  | `rcp_auto_import_log` | 48-hour rolling automation log shown on the Automation Log page. |
|  | `rcp_deleted_posts` | Map of previously imported post IDs → deletion timestamps (90-day retention) to prevent re-importing removed content. |
|  | `rcp_category_cutoffs` | Per-category UNIX timestamps indicating the freshest imported post. Used by `handle_auto_import()` to avoid duplicates. |
|  | `rcp_cron_last_heartbeat`, `rcp_last_auto_import`, `rcp_last_successful_import` | Automation health markers surfaced in diagnostics. |
| Custom tables | `wp_rcp_import_queue`, `wp_rcp_import_history`, `wp_rcp_error_log` | Created by `RCP_API_Database_Manager::migrate_1_0_0()`. The importer primarily uses `import_history` for durable summaries; queue/error tables are available for future expansion. |
| Post meta | `_rcp_original_post_id`, `_rcp_featured_image_mode`, `_rcp_ai_prompt`, `_rcp_ai_generation_log_id`, `_rcp_ai_reservation_id` | Track provenance, image rules, prompts, and pending AI reservations per imported post. |
| Transients | `rcp_feed_cache*`, `rcp_api_categories`, `rcp_import_running`, `rcp_ai_gen_pending_<post>` | Feed/category caches, concurrency lock, and AI pending markers (10 minute TTL). |

## 3. Admin Interfaces
### 3.1 Feed Browser (`feed_page()` + `assets/script.js`)
- Renders an empty table that `script.js` populates via AJAX. The script keeps a sessionStorage-backed category filter and draws pagination, bulk action controls, and AI buttons.
- Buttons trigger AJAX endpoints:
  - `rcp_fetch_feed` (GET) – returns `{ posts, total, total_pages, categories }` filtered to the last 36 hours. Results mark already-imported posts by querying `_rcp_original_post_id` in a single SQL call.
  - `rcp_import_post` (POST) – imports one post immediately or queues it when invoked in bulk mode.
  - `rcp_generate_ai_image` (POST) – generates/attaches an AI image for an imported post.
  - `rcp_fetch_categories` (GET) – lazily retrieve the complete set of categories (cacheable).
- Bulk publish/draft iterates sequentially with a 200 ms delay between requests to stay within rate limits.

### 3.2 Settings (`settings_page()`)
- Displays a per-category grid populated by `get_api_categories()`. Each normalized category row allows:
  - Automation mode (`None`, `Auto Publish`, `Auto Draft`).
  - Target WordPress category select (populated from `get_categories()`).
  - Author override (`rcp_category_authors`).
  - Featured-image strategy (`None`, `Import`, `AI`, `Import else AI`).
- Enqueues a “Test Connection” button hitting `rcp_get_health_check` for live API feedback.
- Provides AI credit balance by calling `credits_request( 'GET', 'balance' )` and linking to the credit dashboard.
- Force import form (`admin_post_rcp_force_auto_import`) resets category cutoffs to 36 hours ago and executes `handle_auto_import()` in forced mode.
- “Next Import Preview” runs `run_test_import()` to list would-be imports, giving quick feedback when automation rules produce no work.
- Cron frequency dropdown persists to `rcp_auto_import_frequency`; the importer re-schedules itself whenever the option changes (`maybe_clear_cache()`).

### 3.3 API Connection (`info_page()`)
- Presents credential fields and connection status (`check_connection()` makes a lightweight feed request). Disconnect wipes credentials and clears caches.

### 3.4 Automation Log (`log_page()`)
- Pulls option-backed log entries plus the latest DB-backed history (`RCP_API_Database_Manager::get_import_history( [ 'limit' => 20 ] )`). Entries include debug details, success flag, counts of published/drafted posts, and error messages.

### 3.5 Diagnostics (`diagnostics_page()`)
- Aggregates cron timings, automation maps, cutoffs, authors, recent logs, rate-limit status (`RCP_API_Rate_Limiter::get_status()`), cache version data, and environment metadata (plugin/WP/PHP versions, timezone, Action Scheduler availability).
- Provides buttons to auto-refresh, download the debug log, clear import locks (`admin_post_rcp_clear_import_lock`), and inspect AI job activity.

### 3.6 AI Images Manager (`ai_images_page()`)
- Shows paginated AI image runs with optional status filtering and retry controls (`admin_post_rcp_ai_images_retry`). Includes buttons to trim the log (`admin_post_rcp_ai_images_clear_log`) and quick links to source posts/attachments.

## 4. AJAX Contracts & Rate Limiting
Every AJAX action calls `RCP_API_Validator::validate_ajax_request()` (capability + nonce) and is subject to `RCP_API_Rate_Limiter` buckets:
- `rcp_fetch_feed` & `rcp_fetch_categories` → `type = 'ajax'`, default 30 requests / 60 s.
- `rcp_import_post` & AI actions → `type = 'import'`, default 10 requests / 5 minutes.
- JSON errors yield HTTP 429 with `Retry-After` headers when the bucket is exhausted. Response headers also expose `X-RateLimit-*` values via `RCP_API_Rate_Limiter::add_headers()`.

### 4.1 `rcp_fetch_feed`
- Query params: `page` (validated integer), optional `categories[]` (normalized names).
- Applies a 36 hour “freshness window” (`rcp_api_max_import_age` filter). If category filters are active, it walks subsequent API pages until enough matching posts are buffered.
- Filters out posts lacking `blog_post_title` or `blog_post_html` before responding.

### 4.2 `rcp_import_post`
- Payload: `post_id`, `status` (`publish|draft`), optional `post_date`, `bulk_import` flag.
- Bulk mode schedules `RCP_API_Job_Queue::process_import_single_post`; manual mode calls `import_post()` synchronously and records a manual log entry via `record_manual_import()`.

### 4.3 `rcp_generate_ai_image`
- Payload: optional `wp_post_id`, `rcp_post_id`. Resolves to the WordPress post via `_rcp_original_post_id` if needed.
- Applies a transient lock (`rcp_ai_gen_pending_<post>`) to avoid duplicate jobs, triggers `generate_ai_image_for_post()`, and logs outcomes (`record_ai_image_log()`).

### 4.4 `rcp_get_health_check`
- Returns a composite object comprising API connectivity, database statistics, job queue availability, cron schedule, and performance monitor status (`RCP_API_Performance_Monitor::get_system_health()`).

## 5. Automated Import Engine (`handle_auto_import()`)
1. **Locking** – Uses transient `rcp_import_running` with 5 minute TTL. Stale locks are cleared after the TTL to prevent deadlocks. The function also updates `rcp_cron_last_heartbeat` on entry.
2. **Safety valve** – If imports have been failing for over two hours, the importer backs cutoffs up by one hour (`SAFETY` block) before proceeding.
3. **Automation validation** – Builds a normalized automation map (`get_normalized_automation_map()`) merging modes and author overrides. If every category is set to `none`, the run records an informational log and exits early.
4. **Cutoff resolution** – Pulls normalized cutoffs, determines `after` timestamp as the latest cutoff (bounded by 36 hour window). Forced imports ignore cutoffs.
5. **Feed traversal** – Fetches API pages greedily (cache bypass = `true`). Each post goes through:
   - Duplicate check via `_rcp_original_post_id` lookup.
   - Deletion check (`rcp_deleted_posts`) unless forced.
   - Freshness check (36 hour limit & per-category cutoff with 60 s grace).
   - Automation match (`evaluate_automation_match()`) returning status and author assignment.
6. **Import** – Eligible posts call `import_post()`. Success counts update `log_published`/`log_drafts`; failures append context-rich messages to `log_errors`.
7. **Cutoff maintenance** – For categories that imported posts, the cutoff is set to `latest_timestamp - grace` (default 2 h, filter `rcp_api_cutoff_grace`). The minimum is clamped to the freshness window and never allowed in the future.
8. **Logging** – Aggregates metrics (execution time, memory, posts processed/skipped/imported, debug trace) and persists via `record_import_log()`. Successful runs also set `rcp_last_successful_import`.
9. **Unlocking** – Deletes the transient lock unless the run was explicitly forced.

`maybe_run_missed_import()` invokes the same pipeline for overdue cron events while respecting locks and intervals.

## 6. Import Pipeline (`import_post()`)
1. **Fetch post** – `fetch_rcp_post()` retrieves the full payload (`blog_post_title`, `blog_post_html`, media info) with caching. Missing blog content raises `WP_Error( 'missing_blog_data' )`.
2. **Author assignment** – Unless provided, categories are normalized and resolved through `rcp_category_authors`.
3. **Duplicate guard** – Any existing WordPress post with `_rcp_original_post_id = post_id` returns immediately with edit/view links and `already_imported = true`.
4. **Content prep** – Title and content fall back to REST fields when the blog variant is unavailable. Excerpts are trimmed with `wp_trim_words()`.
5. **Date handling** – Optional `post_date` is normalized to both local and GMT via `normalize_post_date_for_insert()`; otherwise `current_time()` is used.
6. **Insertion** – `wp_insert_post()` persists the post with sanitized title/content/excerpt and assigned `post_author` (if any).
7. **Featured image mode** – `resolve_category_image_mode()` combines per-category, legacy, and global defaults to choose between `none`, `import`, `ai`, or `fallback`.
   - `import` / `fallback` immediately call `import_featured_image_for_rcp_post()`; errors schedule retries through `RCP_API_Job_Queue::process_images` (unless the error was `no_image`, in which case fallback defers to AI).
   - `ai` or `fallback` without an imported image queue AI generation: synchronous attempt (`generate_ai_image_for_post()`) then asynchronous retry (`RCP_API_Job_Queue::process_ai_generate`) using `rcp_ai_gen_pending_<post>` to avoid duplicate work.
8. **Category taxonomy** – Maps normalized API category names through `rcp_category_mapping`. Unmapped categories are created on the fly (via `term_exists()` / `wp_insert_term()`) before `wp_set_post_categories()` assigns them.
9. **Metadata** – Stores `_rcp_original_post_id`, `_rcp_featured_image_mode`, and optionally `_rcp_ai_prompt` (from API-provided prompt). Returns edit and view URLs tailored to the post status.

## 7. Image & AI Handling
- **Protected media download** – `import_featured_image_for_rcp_post()` consolidates image URLs from multiple API fields (`featured_image`, `featured_image_url`, `featured_image_html`, `featured_media`). `sideload_protected_image()` performs a HEAD check (content-type + size), enforces HTTPS, injects Basic auth (via `add_auth_header()` filter), and uses `media_handle_sideload()` to create the attachment.
- **AI pipeline** – `generate_ai_image_for_post()`:
  1. Ensures no thumbnail exists, resolves the RCP source ID, and determines/records the prompt (`get_ai_prompt()` falls back to API prompt or a generated string).
  2. Reserves credits (`reserve_ai_image_credits()`), storing the reservation ID on the post.
  3. Calls `request_ai_image_generate()` (credits API `POST /generate-image`).
  4. If the response contains a `result_url`, the image is attached immediately; otherwise the generation log ID is stored and `RCP_API_Job_Queue::process_ai_poll` polls `GET /job-status` with exponential backoff until success, failure, or timeout.
  5. On success, reservations are confirmed (`confirm_ai_image_reservation()`); failures cancel reservations and log errors.
- **AI logging** – Every attempt writes to `rcp_ai_image_log` with status (`queued_manual`, `reservation_failed`, `timeout`, etc.) for display and troubleshooting.

## 8. Background Jobs (`includes/class-rcp-api-job-queue.php`)
- Uses Action Scheduler when available (`as_enqueue_async_action`, `as_schedule_single_action`); otherwise falls back to `wp_schedule_event` / `wp_schedule_single_event` with prefixed hooks (`rcp_api_job_*`).
- Registered processors:
  - `import_posts` – Consumes batches by calling `process_import_batch()` and re-queues while more pages remain.
  - `import_single_post` – Imports one post and optionally queues `process_images` for the associated featured image.
  - `process_images` – Retries protected image downloads or escalates to AI fallback.
  - `ai_generate` – Kicks off AI generation for queued posts.
  - `ai_poll` – Polls job status, attaches completed images, manages reservations, and handles rate-limit responses gracefully.
  - `cleanup_transients` – Hourly housekeeping that trims rate-limit buckets, import logs, deletion markers, performance metrics, and AI logs.
- `get_status()` reports scheduler availability, pending counts, and next run times for diagnostics and CLI reporting.

## 9. Logging, Diagnostics, and Error Handling
- **Import log** – `record_import_log()` persists structured entries (published/drafted counts, errors, debug payload including automation maps and timing). Logs are both option-backed and mirrored into `wp_rcp_import_history` for longer retention.
- **Manual imports** – `record_manual_import()` records one-off actions and mirrors them to the history table.
- **Deletion tracking** – `track_deleted_post()` stores RCP IDs when imported posts are deleted or trashed, allowing `handle_auto_import()` to skip them on future runs unless forced.
- **Error handler** – When `WP_DEBUG` is true, `RCP_API_Error_Handler::init()` wraps PHP errors, exceptions, and shutdown fatals to log into option `rcp_api_error_log`. Severe failures can queue email notifications (via `rcp_api_error_email_queue`).
- **Performance monitor** – `RCP_API_Performance_Monitor` offers lightweight timers around AJAX actions and jobs. Metrics can be persisted (opt-in filter `rcp_api_store_performance_metrics`) and appear in health checks.
- **Debug logging** – `debug_log()` respects `rcp_debug_mode` and `WP_DEBUG`, writing verbose traces to `debug.log`—useful for replicating edge conditions.

## 10. CLI Commands (`includes/class-rcp-api-cli.php`)
- `wp rcp import run [--force]` – Triggers `handle_auto_import()` immediately.
- `wp rcp import diagnose` – Prints next cron run, last run timestamps, and job queue status.
- `wp rcp log tail [--count=<n>]` – Streams recent option + DB log entries.
- `wp rcp cache flush` – Calls `RCP_API_Cache_Manager::flush_all()` when available.
- `wp rcp ai balance` – Fetches credit balance via the credits API.
- `wp rcp ai generate --post=<id>` – Queues AI generation for a specific WP post.
- `wp rcp ai generate --missing-thumbnails [--since=<YYYY-mm-dd>] [--dry-run]` – Finds imported posts lacking thumbnails and queues generation.
- `wp rcp ai jobs [--status=<queued|success|failed|timeout>] [--limit=<n>]` – Displays recent entries from `rcp_ai_image_log`.

## 11. Security, Rate Limiting, and Concurrency Controls
- **Capability** – Every menu page and AJAX handler checks `manage_rcp_api`; settings forms use nonces (`settings_fields()` and custom `wp_nonce_field()` calls).
- **Rate limiting** – `RCP_API_Rate_Limiter` stores request timestamps in transients keyed by user/IP and exposes helper methods for headers, resets, and cleanup. CIDR-based whitelisting is supported via `rcp_api_rate_limit_whitelist`.
- **Locks** – Import concurrency is governed by `rcp_import_running`; AI jobs use `rcp_ai_gen_pending_<post>`; transients expire automatically to unblock stuck states.
- **Sanitization** – Every option and AJAX input is sanitized (`sanitize_text_field`, `absint`, etc.). Category data is normalized consistently, ensuring persistence across sessions and API responses.
- **Media safety** – Featured-image downloads are size-checked, MIME-validated, forced to HTTPS, and cleaned up if they exceed configured limits.

## 12. External API Contract
- **Content feed** – `GET https://radiocontentpro.com/wp-json/rcp/v1/feed?per_page=30&page=n&after=YYYY-mm-dd HH:MM:SS`.
- **Single post** – `GET /wp-json/rcp/v1/posts/<id>?fields=...` (fields assembled by `get_post_import_fields()` to include blog HTML, categories, media).
- **AI prompt** – `GET /wp-json/rcp/v1/ai_image_generation_prompt?post_id=<id>`.
- **AI credits** – Base computed by `get_credits_api_base()` (defaults to `https://radiocontentpro.com/wp-json/rcp-ai-credits/v1`). Endpoints:
  - `GET /balance`
  - `GET /action-cost?action_id=image_generation`
  - `POST /reserve` `{ action_id, amount }`
  - `POST /generate-image` `{ prompt, context, reservation_id? }`
  - `GET /job-status?generation_log_id=...`
  - `POST /confirm` / `POST /cancel` `{ reservation_id }`

All requests rely on Basic auth using the stored credentials; timeouts inherit `RCP_API_TIMEOUT` unless filters extend them (`rcp_ai_generate_timeout`).

## 13. Guidance for Re-implementation (Next.js / Vercel / Neon)
- **Persisted state** – Mirror the option/meta layout: credentials, normalized category maps, automation rules, cutoffs, deletion ledger, and AI logs. Normalization ensures UI selectors remain stable between sessions and across API responses; reproducing `normalize_category_name()` semantics is key to “persistent categories.”
- **Import pipeline** – Recreate the `handle_auto_import()` loop: enforce freshness windows, per-category automation, author overrides, deletion skip list, and grace-adjusted cutoffs. Maintain an equivalent lock (e.g., Neon row/flag or Redis key) to prevent concurrent runs.
- **Manual tooling** – Provide an authenticated feed browser that:
  - Caches feed pages per-credential pair with a short TTL.
  - Filters on normalized category keys and flags already-imported posts by looking up original IDs.
  - Streams bulk actions sequentially with progress feedback and surfaces AI-generation shortcuts.
- **AI orchestration** – Implement the two-path featured-image strategy: attempt source import first (with retries), then optionally fall back to AI. Reserve/confirm/cancel credits and store reservation + job identifiers to resume in background workers.
- **Observability** – Sustain structured logs, AI job history, and diagnostics (cron status, health checks, cache insight). Offer CLI or admin endpoints mirroring the WP-CLI commands for operational parity.
- **Safety nets** – Preserve rate limiting, lock expirations, and deletion awareness to match WordPress behaviour and avoid content duplication.

Replicating these moving parts with Next.js services, serverless cron (Vercel scheduled functions), and Neon for storage will reproduce the plugin’s behaviour faithfully while keeping category automation durable for each credential set.
