# RCP API WP Integration - Development Context

**Last Updated**: September 17, 2025

## Overview
This is a WordPress plugin that integrates with the Radio Content Pro API to import posts into WordPress. The plugin provides both manual and automated import capabilities with comprehensive category management, image handling, and enterprise-grade security and performance features.

## Key Architecture

### Main Plugin File
- **File**: `rcp-api-wp-integration.php`
- **Main Class**: `RCP_API_WP_Integration`
- **Version**: 0.8.2
- **API Base**: `https://app.radiocontentpro.com/wp-json/rcp/v1`

### Core Features
1. **API Connection Management** - HTTP Basic auth with WordPress application passwords
2. **Manual Import Interface** - Admin pages for browsing and importing posts
3. **Bulk Import Functionality** - Select multiple posts and import as published or drafts
4. **Automated Import System** - Job queue-based automation with category-specific rules
5. **Category Mapping** - Map RCP categories to WordPress categories
6. **Featured Image Import** - Download and attach images from RCP with SSRF protection
7. **Comprehensive Logging** - Track all import activities with structured error handling
8. **Deleted Posts Tracking** - Prevents automatic re-import of deleted posts
9. **Advanced Security** - Input validation, rate limiting, SQL injection prevention
10. **Performance Optimization** - Multi-tier caching, job queues, database indexing
11. **Health Monitoring** - System health checks, performance metrics, diagnostic tools

### New Core Classes (v0.8.0)
- **RCP_API_Validator** - Comprehensive input validation with SSRF protection
- **RCP_API_Rate_Limiter** - Multi-tier rate limiting with IP whitelisting
- **RCP_API_Performance_Monitor** - Performance tracking and metrics collection
- **RCP_API_Job_Queue** - Action Scheduler integration for background processing
- **RCP_API_Error_Handler** - Structured error handling with severity levels
- **RCP_API_Database_Manager** - Migration system and custom table management
- **RCP_API_Cache_Manager** - Advanced caching with object cache support

## Database Schema

### Options Used
- `rcp_feed_username` - API username (email)
- `rcp_feed_password` - WordPress application password
- `rcp_category_mapping` - Array of API→WP category mappings
- `rcp_category_automation` - Per-category automation rules (none/draft/publish)
- `rcp_category_authors` - Default authors per category
- `rcp_featured_image_cats` - Categories that should import featured images
- `rcp_auto_import_frequency` - Cron frequency (15min/30min/hourly)
- `rcp_last_auto_import` - Timestamp of last import
- `rcp_category_cutoffs` - Per-category timestamps to prevent re-imports
- `rcp_auto_import_log` - Import activity log (48hr retention)
- `rcp_deleted_posts` - Tracks deleted RCP post IDs to prevent auto-reimport (90-day retention)
- `rcp_debug_mode` - Enable/disable debug logging
- `rcp_api_db_version` - Database schema version for migrations
- `rcp_api_error_log` - Structured error log storage

### Custom Tables (v0.8.0)
- `{prefix}_rcp_import_queue` - Job queue for import processing
- `{prefix}_rcp_import_history` - Detailed import history with metrics
- `{prefix}_rcp_error_log` - Structured error logging

### Post Meta
- `_rcp_original_post_id` - Links imported posts back to RCP API post ID

## API Endpoints Used

### RCP API Endpoints
- `GET /feed` - Paginated post feed (30 posts/page)
- `GET /posts/{id}` - Single post details
- `GET /media/{id}` - Media/image details

### WordPress AJAX Endpoints
- `rcp_fetch_feed` - Load posts for admin interface (with validation & rate limiting)
- `rcp_import_post` - Import single post (with duplicate prevention)
- `rcp_fetch_categories` - Get available categories
- `rcp_health_check` - System health status (v0.8.0)

## Admin Interface

### Menu Structure
- **RCP API** (main menu)
  - **Feed** - Browse and import posts manually
  - **Settings** - Configure automation and mappings
  - **API Connection** - Manage credentials
  - **Automation Log** - View import history

### Key Admin Features
- Real-time connection status indicator
- Category-based filtering in feed
- Bulk selection and import
- Per-category automation settings
- Featured image import controls with legal disclaimer

## Automation System

### Job Queue Implementation (v0.8.0)
- **Primary**: Action Scheduler for reliable background processing
- **Fallback**: WordPress cron if Action Scheduler unavailable
- **Hook**: `rcp_auto_import_hook`
- **Frequencies**: 15min, 30min, hourly
- **Batch Processing**: Configurable batch sizes (default: 10)
- **Retry Logic**: Exponential backoff for failed jobs

### Import Logic
1. Fetch posts newer than category cutoff times
2. Skip posts older than 36 hours
3. Check for existing imports via `_rcp_original_post_id`
4. Apply category-specific automation rules
5. Import with designated authors
6. Update cutoff times to prevent re-imports
7. Track performance metrics for each import

## Security Features (Enhanced v0.8.0)

### Comprehensive Input Validation
- **RCP_API_Validator class** - Centralized validation for all inputs
- **Type-specific validation** - Post IDs, dates, URLs, arrays, status values
- **AJAX nonce verification** - All endpoints protected
- **Capability checks** - Admin-only operations verified
- **SQL injection prevention** - All queries use prepared statements

### Advanced Rate Limiting
- **Multi-tier limits** - Different rates for different operations
  - Global: 60 requests/minute
  - AJAX: 30 requests/minute
  - Import: 10 requests/5 minutes
  - API: 100 requests/5 minutes
- **IP whitelisting** - Support for trusted IPs and CIDR ranges
- **Automatic cleanup** - Expired rate limit data removed

### Image Security
- **SSRF protection** - DNS resolution validation
- **Domain restriction** - Only download from RCP domain
- **File type validation** - Proper MIME type checking
- **Sanitized filenames** - WordPress media handling
- **Size limits** - Prevent resource exhaustion

## Caching Strategy (Enhanced v0.8.0)

### Advanced Multi-Tier Cache System
- **Primary**: Object cache (Redis/Memcached) with automatic detection
- **Fallback**: Transient cache for environments without object cache
- **Cache Groups**: Separate groups for different data types
  - API responses: 5min TTL (configurable)
  - Categories: 15min TTL
  - Posts: 10min TTL
  - Settings: 1 hour TTL

### Intelligent Cache Management
- **Version-based invalidation** - Easy cache busting
- **Group-based clearing** - Clear specific cache types
- **Automatic invalidation** on:
  - Settings changes
  - Post imports
  - Category updates
- **Cache warming** - Pre-populate cache after imports
- **Performance tracking** - Cache hit/miss ratios

## Performance Optimizations (Enhanced v0.8.0)

### Database Optimizations
- **Custom indexes** on frequently queried columns
  - `_rcp_original_post_id` meta queries
  - Composite indexes on post status/type/date
  - Import queue priority and scheduled time
- **Migration system** - Version-controlled schema updates
- **Query optimization** - Prepared statements and efficient JOINs
- **Automatic maintenance** - Table optimization routines

### API & Processing
- **Job queue system** - Asynchronous batch processing
- **Configurable batch sizes** - Prevent memory exhaustion
- **Performance monitoring** - Track execution time and memory
- **Parallel processing** - Multiple import jobs can run concurrently
- **Timeout handling** - Configurable timeouts (default: 30 seconds)

### Resource Management
- **Memory tracking** - Monitor usage during imports
- **Query counting** - Track database operations
- **Execution time limits** - Prevent long-running processes
- **Automatic cleanup** - Remove old logs and temporary data

## Error Handling (Enhanced v0.8.0)

### Structured Error System
- **RCP_API_Error_Handler class** - Centralized error management
- **Severity levels** - Critical, Error, Warning, Notice, Info, Debug
- **Custom exception handling** - Graceful error recovery
- **Admin notifications** - Email alerts for critical errors
- **Error context tracking** - File, line, stack trace

### Comprehensive Logging
- **Import logs** - 48-hour retention with detailed metrics
- **Error logs** - Structured storage in custom table
- **Debug mode** - Optional verbose logging
- **Performance metrics** - Execution time, memory, queries
- **Export functionality** - Download logs for analysis

### Graceful Degradation
- **Job queue fallback** - WordPress cron if Action Scheduler unavailable
- **Cache fallback** - Transients if object cache unavailable
- **Partial failure handling** - Continue with working operations
- **Retry mechanisms** - Exponential backoff for failed jobs
- **Circuit breakers** - Prevent cascading failures

## Development Notes

### Code Standards
- WordPress coding standards followed
- Proper escaping and sanitization
- Internationalization ready (`rcp-api-wp-integration` text domain)
- Object-oriented design with single responsibility

### Key Dependencies
- WordPress media handling functions
- Plugin Update Checker library (v5)
- Standard WordPress AJAX and cron systems
- Action Scheduler (optional but recommended)
- Object cache support (Redis/Memcached recommended)

### Testing Considerations
- Test with various WordPress configurations
- Verify cron functionality across hosting environments
- Test image import with different API responses
- Validate category mapping edge cases

## Version 0.8.0 Major Improvements (July 18, 2025)

### Security Enhancements
- **Input Validation System**: New RCP_API_Validator class provides comprehensive validation for all user inputs
- **SQL Injection Prevention**: All database queries now use prepared statements
- **Advanced Rate Limiting**: Multi-tier rate limiting with configurable limits per action type
- **SSRF Protection**: Enhanced validation for image downloads with DNS resolution checks

### Performance Improvements
- **Job Queue System**: Action Scheduler integration for reliable background processing
- **Advanced Caching**: Multi-tier caching with object cache support and intelligent invalidation
- **Database Optimization**: Custom indexes and migration system for better query performance
- **Performance Monitoring**: Built-in metrics tracking for execution time and resource usage

### Reliability Features
- **Structured Error Handling**: Comprehensive error management with severity levels
- **Health Monitoring**: System health checks and diagnostic endpoints
- **Database Migrations**: Version-controlled schema management
- **Graceful Degradation**: Fallback mechanisms for all critical systems

### Developer Experience
- **Modular Architecture**: Seven new specialized classes for better code organization
- **Configurable Constants**: Override key settings via wp-config.php
- **Enhanced Debugging**: Detailed logging with export functionality
- **Backward Compatibility**: All changes maintain compatibility with existing installations

## Common Issues & Solutions

### Cron Problems
- Plugin includes fallback execution in `maybe_run_missed_import()`
- Check `wp_next_scheduled()` for scheduling issues
- Verify server supports WordPress cron

### Image Import Issues
- Check file permissions on uploads directory
- Verify API authentication for protected images
- Review featured image URL variations in API responses

### Category Mapping
- Categories are matched case-insensitively
- Auto-creation of missing WordPress categories
- Proper handling of category name changes

### Empty Content Filtering (Version 0.7.2)
- **Problem**: Posts with empty blog_post_html could appear in feed and potentially be imported
- **Solution**: 
  - Added strict filtering to prevent posts with empty blog_post_html from appearing in the feed
  - Enhanced content requirements checking with detailed debug logging
  - Both automation and manual imports now completely reject posts missing title or blog_post_html
- **Implementation**:
  - Feed display filters out posts before showing them to users
  - Debug logs show exactly why posts are filtered (missing title vs missing content)
  - Clear skip reasons in automation logs specify "blog_post_html" when content is empty
- **User Benefit**: Complete assurance that no posts with empty blog_post_html will ever be imported or shown

### Import Type Specificity & Duplicate Prevention (Version 0.7.1)
- **Problem**: Automation log only showed generic "automation" type; posts could be imported twice (once via schedule, once manually)
- **Solution**: 
  - Enhanced automation log to show specific import types: "Scheduled", "Force Auto", "Manual", "Bulk Import"
  - Added duplicate prevention check to manual imports
  - Feed now prevents re-importing posts that were already imported via any method
- **Implementation**:
  - Modified log_page() to display specific import types based on debug data
  - Added duplicate check in import_post() method before creating new posts
  - Returns existing post data if already imported, showing "Already Imported" status
- **User Benefit**: Clear visibility of how posts were imported and complete prevention of duplicate imports

### Content Requirements Update (Version 0.7.0)
- **Removed Content Requirements Setting**: Posts now always require both title AND content
- **Simplified Logic**: No longer configurable - posts missing either field are automatically skipped
- **Clear Error Messages**: Import failures clearly indicate missing title or content

### Enhanced Debug Logging (Added in 0.6.9)
- **Debug Mode Toggle**: Added checkbox in settings to enable/disable debug logging
- **Comprehensive API Logging**: Logs all API requests, responses, and errors
- **Detailed Import Logging**: Shows exactly why posts are skipped or imported
- **Cutoff Calculations**: Clear logging of timestamp comparisons with grace periods
- **Performance Metrics**: Tracks execution time, memory usage, and throughput
- **Debug Log Export**: Download button on diagnostics page exports full debug history
- **Empty Response Detection**: Warns when API returns no posts for the query

### Automation Scheduling Issues Fixed in 0.6.8
- **Wider Cutoff Window**: Changed from 60 seconds to 10 minutes to prevent missing posts
- **Grace Period**: Increased from 5 to 60 seconds for timestamp comparisons
- **Lock Timeout**: Reduced from 15 to 5 minutes to prevent stuck imports
- **Manual Lock Clear**: Added button to clear stuck import locks
- **Safety Mechanisms**: Auto-adjusts cutoffs if imports fail for 2+ hours
- **Import Preview**: Shows what will be imported on next run
- **Enhanced Diagnostics**: Shows import window settings and cutoff details

### Automation Scheduling Issues (Fixed in Earlier Versions)
- **Category Cutoff Logic**: Fixed bug where cutoffs were set to "now" when settings saved, blocking legitimate imports
- **Timestamp Handling**: Standardized all operations to use UTC timestamps consistently
- **Enhanced Logging**: Added detailed debug logging to track import decisions and failures
- **Cache Management**: Improved cache clearing timing to prevent conflicts
- **Validation**: Added robust validation and error handling for malformed data

### Deleted Posts Tracking (Added in 0.6.4)
- **Problem**: Deleted posts were being re-imported on the next automation run
- **Solution**: Track deleted RCP post IDs to prevent automatic re-import
- **Implementation**:
  - Hooks into `before_delete_post` and `wp_trash_post` actions
  - Stores RCP post ID with deletion timestamp
  - Automatic cleanup of entries older than 90 days
  - Only affects automatic imports - manual imports can still re-import
- **User Benefit**: Posts stay deleted when users delete them
- **Diagnostics**: Deleted posts count displayed on diagnostics page

## Future Development Areas

### Potential Enhancements
1. **Content Filtering** - Pre-import content modification
2. **Multi-site Support** - Network-wide configuration
3. **Advanced Scheduling** - Custom cron intervals
4. **Webhook Support** - Real-time import triggers
5. **Export Functionality** - Send WordPress content back to RCP

### Maintenance Tasks
- Monitor API endpoint changes
- Update Plugin Update Checker library
- Review database indexes as content grows
- Optimize cache strategies based on usage patterns

## File Structure
```
/
├── rcp-api-wp-integration.php  # Main plugin file (v0.8.0)
├── script.js                   # Admin interface JavaScript
├── style.css                   # Admin interface styles
├── includes/                   # Core class files (v0.8.0)
│   ├── class-rcp-api-validator.php         # Input validation
│   ├── class-rcp-api-rate-limiter.php      # Rate limiting
│   ├── class-rcp-api-performance-monitor.php # Performance tracking
│   ├── class-rcp-api-job-queue.php          # Job queue handler
│   ├── class-rcp-api-error-handler.php      # Error management
│   ├── class-rcp-api-database-manager.php   # Database operations
│   └── class-rcp-api-cache-manager.php      # Cache management
├── lib/                        # Plugin Update Checker library
├── assets/                     # Plugin assets and documentation
├── CLAUDE.md                   # Development context (this file)
└── plugin-update.json          # Private update configuration
```

## Testing Commands
Since this is a WordPress plugin, testing should be done in a WordPress environment:
- Activate plugin and configure API credentials
- Test manual import via admin interface
- Verify automated import with cron
- Check category mapping and image import
- Review import logs for errors
